/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <iostream>
#include <boost/bind.hpp>
#include <boost/filesystem/convenience.hpp>
#include <boost/filesystem/operations.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/thread/mutex.hpp>

#include <si/interface/Config.h>
#include "DefaultLogger.h"
using namespace si::interface;
using namespace modules::logger;


#define WITH_THREAD

std::string lv2s(int level)
{
    switch (level) {
    case Logger::FATAL_LOG :
        return "fatal";
    case Logger::ERROR_LOG :
        return "error";
    case Logger::WARN_LOG :
        return "warn";
    case Logger::DEBUG_LOG :
        return "debug";
    default :
        return "notice";
    }
}

int s2lv(const std::string &s)
{
    if (s == "FATAL")
        return Logger::FATAL_LOG;
    else if (s == "ERROR")
        return Logger::ERROR_LOG;
    else if (s == "WARN")
        return Logger::WARN_LOG;
    else if (s == "DEBUG")
        return Logger::DEBUG_LOG;
    else
        return Logger::INFO_LOG;
}


//
// Constructor/Destructor
//

DefaultLogger::DefaultLogger()
    : _is_open(false)
{ }


DefaultLogger::~DefaultLogger()
{
    close();
}


//
// Member functions.
//

void DefaultLogger::config(const si::interface::Config& c)
{
    _config = c;
    _home   = c.attr("home");
    _dir    = c.attr("directory");
    _prefix = c.attr("prefix");
    _suffix = c.attr("suffix");

    if (_dir.empty())
        throw std::runtime_error("DefaultLogger::open - "
                                 "Empty in path strings. \"directory\"");
    if (_prefix.empty())
        throw std::runtime_error("DefaultLogger::open - "
                                 "Empty in path strings. \"prefix\"");
    if (_suffix.empty())
        throw std::runtime_error("DefaultLogger::open - "
                                 "Empty in path strings. \"suffix\"");

    _stdout = c.attr("stdout") == "true" ? true : false;

    try {
        _level = boost::lexical_cast<int>(c.attr("verbosityLevel"));
    } catch (boost::bad_lexical_cast& ) {
        _level = s2lv(c.attr("verbosityLevel"));
    }
}

Config DefaultLogger::config()
{
    return _config;
}

void DefaultLogger::open()
{
    boost::mutex::scoped_lock lock(_mutex);

    if (_is_open)
        return;

    std::string dir = _home + "/" + _dir;
    if (!boost::filesystem::exists(dir))
        boost::filesystem::create_directories(dir);

    std::string file = _home + "/" + _dir + "/" + _prefix + "." + _suffix;
    _ofs.open(file.c_str());
    if (!_ofs)
        throw std::runtime_error("std::runtime_error: Can't open file." + file);

    _is_open = true;

#ifdef WITH_THREAD
    _output_thr =
        new boost::thread(boost::bind(&DefaultLogger::output_thr, this));
#endif
}

void DefaultLogger::close()
{ 
    boost::mutex::scoped_lock lock(_mutex);

    if (!_is_open)
        return ;

    _is_open = false;
    lock.unlock();

#ifdef WITH_THREAD
    _output_thr->join();
    delete _output_thr;
#endif
    _ofs.close();
}

void DefaultLogger::log(const std::string &msg)
{
    log(msg, INFO_LOG);
}

void DefaultLogger::log(const std::string &msg, int level)
{
    boost::mutex::scoped_lock lock(_mutex);
    if (!_is_open)
        throw std::runtime_error("std::runtime_error: It has closed yet.");

#ifdef WITH_THREAD
    _queue.push_back(msg_part(time(NULL), level, msg));
    _cond.notify_one();
#else
    write(msg_part(time(NULL), level, msg));
#endif
}

void DefaultLogger::log(const std::string &msg, const std::exception &e)
{
    log(msg + " exception \"" + e.what() + "\"", ERROR_LOG);
}

void DefaultLogger::output_thr()
{
    boost::mutex::scoped_lock lock(_mutex);
    lock.unlock();
    for (;;) {
        lock.lock();
        while (_queue.empty())
            _cond.wait(lock);

        msg_part m = _queue.front();
        _queue.pop_front();
        lock.unlock();

        if (_level >= m._level)
            write(m);

        lock.lock();
        if (!_is_open && _queue.empty())
            break;
        lock.unlock();
    }
}

void DefaultLogger::write(const msg_part& m)
{
    char buffer[32];
    std::string t(::ctime_r(&m._time, buffer));
    t.erase(t.length() - 1);

    std::stringstream ss;
    ss << "[" << t << "] [" << lv2s(m._level) << "] " << m._msg;

    if (_stdout)
        std::cout << ss.str() << std::endl;
    else
        _ofs << ss.str() << std::endl;
}
