/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <boost/algorithm/string/predicate.hpp>
#include <boost/algorithm/string/replace.hpp>
#include <boost/filesystem/convenience.hpp>
#include <boost/filesystem/exception.hpp>
#include <boost/filesystem/operations.hpp>
using namespace std;
using namespace boost;
namespace fs = boost::filesystem;

#include <sl/java/lang/Object.h>
#include <sl/java/lang/String.h>
#include <sl/java/lang/ClassLoader.h>
#include <servlet/http/HttpServlet.h>
#include <servlet/GenericServlet.h>

#include "DefaultWrapper.h"

#include <si/shibainu.h>
#include <si/core/ErrorPage.h>
#include <si/core/HttpServletRequestImpl.h>
#include <si/core/HttpServletResponseImpl.h>
#include <si/core/ServletConfigImpl.h>
#include <si/core/ServletContextImpl.h>
#include <si/core/url_pattern.h>
#include <si/interface/Config.h>
#include <si/interface/Connection.h>
#include <si/interface/Loader.h>
#include <si/interface/Logger.h>
#include <si/interface/Manager.h>
using namespace si::interface;
using namespace si::core;
using namespace modules::core;



//
// Constructor/Destructor.
//

DefaultWrapper::DefaultWrapper()
    : _context(0), _config(0), _invoke_time(0)
{ }


DefaultWrapper::~DefaultWrapper()
{ }

//
// Member functions.
//

void DefaultWrapper::docBase(const string &docBase)
{
    _doc_base = docBase;
}


string DefaultWrapper::docBase()
{
    return _doc_base;
}


void DefaultWrapper::servletClassPath(const string& path)
{
    _class_path = path;
}


string DefaultWrapper::servletClassPath()
{
    return _class_path;
}


void DefaultWrapper::servletClassName(const string& name)
{
    _class_name = name;
}


string DefaultWrapper::servletClassName()
{
    return _class_name;
}


void DefaultWrapper::setManager(shared_ptr<Manager> manager)
{
    _manager = manager;
}


shared_ptr<Manager> DefaultWrapper::getManager()
{
    return _manager;
}


void DefaultWrapper::addMapping(const string &pattern)
{
    _mapping_list.push_back(pattern);
}


vector<string> DefaultWrapper::getMapping()
{
    return _mapping_list;
}


bool DefaultWrapper::match(const string& uri) const
{
#ifdef DEBUG
    std::cerr << "DefaultWrapper:match(" << uri << ")" << std::endl;
    std::vector<std::string>::const_iterator n = _mapping_list.begin();
    for (; n != _mapping_list.end(); n++)
        std::cerr << *n << std::endl;
#endif
    std::vector<std::string>::const_iterator i =
            std::find_if(_mapping_list.begin(), _mapping_list.end(),
                         std::bind2nd(pattern_compare(), uri));
#ifdef DEBUG
    std::cerr << "DefaultWrapper:match:" << (i != _mapping_list.end()) << std::endl;
#endif
    return i != _mapping_list.end();
}


void DefaultWrapper::addInitParam(const string &n, const string &v)
{
    _init_para_map.insert(init_para_map_t::value_type(n, v));
}

servlet::ServletContext& DefaultWrapper::getServletContext()
{
    return *_context;
}

void DefaultWrapper::setServletContext(servlet::ServletContext& c)
{
    _context = &c;
}

time_t DefaultWrapper::invokeTime()
{
    return _invoke_time;
}


/**
 * Servlet 饹Υɽ.
 *
 * Servlet ɤΤϳƥƥȤΥ˥ԡ⥸塼
 * ФƹԤʤ桼 webapps ΥꥸʥϤΤޤ޿ʤ.
 * ϰ OS Ǥ Υ⥸塼(*.so) 񤭴
 * ɤƤץ(Ƹ) SEGV ȤʤäƤޤ.
 */
void DefaultWrapper::loadInstance()
{
    using namespace servlet;

    /* 饹ѥºݤΥ饹̾ */

    /*
     * 饹ѥˤUnix Ʊͤ "/" Ƕڤ "." Ƕڤ
     * Ȥ˵ƤƤΤǤǤν "/" ѴƤԤʤ.
     */
    std::string class_name = _class_name;

    string::size_type pos;
    while ((pos = class_name.find('.')) != std::string::npos)
        class_name.replace(pos, 1, "/");

    /* ԡեѥ */
    std::string src_file = _class_path +"/"+ class_name +sl::config::dso_suffix;

    /*
     * ԡΥǥ쥯ȥ.
     * ԡ getWorkDir  "WEB-INF/classes" ɲäΤȤ.
     * WEB-INF/classes ϤޤǤ webץꥱȤƤǤ
     * ƥʤݻ Servlet(default/cgi) Ϥ
     * "WEB-INF/classes" ۲¸ߤʤ.
     * ʤΤǤǤ class-path  WEB-INF/classes ¸ߤʤ
     * ƥʤݻ Servlet ȸʤ "WEB-INF/classes" Ϳʤ.
     */
    Context* parent = dynamic_cast<Context*>(getParent());
    std::string work_dir = parent->getWorkDir();

    std::string dst_dir;    // ԡǥ쥯ȥ
    std::string dst_file;   // ԡǥ쥯ȥ + ե̾
    if (src_file.find("WEB-INF") != std::string::npos) {
        dst_dir  = work_dir + "/WEB-INF/classes";
        dst_file = work_dir + "/WEB-INF/classes/" + class_name + sl::config::dso_suffix;
    } else {
        dst_dir  = work_dir;
        dst_file = work_dir + "/" +  class_name + sl::config::dso_suffix;
    }

    getLogger()->log("class_name:" + class_name, Logger::DEBUG_LOG);
    getLogger()->log("src_file  :" + src_file, Logger::DEBUG_LOG);
    getLogger()->log("dst_file  :" + dst_file, Logger::DEBUG_LOG);
    getLogger()->log("It Copies it from \"" + src_file + "\" to \"" +
                     dst_file + "\" .", Logger::DEBUG_LOG);

    try {
        fs::create_directories(fs::path(dst_file).branch_path());
        fs::remove(dst_file);
        fs::copy_file(src_file, dst_file);
    } catch(fs::filesystem_error& e) {
        getLogger()->log("It failed in the copy of \"" + src_file + "\".", e);
        return;
    }

    try {
"XXXXX";
        if (chdir(dst_dir.c_str()) != 0) {
            getLogger()->log("\tIt failed in the initialization of the Servlet.", Logger::ERROR_LOG);
            return;
        }

        // ԡ *.so ɤ
        shared_ptr<Servlet> p = loadClass<Servlet>(class_name);

        if ((_servlet = dynamic_pointer_cast<http::HttpServlet>(p))) {
            getLogger()->log("DefaultWrapper::loadInstance - "
                             "load servlet instance of HttpServiet.",
                             Logger::DEBUG_LOG);

        } else if ((_servlet = dynamic_pointer_cast<GenericServlet>(p))) {
            getLogger()->log("DefaultWrapper::loadInstance - "
                             "load servlet instance of GenericServiet.",
                             Logger::DEBUG_LOG);
        } else {
            getLogger()->log("DefaultWrapper::loadInstance - "
                             "load servlet instance of unknown.",
                             Logger::ERROR_LOG);
            return;
        }

        if (chdir(home().c_str()) != 0) {
            getLogger()->log("\tIt failed in the initialization of the Servlet.", Logger::ERROR_LOG);
            return;
        }

        _servlet->init(*_config);

    } catch(fs::filesystem_error& e) {
        getLogger()->log("It failed in the copy processing of "
                         "\"" + src_file + "\".", e);
    } catch(sl::java::lang::ClassNotFoundException& e) {
        getLogger()->log("\tIt failed in the copy processing of "
                        "\"" + src_file + "\".", e);
    } catch(std::exception& e) {
        getLogger()->log("\tIt failed in the initialization of the Servlet.",e);
    }

    _invoke_time = ::time(0);
#if 1
    getLogger()->log("DefaultWrapper::loadInstance - invoke time is " +
                     lexical_cast<string>(_invoke_time), Logger::DEBUG_LOG);
#endif
}


void DefaultWrapper::unloadInstance()
{
    getLogger()->log("DefaultWrapper::unloadInstance", Logger::DEBUG_LOG);

    if (!_servlet)
        return;

    _servlet->destroy();
    _servlet.reset();

    // ǥ쥯ȥ˥ԡ줿 *.so ե򥢥ɤ
    string cl = _class_name;
    string::size_type pos = cl.rfind("/");
    if (pos != string::npos)
        cl.erase(0, pos + 1);

    Context* parent = dynamic_cast<Context*>(getParent());
    string work_dir;
    try {
        work_dir = parent->getWorkDir() + "/" +
                      fs::path(_class_name).branch_path().string() + "/" + cl;
    } catch (fs::filesystem_error& e) {
        getLogger()->log("\tunload target not found:" + _class_name,
                         Logger::WARN_LOG);
        return;
    }

//    replace_all(work_dir, "//", "/");

    while ((pos = work_dir.find("//")) != std::string::npos)
        work_dir.replace(pos, 2, "/");

    unloadClass(work_dir);
}


void DefaultWrapper::init()
{
    if (!getLoader())
        setLoader(getParent()->getLoader());

    if (!getLogger())
        setLogger(getParent()->getLogger());

    shared_ptr<Logger> logger = getLogger();
    logger->log("DefaultWrapper::init - Beginning of initialization.",
                Logger::DEBUG_LOG);
    logger->log("DefaultWrapper::init - servletClassPath " + servletClassPath(),
                Logger::DEBUG_LOG);
    logger->log("DefaultWrapper::init - servletClassName " + servletClassName(),
                Logger::DEBUG_LOG);
    logger->log("DefaultWrapper::init - className    --- " + className(),
                Logger::DEBUG_LOG);

    if (servletClassName().empty()) {
        string msg("The name to Servlet is not set.");
        logger->log(msg, Logger::ERROR_LOG);
        throw std::runtime_error(msg);
    }

    if (!getParent()) {
        string msg("The class that becomes parents is not set.");
        logger->log(msg, Logger::ERROR_LOG);
        throw std::runtime_error(msg);
    }

    if (!getManager()) {
        string msg("The class that becomes \"Manager\" is not set.");
        getLogger()->log(msg, Logger::ERROR_LOG);
        throw std::runtime_error(msg);
    }

    if (!getServletContext()) {
        string msg("The class that becomes \"ServletContext\" is not set.");
        getLogger()->log(msg, Logger::ERROR_LOG);
        throw std::runtime_error(msg);
    }

    //
    // ServletConfig μ饹󥹥.
    //
    ServletConfigImpl* p = new ServletConfigImpl();
    p->setServletContext(_context);
    p->setServletName(_class_name);
    p->setInitParameter(_init_para_map);

    _config = p;

    loadInstance();
}


void DefaultWrapper::destroy()
{
    shared_ptr<Logger> logger = getLogger();
    logger->log("DefaultWrapper::destroy - Beginning of termination.",
                Logger::DEBUG_LOG);

    logger->log("DefaultWrapper::destroy - className    --- " + className(),
                Logger::DEBUG_LOG);

    logger->log("DefaultWrapper::destroy - servletClassPath "
                + servletClassPath(), Logger::DEBUG_LOG);
    logger->log("DefaultWrapper::destroy - servletClassName "
                + servletClassName(), Logger::DEBUG_LOG);

    setManager(shared_ptr<Manager>());

    unloadInstance();

    delete _config;
}


void DefaultWrapper::service(Connection *)
{
}

void DefaultWrapper::service(servlet::ServletRequest& req,
                             servlet::ServletResponse& res)
{
    instance()->service(req, res);
}


servlet::Servlet* DefaultWrapper::instance()
{
    if (!_servlet) {
        getLogger()->log("DefaultWrapper::instance - Servlet not loaded",
                        Logger::WARN_LOG);
        return 0;
    }
    return _servlet.get();
}
