/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <iostream>
#include <boost/bind.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/function.hpp>
using namespace std;
using namespace boost;

#include "DefaultService.h"

#include <si/interface/Config.h>
#include <si/interface/Connector.h>
#include <si/interface/Engine.h>
#include <si/interface/Logger.h>
#include <si/interface/Loader.h>
using namespace si::interface;
using namespace modules::core;


static const std::string DEFAULT_ENGINE = "modules.core.DefaultEngine";
static const std::string DEFAULT_VALVE  = "modules.core.DefaultServiceValve";
static const std::string DEFAULT_CONNECTOR = "modules.connector.HttpConnector";


//
// Constructor/Destructor.
//

DefaultService::DefaultService()
{ 
}


DefaultService::~DefaultService()
{
}


//
// Member functions.
//


void DefaultService::addConnector(shared_ptr<Connector> connector)
{
    _connector_list.insert(connector);
}


set<shared_ptr<Connector> > DefaultService::getConnectorList()
{
    return _connector_list;
}

//
// Ǥ Engine 󥹥󥹤ʣϿǤʤ.
// ¿ʬ Tomcat ʣϿǤưǽ...
//
void DefaultService::setEngine(shared_ptr<Engine> engine)
{ 
    _engine = engine;
}


//
// Ǥ Engine 󥹥󥹤ʣϿǤʤ.
// ¿ʬ Tomcat ʣϿǤưǽ...
//
shared_ptr<Engine> DefaultService::getEngine() const
{ 
    return _engine;
}

void DefaultService::init()
{
    if (getLoader())
        setLoader(getParent()->getLoader());

    //
    // Logger ¸ߤ Logger ɤ.
    //
    if (config().exist("Logger")) {
        si::interface::Config c = config().get("Logger");
        c.attr("home", home());

        shared_ptr<Logger> logger;
        try {
            logger = loadLogger(c);
            logger->open();
        } catch (std::exception& e) {
            logger = shared_ptr<Logger>();
            unloadLogger();

            string msg("DefaultServer::init - \"Logger\" failed in opening.");
            msg = msg + "reason is '" + e.what() + "'.";
            throw runtime_error(msg);
        }
    } else {
        setLogger(getParent()->getLogger());
    }

    shared_ptr<Logger> logger = getLogger();

    logger->log("DefaultService::init - Beginning of initialization.",
                Logger::INFO_LOG);

    //
    // ꤵ줿 Connector ɤ.
    //
    vector<si::interface::Config> clist = config().gets("Connector");
    vector<si::interface::Config>::const_iterator i = clist.begin();
    for (; i != clist.end(); i++) {
        string class_name = i->attr("className");
        if (class_name.empty())
            class_name = DEFAULT_CONNECTOR;

        shared_ptr<Connector> p;
        try {
            p = loadClass<Connector>(class_name);
            if (!p) {
                logger->log("DefaultService::init - "
                        "It failed in loading \"Connector\"",Logger::ERROR_LOG);
                continue;
            }

            //p->setPort(boost::lexical_cast<int>(i->attr("port")));
            p->name(i->attr("name"));
            p->home(home());
            p->config(*i);
            p->className(class_name);
            p->setParent(this);
            addConnector(p);
        } catch (boost::bad_lexical_cast) {
            string msg("DefaultService::init - The module \"Connector\" "
                        "is not loaded.");
            logger->log(msg, Logger::ERROR_LOG);

            p = shared_ptr<Connector>();
            unloadClass(class_name);

            //throw runtime_error(msg);
    
        } catch (std::exception& e) {
            string msg("DefaultService::init - "
                        "It failed in loading \"Connector\"");
            logger->log(msg, e);
            throw runtime_error(msg);
        }
    }

    //
    // ꤵ줿 Engine ɤ.
    //
    if (!config().exist("Engine")) {
        string msg("DefaultService::init - \"Engine\" element doesn't exist.");
        logger->log(msg, Logger::ERROR_LOG);
        throw runtime_error(msg);
    }

    si::interface::Config c = config().get("Engine");

    if (c.attr("name").empty()) {
        std::string msg("DefaultService::init - \"name\" doesn't exist "
                        "in the attribute value of \"Engine\" element.");
        logger->log(msg, Logger::ERROR_LOG);
    }

    string class_name = c.attr("className");
    if (class_name.empty())
        class_name = DEFAULT_ENGINE;

    try {
        shared_ptr<Engine> p = loadClass<Engine>(class_name);
        if (p) {
            p->home(home());
            p->config(c);
            p->className(class_name);
            p->setParent(this);
            setEngine(p);
        } else
            logger->log("DefaultService::init - "
                        "It failed in loading \"Engine\"", Logger::ERROR_LOG);
    } catch (std::exception& e) {
        string msg("DefaultService::init - The module \"Engine\" " "is not loaded.");
        logger->log(msg, e);
        throw runtime_error(msg);
    }

    //
    // ꤵ줿 Valve ɤ.
    //
    loadValves(config().gets("Valve"));

    setBase(this);

    //
    // Engine 󥹥󥹤.
    //
    try {
        _engine->init();
    } catch (std::runtime_error& e) {
        std::string class_name = _engine->className();
        string msg = "DefaultService::init - \"Engine\" initialization "
                     "failed.";
        logger->log(msg, e);
        _engine = shared_ptr<Engine>();
        unloadClass(class_name);
    }

    //
    // Connector 󥹥󥹿ʬΥåɴؿ.
    //
    logger->log("DefaultService::init - The connector is started. ",
                Logger::INFO_LOG);

    // void (*handler)(Connection&);
    boost::function<void (Connection&)> handler;
    handler = boost::bind(&DefaultService::invoke, this, _1);

    vector<string> nlist;
    connector_list_t::iterator con = _connector_list.begin();
    for (; con != _connector_list.end(); con++) {
        try {
            (*con)->acceptHandler(handler);
            (*con)->init();
            (*con)->accept();
        } catch (std::exception& e) {
            string msg = "DefaultService::init - "
                         "\"Connector\" initialization failed.";
            logger->log(msg, e);

            string class_name = (*con)->className();
            _connector_list.erase(con);
            nlist.push_back(class_name);
        }
    }

    vector<string>::iterator j = nlist.begin();
    for (; j != nlist.end(); j++)
        unloadClass(*j);

    if (_connector_list.size() == 0) {
        string msg("DefaultService::init - The module \"Connector\" "
                    "is not loaded.");
        logger->log(msg, Logger::WARN_LOG);
        throw runtime_error(msg);
    }

    _available = true;
}

void DefaultService::destroy()
{
    if (!_available)
        return;

    _available = false;
    getLogger()->log("DefaultService::destroy - Beginning of termination.",
                     Logger::INFO_LOG);

    //
    // ɤƤ Connector Ф destroy дؿ¹Ԥ
    // Ѥ nlist  Connector Υ饹̾¸Ƥ. 
    //
    vector<string> nlist;
    vector<string>::iterator j;

    connector_list_t::iterator i = _connector_list.begin();
    for (; i != _connector_list.end(); i++) {
        nlist.push_back((*i)->className());
        (*i)->destroy();
    }
    _connector_list.clear();
#if 1
    for (j = nlist.begin(); j != nlist.end(); j++) {
        try {
            unloadClass(*j);
        } catch (std::exception& e) {
            std::cerr << e.what() << std::endl;
        }
    }
#endif
    //
    // Engine ɤƤϻȤߤ˥ɤ.
    //
    if (_engine) {
        std::string class_name = _engine->className();
        _engine->destroy();
        _engine = shared_ptr<Engine>();
        unloadClass(class_name);
    }

    //
    // DefaultPipeline ݻƤƤ Valve ƻȤߤ
    // Ѥ nlist  Valve Υ饹̾¸Ƥ. 
    //
    nlist.clear();

    vector<shared_ptr<Valve> > valve_list = getValve();
    vector<shared_ptr<Valve> >::iterator v = valve_list.begin();
    for (; v != valve_list.end(); v++) {
        removeValve(*v);
        nlist.push_back((*v)->config().attr("className"));
    }
    valve_list.clear();

    for (j = nlist.begin(); j != nlist.end(); j++)
        unloadClass(*j);

    unloadLogger();
}

void DefaultService::loadValves(const vector<si::interface::Config>& clist)
{
    vector<si::interface::Config>::const_iterator i = clist.begin();
    for (; i != clist.end(); i++) {
        try {
            std::string class_name = i->attr("className");
            shared_ptr<Valve> p = loadClass<Valve>(class_name);
            if (p) {
                p->config(*i);
                addValve(p);
            }
        } catch (std::exception& e) {
            getLogger()->log("DefaultService::loadValve - "
                            "It failed in loading \"Valve\"", e);
        }
    }

    shared_ptr<Valve> p = loadClass<Valve>(DEFAULT_VALVE);
    if (p) {
        si::interface::Config c;
        c.attr("className", DEFAULT_VALVE);
        p->config(c);
        addValve(p);
    }
}
