/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <iostream>
#include <string>
#include <boost/shared_ptr.hpp>
using namespace std;
using namespace boost;

#include "DefaultServer.h"

#include <si/interface/Connection.h>
#include <si/interface/Loader.h>
#include <si/interface/Logger.h>
#include <si/interface/Service.h>
using namespace si::interface;
using namespace modules::core;


static const string DEFAULT_SERVICE = "modules.core.DefaultService";
static const string DEFAULT_LOGGER  = "modules.logger.DefaultLogger";

//
// Constructor/Destructor.
//

DefaultServer::DefaultServer()
{ 
}


DefaultServer::~DefaultServer()
{
}


//
// Member functions.
//

void DefaultServer::setService(shared_ptr<Service> service)
{
    _service_list.insert(service);
}

shared_ptr<Service> DefaultServer::getService(const string& name)
{
    service_list_t::iterator i = _service_list.begin();
    for (; i != _service_list.end(); i++) {
        if ((*i)->name() == name)
            return *i;
    }
    return shared_ptr<Service>();
}

void DefaultServer::init()
{
    //
    // γǧ.
    // Loader ǤʤǤϿƤΥѤ뤿ᡢ
    // ФȤƥȥåץ٥Υ¸ߤʤϰʹߤνԤʤ鷺
    // ƥʤεǽߤ.
    //
    if (!getLoader())
        throw runtime_error("DefaultServer::init - The module \"Loader\" "
                            "is not loaded.");
    //
    // ФȤƥȥåץ٥ Logger .
    //
    si::interface::Config c;
    if (config().exist("Logger")) {
        c = config().get("Logger");
        c.attr("home", home());
    } else {
        c.attr("home", home());
        c.attr("className", DEFAULT_LOGGER);
        c.attr("directory", "logs");
        c.attr("prefix", "server");
        c.attr("suffix", "log");
        c.attr("verbosityLevel", "INFO");
    }

    shared_ptr<Logger> logger;
    try {
        logger = loadLogger(c);
        logger->open();
    } catch (std::exception& e) {
        logger = shared_ptr<Logger>();
        unloadLogger();

        string msg("DefaultServer::init - \"Logger\" failed in opening.");
        msg = msg + " reason is '" + e.what() + "'.";
        throw runtime_error(msg);
    }

    logger->log("DefaultServer::init - \"Logger\" module used at the "
                "\'TOP level\' was loaded. Use begins.", Logger::DEBUG_LOG);
    logger->log("DefaultServer::init - Beginning of initialization.",
                 Logger::INFO_LOG);

    //
    // Service 󥹥󥹤ɤ.
    //
    if (!config().exist("Service")) {
        std::string msg("DefaultServer::init - "
                        "\"Service\" element doesn't exist.");
        logger->log(msg, Logger::ERROR_LOG);
        throw runtime_error(msg);
    }

    c = config().get("Service");

    if (c.attr("name").empty()) {
        std::string msg("DefaultServer::init - \"name\" doesn't exist "
                        "in the attribute value of \"Service\" element.");
        logger->log(msg, Logger::ERROR_LOG);
    }

    string class_name = c.attr("className");
    if (class_name.empty())
        class_name = DEFAULT_SERVICE;
    
    boost::shared_ptr<Service> service = loadClass<Service>(class_name);
    if (!service) {
        string msg = "DefaultServer::init - failed in initialization.";
        logger->log(msg, Logger::ERROR_LOG);
        throw runtime_error(msg);
    }
    
    service->config(c);
    service->home(home());
    service->className(class_name);
    service->setParent(this);
    service->setLoader(getLoader());
    service->setLogger(getLogger());
    setService(service);

    // Service::init ǥ顼ȯϡξǥ饹Υɤޤ
    // Ԥ١ǰöȤߤ
    service = shared_ptr<Service>();

    vector<string>    nlist;

    service_list_t::iterator i = _service_list.begin();
    for (; i != _service_list.end(); i++) {
        try {
            (*i)->init();
        } catch (std::exception& e) {
            std::string class_name = (*i)->className();

            string msg = "DefaultServer::init - \"Service\" initialization "
                         "failed.";
            logger->log(msg, e);

            _service_list.erase(i);
            nlist.push_back(class_name);
        }
    }

    vector<string>::iterator j = nlist.begin();
    for (; j != nlist.end(); j++)
        unloadClass(*j);

    _available = true;
}

void DefaultServer::destroy()
{
    if (getLogger()) {
        getLogger()->log("DefaultServer::destroy - Beginning of termination.",
                         Logger::INFO_LOG);
    }

    if (!_available)
        return;

    _available = false;

    vector<string>    nlist;

    //
    // ɤƤ Service Ф destroy дؿθƤӽФԤ
    // Ѥ˥饹̾ݻƤ.
    //
    service_list_t::iterator i = _service_list.begin();
    for (; i != _service_list.end(); i++) {
        nlist.push_back((*i)->className());
        (*i)->destroy();
    }

    // ɤƤλȤ.
    _service_list.clear();

    vector<string>::iterator j = nlist.begin();
    for (; j != nlist.end(); j++)
        unloadClass(*j);

    unloadLogger();
}

void DefaultServer::service(Connection *)
{
    getLogger()->log("DefaultServer::service.", Logger::INFO_LOG);
#if 0
    if (!_available) {
        getLogger()->log("DefaultServer::service - unavaiable",
                         Logger::ERROR_LOG);
        throw runtime_error("DefaultServer::service - unavaiable");
    }

    service_list_t::iterator i = _service_list.begin();
    for (; i != _service_list.end(); i++)
        (*i)->service(NULL);
#endif
}


//
// private member functions.
//

