/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <iostream>
#include <boost/algorithm/string/replace.hpp>
#include <boost/bind.hpp>
#include <boost/filesystem/convenience.hpp>
#include <boost/filesystem/exception.hpp>
#include <boost/filesystem/operations.hpp>
#include <boost/filesystem/path.hpp>
#include <boost/lexical_cast.hpp>
using namespace std;
using namespace boost;
namespace fs = boost::filesystem;

#include "DefaultHost.h"
#include <si/interface/Config.h>
#include <si/interface/Context.h>
#include <si/interface/Engine.h>
#include <si/interface/Loader.h>
#include <si/interface/Logger.h>
#include <si/interface/Manager.h>
#include <si/interface/Realm.h>
#include <si/core/ContextConfig.h>
using namespace si::interface;
using namespace si::core;
using namespace modules::core;


// work directory : "${Home}/work/${Host.Name}/${Context.Path}"
// conf directory : "${Home}/conf/${Host.Name}/"
static const string DEFAULT_WORK_DIR    = "/work";
static const string DEFAULT_CONF_DIR    = "/conf";
static const string DEFAULT_DEPLOY_XML  = "false";
static const int    DEFAULT_INTERVAL    = 5;

static const std::string DEFAULT_CONTEXT = "modules.core.DefaultContext";
static const std::string DEFAULT_MANAGER = "modules.core.DefaultManager";
static const std::string DEFAULT_VALVE   = "modules.core.DefaultHostValve";
static const std::string DEFAULT_LOGGER  = "modules.logger.DefaultLogger";
static const std::string DEFAULT_REALM   = "modules.realm.MemoryRealm";

//
// Constructor/Destructor.
//

DefaultHost::DefaultHost()
    : _is_load_logger(false), _is_load_realm(false)
{ 
}


DefaultHost::~DefaultHost()
{
}


//
// Member functions.
// 

void DefaultHost::appBase(const string &app_base)
{
    _app_base = app_base;
}


string DefaultHost::appBase()
{
    return _app_base;
}


void DefaultHost::setConfDir(const string &dir)
{
    _conf_dir = dir;
}


string DefaultHost::getConfDir()
{
    return _conf_dir;
}


void DefaultHost::setWorkDir(const string &dir)
{
    _work_dir = dir;
}


string DefaultHost::getWorkDir()
{
    return _work_dir;
}


void DefaultHost::setDefaultContext(shared_ptr<Context> context)
{ 
    _default_context = context;
    _default_context->setParent(this);
}


shared_ptr<Context> DefaultHost::getDefaultContext()
{ 
    return _default_context;
}


void DefaultHost::setContext(shared_ptr<Context>& context)
{ 
    _context_list.push_back(context);
}


shared_ptr<Context> DefaultHost::getContext(const string& context_path)
{
    context_list_t::iterator i = _context_list.begin();
    for (; i != _context_list.end(); i++) {
        if ((*i)->match(context_path))
            return *i;
    }
    return shared_ptr<Context>();
}


vector<shared_ptr<Context> > DefaultHost::getContextSet()
{
    return _context_list;
}


void DefaultHost::setManager(shared_ptr<Manager> manager)
{ 
    _manager = manager;
}


shared_ptr<Manager> DefaultHost::getManager()
{
    return _manager;
}


void DefaultHost::init()
{
    if (!getLoader())
        setLoader(getParent()->getLoader());

    //
    // Logger ¸ߤ Logger ɤ.
    //
    if (config().exist("Logger")) {
        si::interface::Config c = config().get("Logger");
        c.attr("home", home());

        shared_ptr<Logger> logger;
        try {
            logger = loadLogger(c);
            logger->open();
        } catch (std::exception& e) {
            logger = shared_ptr<Logger>();
            unloadLogger();

            string msg("DefaultServer::init - \"Logger\" failed in opening.");
            msg = msg + "reason is '" + e.what() + "'.";
            throw runtime_error(msg);
        }
    } else {
        setLogger(getParent()->getLogger());
    }

    shared_ptr<Logger> logger = getLogger();

    logger->log("DefaultHost::init - Beginning of initialization.",
                Logger::INFO_LOG);

    //
    // ۥ̾.
    //
    if (name().empty()) {
        name(config().attr("name"));
        if (name().empty()) {
            string msg("DefaultHost::init: - \"name\" doesn't exist "
                        "in the attribute value.");
            logger->log(msg, Logger::ERROR_LOG);
            throw std::runtime_error(msg);
        }
    }

    //
    // WEBץꥱΥ롼ȥǥ쥯ȥ.
    //
    if (_app_base.empty()) {
        appBase(config().attr("appBase"));
        if (_app_base.empty() || _app_base[0] == '/') {
            string msg("DefaultHost::init: - \"appBase\" doesn't exist "
                        "in the attribute value.");
            logger->log(msg, Logger::ERROR_LOG);
            throw std::runtime_error(msg);
        }
    }

    //
    // WEBץꥱѰեǼǥ쥯ȥ.
    //
    // ${Home}/conf/${Engine.Name}  ${Home}/work/${Engine.Name}
    // ǥ쥯ȥ꤬¸ߤʤ硢
    // ޤϡǥ쥯ȥʳǤϡǥ쥯ȥ.
    //
    string this_path = getParent()->name() + "/" + name();

    string path = home() + "/conf/" + this_path;
    setConfDir(path);

    logger->log("DefaultHost::init - configuration file in " + path,
                Logger::DEBUG_LOG);

    try {
        if (!fs::exists(path))
            fs::create_directory(path);
        else if (!fs::is_directory(path)) {
            fs::remove(path);
            fs::create_directory(path);
        }    
    } catch(fs::filesystem_error &e) {
        string msg("DefaultHost::init: - It failed in making the directory.");
        msg = msg + " The reason is '" + e.what() + "'.";
        logger->log(msg, Logger::ERROR_LOG);
        throw std::runtime_error(msg);
    }

    path = home() + "/work/" + this_path;
    setWorkDir(path);

    logger->log("DefaultHost::init - working directory is " + path,
                Logger::DEBUG_LOG);

    try {
        if (!fs::exists(path))
            fs::create_directories(path);
        else if (!fs::is_directory(path)) {
            fs::remove(path);
            fs::create_directories(path);
        }    
    } catch(fs::filesystem_error &e) {
        string msg("DefaultHost::init: - It failed in making the directory.");
        msg = msg + " The reason is '" + e.what() + "'.";
        logger->log(msg, Logger::ERROR_LOG);
        throw std::runtime_error(msg);
    }

    //
    // ƥȥե饰Ȥˤ̵ͭ.
    //
    if (_deploy_xml.empty()) {
        string deploy = config().attr("deployXML");
        if (deploy == "true" || deploy == "false")
            _deploy_xml = deploy;
        else
            _deploy_xml = DEFAULT_DEPLOY_XML;
    }

#if 0
    // ǥեȥƥȤ.
    if (!_default_context) {
            throw std::runtime_error("std::runtime_error :\"DefaultContext\" is invalid");
    }
#endif

    //
    // Realm ¸ߤ Realm ɤ.
    //
    if (config().exist("Realm")) {
        si::interface::Config c = config().get("Realm");
        c.attr("home", home());

        if (c.attr("className").empty())
            c.attr("className", DEFAULT_REALM);

        try {
            loadRealm(c);
        } catch (std::exception& e) {
            string msg("DefaultHost::loadRealm - \"Reaml\" failed in loading.");
            getLogger()->log(msg, e);
        }
    } else {
        std::string msg("DefaultHost::init - \"Realm\" element doesn't exist.");
        logger->log(msg, Logger::WARN_LOG);
    }

    loadContexts(config().gets("Context"));

    loadValves(config().gets("Valve"));


    logger->log("DefaultHost::init - name is   --- '" + name() + "'",
                Logger::DEBUG_LOG);
    logger->log("DefaultHost::init - appBase   --- '" + _app_base + "'",
                Logger::DEBUG_LOG);
    logger->log("DefaultHost::init - deployXML --- '" + _deploy_xml + "'",
                Logger::DEBUG_LOG);

    if (_deploy_xml == "true") {
        logger->log("DefaultHost::init - The directory like the "
                    "WEB application in \"" + appBase() + "\" is retrieved.",
                    Logger::INFO_LOG);

        // WEBץꥱ롼Ȱʲ'äݤǥ쥯ȥ'
        // Ф Context XML ե.
        string app_dir = home() + "/" + appBase();

        fs::directory_iterator i(app_dir);
        fs::directory_iterator end;
        for (; i != end; i++) {
            if (fs::is_directory(*i))
                createContextXML(i->string());
        }

        string xml_dir = home() + DEFAULT_CONF_DIR +
                        "/" + getParent()->name() + "/" + name();

        fs::directory_iterator j(xml_dir);
        for (; j != end; j++) {
            if (!fs::is_directory(*j) && extension(*j) == ".xml")
                // ƥȥե饰XMLեɤ߹.
//                shared_ptr<Context> p = loadFromFragmentContextXML(j->string());
                loadFromFragmentContextXML(j->string());
        }
    }

    if (getRealm()) {
        try {
            getRealm()->init();
        } catch (std::runtime_error& e) {
            std::string class_name = getRealm()->className();
            string msg = "DefaultHost::init - \"Realm\" initialization failed.";
            logger->log(msg, e);
    
            unloadRealm();
        }
    }

    logger->log("Host: count of Context:" +
                boost::lexical_cast<std::string>(_context_list.size()));
    vector<string>    nlist;

    context_list_t::iterator i = _context_list.begin();
    for (; i != _context_list.end();) {
        try {
            (*i)->setParent(this);
            (*i)->init();
            ++i;
        } catch (std::runtime_error &e) {
            std::string class_name = (*i)->className();

            string msg = "DefaultHost::init - \"Context\" initialization "
                         "failed.";
            logger->log(msg, e);

            i = _context_list.erase(i);
            nlist.push_back(class_name);
        }
    }

    if (_context_list.empty())
        logger->log("\"Context\" doesn't exist any.", Logger::WARN_LOG);

    vector<string>::iterator j = nlist.begin();
    for (; j != nlist.end(); j++)
        unloadClass(*j);

    //
    // ưԤʤåɤư.
    //
    string auto_deploy = config().attr("autoDeploy");
    _deploy_interval = DEFAULT_INTERVAL;
    if (auto_deploy == "true") {
        getLogger()->log("Host : autoDeploy is true. check start.",
                         Logger::DEBUG_LOG);

        _deploy = true;
        _deploy_thread =
            new boost::thread(boost::bind(&DefaultHost::deployCheck, this));
    }

    // ValveContext κԤȤϿƤ.
    setBase(this);
}


void DefaultHost::destroy()
{
    getLogger()->log("DefaultHost::destroy - Beginning of termination.",
                     Logger::INFO_LOG);

    //
    // XXXXX ʤ FreeBSD Ǥ "true"  config Υǡ
    // ΤޤӤ...
    //
    // if (config().attr("autoDeploy") == "true") {
    //

    if (std::string("true") == config().attr("autoDeploy")) {
        {
            boost::mutex::scoped_lock lock(_deploy_mutex);
            _deploy = false;
            _deploy_cond.notify_all();
        }
        _deploy_thread->join();
        delete _deploy_thread;
    }

    //
    // Host ݻƤƤ Context ƻȤߤ
    // Ѥ nlist  Context Υ饹̾¸Ƥ. 
    //
    vector<string>    nlist;

    context_list_t::iterator i = _context_list.begin();
    for (; i != _context_list.end(); i++) {
        nlist.push_back((*i)->className());
        (*i)->destroy();
        (*i).reset();
    }
    _context_list.clear();

    vector<string>::iterator j = nlist.begin();
    for (; j != nlist.end(); j++)
        unloadClass(*j);

    if (getRealm()) {
        getRealm()->destroy();
        unloadRealm();
    }

    //
    // DefaultPipeline ݻƤƤ Valve ƻȤߤ
    // Ѥ nlist  Valve Υ饹̾¸Ƥ. 
    //
    nlist.clear();

    vector<shared_ptr<Valve> > valve_list = getValve();
    vector<shared_ptr<Valve> >::iterator v = valve_list.begin();
    for (; v != valve_list.end(); v++) {
        removeValve(*v);
        nlist.push_back((*v)->config().attr("className"));
    }
    valve_list.clear();

    for (j = nlist.begin(); j != nlist.end(); j++)
        unloadClass(*j);

    unloadLogger();
}


//
// --- private -------------------------------------------------------------- //
// 
bool DefaultHost::createContextXML(const std::string& path)
{
    // Context ʲ WEB-INF ǥ쥯ȥ꤬¸ߤƤʤ̵
    std::string web_inf = path + "/WEB-INF";
    if (!fs::exists(web_inf) || !fs::is_directory(web_inf))
        return false;

    string context_path = fs::path(path).leaf();

    // ˥ƥXML¸ߤ뤫
    if (findContextFlagmentXML(context_path + ".xml"))
        return true;

    // ˥ɺѤΥƥȤǤ뤫
    if (getContext(context_path))
        return true;

    // ƥXML񤭽Ф.
    return outputContextXML(context_path);
}


bool DefaultHost::findContextFlagmentXML(const std::string &path)
{
    string xml_dir = home() + DEFAULT_CONF_DIR +
                "/" + getParent()->name() + "/" + name();

    fs::directory_iterator i(xml_dir);
    fs::directory_iterator end;
    for (; i != end; i++) {
        if (fs::path(*i).leaf() == path) {
            if (getLogger())
                getLogger()->log("ContextFlagmentXML:" + path + " is found.", Logger::DEBUG_LOG);
            return true;
        }
    }
    if (getLogger())
        getLogger()->log("ContextFlagmentXML:" + path + " is not found.", Logger::DEBUG_LOG);
    return false;
}


bool DefaultHost::outputContextXML(const std::string &path)
{
    string xml_dir = home() + DEFAULT_CONF_DIR + "/" +
                     getParent()->name() + "/" + name();
    string file = xml_dir + "/" + path + ".xml";

    getLogger()->log("DefaultHost::outputContextXML - "
                     "The fragmentation context XML file is made, and output. "
                     "'" + path + "'", Logger::INFO_LOG);

    ofstream ofs(file.c_str(), ios::out | ios::binary);
    if (!ofs) {
        cerr << "can't open file :" << file << endl;
        return false;
    }

    // XXXXX
    std::string url_path = (path == "ROOT"||path == "/ROOT") ? "/" : "/" + path;

    ofs << "<?xml version='1.0'?>" << endl;
    ofs << "<Context docBase=\"" << path << "\" ";
    ofs << "path=\"" << url_path << "\">" << endl;
    ofs << "  <Valve name=\"DigestAutenticator\" className=\"modules.valve.DigestAuthenticateValve\" />" << endl;;
    ofs << "  <Valve name=\"BasicAutenticator\" className=\"modules.valve.BasicAuthenticateValve\" />" << endl;;
    ofs << "</Context>" << endl;;

    ofs.close();

    return true;
}


shared_ptr<Context> DefaultHost::loadFromFragmentContextXML(const std::string& path)
{
    string xml_dir = home() + DEFAULT_CONF_DIR +
                "/" + getParent()->name() + "/" + name();

    getLogger()->log("DefaultHost::loadFromFragmentContextXML '" +
                     fs::path(path).leaf() + "'",
                     Logger::INFO_LOG);

    ContextConfig c(path);
    return loadContext(c);
}


void DefaultHost::deployCheck()
{
    time_t i = _deploy_interval;

    getLogger()->log("Host : deployCheck " + lexical_cast<string>(_app_base) +
                     ". interval time is " + boost::lexical_cast<string>(i) +
                     " sec.", Logger::INFO_LOG);

    for (;;) {
        boost::mutex::scoped_lock lock(_deploy_mutex);
        if (!_deploy)
            break;

        xtime xt;
        xtime_get(&xt, boost::TIME_UTC);
        xt.sec += i;

        if (_deploy_cond.timed_wait(lock, xt))
            continue;

        //
        // оݤȤʤ륳ƥȤå.
        //
        context_list_t::iterator j = _context_list.begin();
        for (; j != _context_list.end(); j++) {
            shared_ptr<Context> c = *j;
            if (!c) {
#ifdef DEBUG
    cerr << "invalid Context:" << c->docBase() << endl;
#endif
                continue;
            }

            string file = home() + "/" + c->docBase() + "/WEB-INF/web.xml";
            boost::replace_all(file, "//", "/");

#ifdef DEBUG
    cerr << "context::docBase:[" << c->docBase() << "]" << endl;
    cerr << file << endl;
#endif
            time_t last = 0;
            if (fs::exists(file))
                last = fs::last_write_time(file);
#ifdef DEBUG
    cerr << "invoke      :" << c->invokeTime() << endl;
    cerr << "last access :" << last << endl;
#endif
            if (!last || c->invokeTime() < last) {
                c->destroy();
                j = _context_list.erase(j);
                unloadContext(c);
            }
        }

        //
        // ɲäΥƥȤ¸ߤƤ뤫å
        //
        fs::directory_iterator i(home() + "/" + appBase());
        fs::directory_iterator end;

        for (; i != end; i++) {
            if (!fs::is_directory(*i))
                continue;

            string path = i->string();
            string target = fs::path(*i).leaf();
#ifdef DEBUG
    cout << "deployCheck Target:" << target << endl;
#endif
            bool found = false;
            context_list_t::iterator j = _context_list.begin();
            for (; j != _context_list.end(); j++) {
                if (target == fs::path((*j)->docBase()).leaf())
                    found = true;
            }

            if (!found) {
#ifdef DEBUG
    cout << "deployCheck: trying start " << target << endl;
#endif
                if (createContextXML(path)) {
                    string xml = home() + DEFAULT_CONF_DIR + "/" +
                                getParent()->name() + "/" +
                                name() + "/" + target + ".xml";

                    shared_ptr<Context> p = loadFromFragmentContextXML(xml);
                    if (!p)
                        continue;

                    try {
                        p->init();
                    } catch(std::runtime_error &e) {
                        getLogger()->log("Host :", e);
                        //_context_list.remove(p);
                        _context_list.erase(std::remove(_context_list.begin(),
                                                        _context_list.end(), p),
                                            _context_list.end());
                        unloadContext(p);
                    }
                }
            }
        }
    }
}


void DefaultHost::loadContexts(const vector<si::interface::Config>& c)
{
    vector<si::interface::Config>::const_iterator i = c.begin();
    for (; i != c.end(); i++)
        loadContext(*i);
}

shared_ptr<Context> DefaultHost::loadContext(const si::interface::Config& c)
{
    string class_name = c.attr("className");
    if (class_name.empty())
        class_name = DEFAULT_CONTEXT;

    shared_ptr<Context> p;
    try {
         getLogger()->log("DefaultHost::loadContext '" + c.attr("path") + "'",
                         Logger::INFO_LOG);
        p = loadClass<Context>(class_name);
        if (!p) {
            unloadClass(class_name);
            return p;
        }
        p->home(home());
        p->className(class_name);
        p->config(c);
        p->path(c.attr("path"));
        p->setParent(this);
        setContext(p);
    } catch (std::exception& e) {
        string msg("DefaultHost::loadContext - \"Context\" failed in loading.");
        getLogger()->log(msg, e);
    }
    return p;
}

void DefaultHost::unloadContext(shared_ptr<Context>& p)
{
    string class_name = p->className();
    p = shared_ptr<Context>();
    try {
        getLoader()->unloadClass(class_name);
    } catch (std::exception& e) {
        string msg("DefaultHost::unloadContext - "
                    "\"Context\" failed in unloading.");
        msg = msg + " reason is '" + e.what() + "'.";
        getLogger()->log(msg, Logger::ERROR_LOG);
    }
}

void DefaultHost::loadValves(const vector<si::interface::Config>& c)
{
    vector<si::interface::Config>::const_iterator i = c.begin();
    for (; i != c.end(); i++) {
        try {
            shared_ptr<Valve> p = loadClass<Valve>(i->attr("className"));
            if (p) {
                p->config(*i);
                addValve(p);
            }
        } catch (std::exception& e) {
            getLogger()->log("DefaultHost::loadValve - "
                            "It failed in loading \"Valve\"", e);
        }
    }

    shared_ptr<Valve> p = loadClass<Valve>(DEFAULT_VALVE);
    if (p) {
        si::interface::Config c;
        c.attr("className", DEFAULT_VALVE);
        p->config(c);
        addValve(p);
    }
}

