/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <iostream>
#include <stdexcept>
#include <boost/filesystem/exception.hpp>
#include <boost/filesystem/operations.hpp>
#include <boost/filesystem/path.hpp>
using namespace std;
using namespace boost;

#include "DefaultEngine.h"

#include <si/shibainu.h>
#include <si/interface/Connection.h>
#include <si/interface/Context.h>
#include <si/interface/Host.h>
#include <si/interface/Loader.h>
#include <si/interface/Logger.h>
#include <si/interface/Service.h>
#include <si/interface/Valve.h>
using namespace si::interface;
using namespace modules::core;

static const string DEFAULT_HOST   = "modules.core.DefaultHost";
static const string DEFAULT_VALVE  = "modules.core.DefaultEngineValve";

//
// Constructor/Destructor.
//

DefaultEngine::DefaultEngine()
{ 
}


DefaultEngine::~DefaultEngine()
{
}


//
// Member functions.
//

void DefaultEngine::setHost(shared_ptr<Host> host)
{
    _host_list.insert(host);
}

shared_ptr<Host> DefaultEngine::getHost(const string& name)
{
    std::string tmp;

    if (name.empty())
        return getDefaultHost();

    string::size_type pos;
    if ((pos = tmp.find(":")) != string::npos)
        tmp.erase(pos);

    host_list_t::iterator i = _host_list.begin();
    for (; i != _host_list.end(); i++) {
        if ((*i)->name() == tmp)
            return *i;
    }
    return getDefaultHost();
}

shared_ptr<Host> DefaultEngine::getDefaultHost()
{
    host_list_t::iterator i = _host_list.begin();
    for (; i != _host_list.end(); i++) {
        if ((*i)->name() == _default_host)
            return *i;
    }
    return shared_ptr<Host>();
}

void DefaultEngine::setDefaultContext(shared_ptr<Context> context)
{ 
    _context = context;
}

shared_ptr<Context> DefaultEngine::getDefaultContext()
{ 
    return _context;
}


void DefaultEngine::init()
{
    if (!getLoader())
        setLoader(getParent()->getLoader());

    //
    // Logger ¸ߤ Logger ɤ.
    //
    if (config().exist("Logger")) {
        si::interface::Config c = config().get("Logger");
        c.attr("home", home());

        shared_ptr<Logger> logger;
        try {
            logger = loadLogger(c);
            logger->open();
        } catch (std::exception& e) {
            logger = shared_ptr<Logger>();
            unloadLogger();

            string msg("DefaultEngine::init - \"Logger\" failed in opening.");
            msg = msg + "reason is '" + e.what() + "'.";
            throw runtime_error(msg);
        }
    } else {
        setLogger(getParent()->getLogger());
    }

    shared_ptr<Logger> logger = getLogger();

    logger->log("DefaultEngine::init - Beginning of initialization.",
                Logger::INFO_LOG);

    //
    // Host 󥹥󥹤ɤ.
    //
    if (!config().exist("Host")) {
        std::string msg("DefaultEngine::init - "
                        "\"Host\" element doesn't exist.");
        logger->log(msg, Logger::ERROR_LOG);
        throw runtime_error(msg);
    }

    loadHosts(config().gets("Host"));

    if (name().empty()) {
        name(config().attr("name"));
        if (name().empty()) {
            string msg("DefaultEngine::init: - \"name\" doesn't exist "
                        "in the attribute value.");
            logger->log(msg, Logger::ERROR_LOG);
            throw runtime_error(msg);
        }
    }

    if (_default_host.empty()) {
        _default_host = config().attr("defaultHost");
        if (name().empty()) {
            string msg("DefaultEngine::init: - \"defaultHost\" doesn't exist "
                        "in the attribute value.");
            logger->log(msg, Logger::ERROR_LOG);
            throw runtime_error(msg);
        }
    }

    string path = home() + "/conf/" + name();

    logger->log("DefaultEngine::init: - The configuration file is preserved in "
                "\"" + path + "\" directory.", Logger::INFO_LOG);

    try {
        if (!boost::filesystem::exists(path))
            boost::filesystem::create_directory(path);
        else if (!boost::filesystem::is_directory(path)) {
            boost::filesystem::remove(path);
            boost::filesystem::create_directory(path);
        }
    } catch(boost::filesystem::filesystem_error& e) {
        logger->log("DefaultEngine::init - It failed in the confirmation or "
                    "making the directory.", e);
    }

    if (_host_list.size() == 0) {
        getLogger()->log("DefaultEngine::init - The module \"Host\" is not loaded",
                         Logger::ERROR_LOG);
    }

#if 0
    if (!_context) {
        cerr << "DefaultContext is not set" << endl;
        throw runtime_error("DefaultContext is not set");
    }
#endif

    loadValves(config().gets("Valve"));

    setBase(this);

    vector<string> nlist;

    host_list_t::iterator i = _host_list.begin();
    for (; i != _host_list.end(); i++) {
        try {
            (*i)->init();
        } catch (runtime_error &e) {
            std::string class_name = (*i)->className();
            string msg = "DefaultEngine::init - \"Host\" initialization "
                         "failed.";
            logger->log(msg, e);

            _host_list.erase(i);
            nlist.push_back(class_name);
        }
    }

    vector<string>::iterator j = nlist.begin();
    for (; j != nlist.end(); j++)
        unloadClass(*j);
}


void DefaultEngine::destroy()
{
    getLogger()->log("DefaultEngine::destroy - Beginning of termination.",
                     Logger::INFO_LOG);

    //
    // ɤƤ Host Ф destroy дؿ¹Ԥ
    // Ѥ nlist  Host Υ饹̾¸Ƥ. 
    //
    vector<string> nlist;

    host_list_t::iterator i = _host_list.begin();
    for (; i != _host_list.end(); i++) {
        nlist.push_back((*i)->className());
        (*i)->destroy();
    }
    _host_list.clear();

    vector<string>::iterator j = nlist.begin();
    for (; j != nlist.end(); j++)
        unloadClass(*j);

    //
    // DefaultPipeline ݻƤƤ Valve ƻȤߤ
    // Ѥ nlist  Valve Υ饹̾¸Ƥ. 
    //
    nlist.clear();

    vector<shared_ptr<Valve> > valve_list = getValve();
    vector<shared_ptr<Valve> >::iterator v = valve_list.begin();
    for (; v != valve_list.end(); v++) {
        removeValve(*v);
        nlist.push_back((*v)->config().attr("className"));
    }
    valve_list.clear();

    for (j = nlist.begin(); j != nlist.end(); j++)
        unloadClass(*j);

    unloadLogger();
}


void DefaultEngine::loadHosts(const vector<si::interface::Config>& c)
{
    vector<si::interface::Config>::const_iterator i = c.begin();
    for (; i != c.end(); i++)
        loadHost(*i);
}

void DefaultEngine::loadHost(const si::interface::Config& c)
{
    string name = c.attr("name");
    string class_name = c.attr("className");

    //
    // Ʊ name ° Host ¸ߤƤϥɤʤ.
    //
    if (getHost(name)) {
        getLogger()->log("DefaultEngine::loadHost - already loaded.",
                         Logger::ERROR_LOG);
        return;
    }

    if (class_name.empty())
        class_name = DEFAULT_HOST;

    try {
        shared_ptr<Host> p = loadClass<Host>(class_name);
        p->home(home());
        p->config(c);
        p->className(class_name);
        p->setParent(this);
        setHost(p);
    } catch (std::exception& e) {
        cerr << e.what() << endl;
    }
}

void DefaultEngine::loadValves(const vector<si::interface::Config>& clist)
{
    vector<si::interface::Config>::const_iterator i = clist.begin();
    for (; i != clist.end(); i++) {
        std::string class_name = i->attr("className");

        getLogger()->log("DefaultEngine::loadValve - name of " + class_name,
                        Logger::DEBUG_LOG);
        try {
            shared_ptr<Valve> p = loadClass<Valve>(class_name);
            if (p) {
                p->config(*i);
                addValve(p);
            }
        } catch (std::exception& e) {
            getLogger()->log("DefaultEngine::loadValve - "
                            "It failed in loading \"Valve\"", e);
        }
    }

    shared_ptr<Valve> p = loadClass<Valve>(DEFAULT_VALVE);
    if (p) {
        si::interface::Config c;
        c.attr("className", DEFAULT_VALVE);
        p->config(c);
        addValve(p);
    }
}
