/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <sys/types.h>
#include <sys/stat.h>

#include <iostream>
#include <string>
#include <vector>
#include <boost/lexical_cast.hpp>
#include <boost/regex.hpp>
#include <boost/algorithm/string/compare.hpp>
#include <boost/algorithm/string/predicate.hpp>
using namespace std;
using namespace boost;

#include <sl/java.hpp>
#include <sl/text/random.h>
#include <servlet/ServletRequestListener.h>

#include "DefaultContextValve.h"

#include <si/interface/Config.h>
#include <si/interface/Context.h>
#include <si/interface/Connection.h>
#include <si/interface/Request.h>
#include <si/interface/Response.h>
#include <si/interface/Logger.h>
#include <si/interface/ServletWrapper.h>
#include <si/core/HttpServletRequestImpl.h>
#include <si/core/HttpServletResponseImpl.h>
#include <si/core/FilterChainImpl.h>
#include <si/core/url_pattern.h>

using namespace si::interface;
using namespace si::core;
using namespace modules::core;

//
// Constructor/Destructor.
//

DefaultContextValve::DefaultContextValve()
    : _context(0), _security_existance_judged(false)
{ }


DefaultContextValve::~DefaultContextValve()
{ }


//
// Member functions.
//

void DefaultContextValve::config(const si::interface::Config& config)
{
    _config = config;
}


Config DefaultContextValve::config() const
{
    return _config;
}


void DefaultContextValve::invoke(Connection& connect, ValveContext& context)
{
    if (!_context)
        _context = dynamic_cast<Context*>(context.getBase());

    if (!_logger)
        _logger = _context->getLogger();

    if (_path.empty())
        _path = _context->home() + "/" + _context->docBase();

    service(connect, *_context, *_logger.get());
}


void DefaultContextValve::service(Connection& connect, Context& context, Logger& logger)
{
    // Request  servlet::ServletRequest إǡ򥳥ԡ.
    Request&  req = connect.request();
    Response& res = connect.response();

    HttpServletRequestImpl  request(connect, context);
    HttpServletResponseImpl    response(connect, context);

    /*
     * Request URI 饳ƥȥѥ뤷ͤƤ.
     *  servlet-path ӡ path-info ȤѤ.
     * Request::uri()  abs_path(ޤޤʤ '/' ϤޤURI)֤.
     */
    string request_uri = req.uri();
    string servlet_path;
    if (context.path() == "/")
        servlet_path = request_uri;
    else
        servlet_path = request_uri.substr(context.path().length());

    /*
     * ꥯȤ'/'ǽüƤ
     * welcome-file-list Ϣ뤷ե̾ơ
     * Ԥʤ¸ߤե򥿡åȥեȤƻѤ
     */
    if (servlet_path[servlet_path.length() - 1] == '/') {
        string file_path = _path + servlet_path;

        std::list<std::string> file_list = context.welcomeFileList();
        std::list<std::string>::iterator i = file_list.begin();
        for (; i != file_list.end(); i++) {
            string tmp = file_path + *i;
            struct stat sb;
            if (::stat(tmp.c_str(), &sb) != -1 && S_ISREG(sb.st_mode)) {
                servlet_path += *i;
                break;
            }
        }
    }

    /*
     * åȤ Servlet ݻ Wrapper .
     * Ǥ servlet_path  Request-URI(abs_path) 
     * ƥȥѥ servlet-path  path-info ޤʸ.
     */
    shared_ptr<ServletWrapper> wrapper = context.getWrapper(servlet_path);
    if (!wrapper) {
        response.sendError(404);
        return;
    }
    if (!wrapper->instance()) {
        response.sendError(503);
        return;
    }

    /*
     * servlet_path  path info Ф.
     * web.xml ˤ Servlet Υޥåԥ󥰤ʣ꤬ǽʤΤǡ
     * νǼ Wrapper ޥåԥ󥰾ơ
     * Υޥåԥ󥰾󤫤ޥåѥȽꤹ.
     * ޥåԥʸ󤫤ޥåѥ path-info ˤʤϤ.
     */
    string path_info;

    vector<string> mapping = wrapper->getMapping();
    vector<string>::iterator i = mapping.begin();
    for (; i != mapping.end(); i++) {
        string map_string = *i;
        // ̤Ǥϥޥåʤ
        if (map_string.empty()) // || map_string[0] == '*')
            continue;

        if (map_string[0] == '*') {
            path_info = servlet_path;
            servlet_path.clear();
            break;
        }

        // "*"  "/*" ξϺʸ󤫤 path_info 
        if (map_string[map_string.length() - 1] == '*') {
            map_string.erase(map_string.length() - 1);
            if (map_string[map_string.length() - 1] == '/')
                map_string.erase(map_string.length() - 1);
        }

        if ((servlet_path.find(map_string)) == 0) {
            path_info = servlet_path.substr(map_string.length());
            break;
        }
    }

    /*
     * ꥯȤ'/'ǽüƤ
     * welcome-file-list Ϣ뤷ե̾ơ
     * Ԥʤ¸ߤե򥿡åȥեȤƻѤ
     */
    if (servlet_path[0] == '/' && path_info[0] == '/') {
        string file_path = _path + servlet_path;

        std::list<std::string> file_list = context.welcomeFileList();
        std::list<std::string>::iterator i = file_list.begin();
        for (; i != file_list.end(); i++) {
            string tmp = file_path + *i;
            struct stat sb;
            if (::stat(tmp.c_str(), &sb) != -1 && S_ISREG(sb.st_mode)) {
                path_info += *i;
                break;
            }
        }
    }

    request.setContextPath(context.path());
    request.setServletPath(
        servlet_path.substr(0, servlet_path.length() - path_info.length()));
    request.setPathInfo(path_info);

    //
    // ServletRequestListener  requestInitialized дؿθƤӽФ
    // Ԥʤ.
    //
    vector<boost::shared_ptr<servlet::ServletRequestListener> > rl;

    servlet::ServletRequestEvent event(context.getServletContext(), request);

    vector<boost::shared_ptr<sl::java::util::EventListener> > el =
        context.getListeners();

    vector<boost::shared_ptr<sl::java::util::EventListener> >::iterator e =
        el.begin();
    for (; e != el.end(); e++) {
        boost::shared_ptr<servlet::ServletRequestListener> p1 =
            dynamic_pointer_cast<servlet::ServletRequestListener>(*e);
        if (p1) {
            p1->requestInitialized(event);
            rl.push_back(p1);
            continue;
        }

        boost::shared_ptr<servlet::ServletRequestAttributeListener> p2 =
        dynamic_pointer_cast<servlet::ServletRequestAttributeListener>(*e);
        if (p2) {
            // ServletRequest  AttributeListener ꤹ.
            request.addAttributeListener(p2.get());
            continue;
        }
    }

    try {
        //
        // ǥե륿ư뤿ˡ
        // ºݤ Filter 󥹥󥹤μ¹ԤϥԡǹԤʤ.
        //
        servlet::FilterChain& filter_chain = context.getFilterChain();
        FilterChainImpl chain_copy =
            dynamic_cast<FilterChainImpl&>(filter_chain);
        chain_copy.setWrapper(wrapper.get());
        chain_copy.doFilter(request, response);

    } catch(servlet::ServletException &e) {
        logger.log("Context :", e);
        response.sendError(500, e.what());

    } catch(std::exception &e) {
        logger.log("Context :", e);
        response.sendError(500, e.what());
    }

    if (!response.isCommitted())
        response.flushBuffer();

    vector<boost::shared_ptr<
        servlet::ServletRequestListener> >::iterator n = rl.begin();
    for (; n != rl.end(); n++)
        (*n)->requestDestroyed(event);
}
