/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <iostream>
#include <boost/algorithm/string/compare.hpp>
#include <boost/algorithm/string/predicate.hpp>
#include <boost/algorithm/string/replace.hpp>
#include <boost/algorithm/string/trim.hpp>
#include <boost/algorithm/string/classification.hpp>
#include <boost/algorithm/string/predicate.hpp>
#include <boost/bind.hpp>
#include <boost/filesystem/exception.hpp>
#include <boost/filesystem/operations.hpp>
#include <boost/lexical_cast.hpp>
using namespace std;
using namespace boost;
namespace fs = boost::filesystem;

#include <sl/java/util/EventListener.h>
#include <servlet/ServletContextListener.h>
#include <servlet/ServletContextAttributeListener.h>
#include <servlet/ServletRequestListener.h>
#include <servlet/ServletRequestAttributeListener.h>
#include <servlet/http/HttpSessionListener.h>
#include <servlet/http/HttpSessionAttributeListener.h>

#include "DefaultContext.h"

#include <si/shibainu.h>
#include <si/core/ErrorPage.h>
#include <si/core/FilterChainImpl.h>
#include <si/core/FilterConfigImpl.h>
#include <si/core/FilterMapping.h>
#include <si/core/ServletContextImpl.h>
#include <si/core/url_pattern.h>
#include <si/interface/Host.h>
#include <si/interface/Loader.h>
#include <si/interface/Logger.h>
#include <si/interface/Manager.h>
#include <si/interface/Realm.h>
#include <si/interface/ServletWrapper.h>
#include <si/interface/Config.h>
#include <si/webxml/ApplicationConfig.h>
using namespace si::interface;
using namespace si::core;
using namespace modules::core;


static const string WEB_INF_CLASSES = "/WEB-INF/classes";
static const string WEB_INF_LIB     = "/WEB-INF/lib";
static const string WEB_INF_WEB_XML = "/WEB-INF/web.xml";
static const string DEFAULT_WEB_XML = "conf/web.xml";
static const string DEFAULT_RELOAD  = "false";
static const int    DEFAULT_INTERVAL= 15;

static const string SERVLET_WEBAPPS_PATH    = "javax.servlet.webapps.path";
static const string SERVLET_CONTEXT_PATH    = "javax.servlet.context.path";
static const string SERVLET_CONTEXT_TEMPDIR = "javax.servlet.context.tempdir";
static const string WELCOME_FILE_LIST       = "--welcome-file-list";

static const string DEFAULT_VALVE   = "modules/core/DefaultContextValve";
static const string DEFAULT_MANAGER = "modules/core/DefaultManager";
static const string DEFAULT_WRAPPER = "modules/core/DefaultWrapper";
static const string DEFAULT_REALM   = "modules/realm/MemoryRealm";

//
// Constructor/Destructor.
//

DefaultContext::DefaultContext()
    : _reload_interval(0), _web_xml(0)
{
}


DefaultContext::~DefaultContext()
{
}


//
// Member functions.
//

void DefaultContext::path(const string& path)
{
    /*
     * 饤Ȥ׵˹Ԥʤѥޥå󥰤Ǥ
     * boost::regex_search ѤƤɽǤΥޥå󥰤Ȥ뤿ᡢ
     *  path ʸɽѴ boost::regex .
     */
    _path = path;
    _path_regex = boost::regex(path_to_regex(path));
}


string DefaultContext::path() const
{
    return _path;
}


void DefaultContext::docBase(const string& doc_base)
{
    _doc_base = doc_base;
}


string DefaultContext::docBase() const
{
    return _doc_base;
}


void DefaultContext::setWorkDir(const string&)
{

}


string DefaultContext::getWorkDir() const
{
    Host* p = dynamic_cast<Host*>(getParent());
    return p->getWorkDir() + "/" + docBase();
}


bool DefaultContext::match(const string& context_path) const
{
    return context_path == _path;
}

 
void DefaultContext::setManager(const shared_ptr<Manager>& manager)
{
    _manager = manager;
}


shared_ptr<Manager> DefaultContext::getManager() const
{
    return _manager;
}


void DefaultContext::setWrapper(shared_ptr<ServletWrapper> wrapper)
{
    _wrapper_list.push_back(wrapper);
}


shared_ptr<ServletWrapper> DefaultContext::getWrapper(const string& path) const
{
    wrapper_list_t::const_iterator i = _wrapper_list.begin();
    for (; i != _wrapper_list.end(); i++) {
        if ((*i)->match(path))
            return *i;
    }

    // ĤʤϥǥեȤ Servlet õ.

    for (i = _wrapper_list.begin(); i != _wrapper_list.end(); i++) {
        if ((*i)->match("/"))
            return *i;
    }
    return shared_ptr<ServletWrapper>();
}

shared_ptr<ServletWrapper> DefaultContext::getNamedWrapper(const std::string& name) const
{
    wrapper_list_t::const_iterator i = _wrapper_list.begin();
    for (; i != _wrapper_list.end(); i++) {
        if ((*i)->name() == name)
            return *i;
    }
    return shared_ptr<ServletWrapper>();
}

void DefaultContext::addFilter(boost::shared_ptr<servlet::Filter> /* filter */)
{

}

servlet::FilterChain& DefaultContext::getFilterChain()
{
    return *_filter_chain;
}


vector<shared_ptr<servlet::Filter> > DefaultContext::getFilter() const
{
    /*
     * An invalid value is returned.
     */
    return vector<shared_ptr<servlet::Filter> >();
}

std::vector<boost::shared_ptr<sl::java::util::EventListener> >
DefaultContext::getListeners() const
{
    std::vector<shared_ptr<sl::java::util::EventListener> > ret;
    shared_ptr<sl::java::util::EventListener> e;

    ctx_listeners_t::const_iterator cl = _ctx_listeners.begin();
    for (; cl != _ctx_listeners.end(); cl++) {
        e = dynamic_pointer_cast<sl::java::util::EventListener>(*cl);
        ret.push_back(e);
    }

    ctx_attr_listeners_t::const_iterator ca = _ctx_attr_listeners.begin();
    for (; ca != _ctx_attr_listeners.end(); ca++)
        ret.push_back(dynamic_pointer_cast<sl::java::util::EventListener>(*ca));

    req_listeners_t::const_iterator rl = _req_listeners.begin();
    for (; rl != _req_listeners.end(); rl++)
        ret.push_back(dynamic_pointer_cast<sl::java::util::EventListener>(*rl));

    req_attr_listeners_t::const_iterator ra = _req_attr_listeners.begin();
    for (; ra != _req_attr_listeners.end(); ra++)
        ret.push_back(dynamic_pointer_cast<sl::java::util::EventListener>(*ra));

    ses_listeners_t::const_iterator sl = _ses_listeners.begin();
    for (; sl != _ses_listeners.end(); sl++)
        ret.push_back(dynamic_pointer_cast<sl::java::util::EventListener>(*sl));

    ses_attr_listeners_t::const_iterator sa = _ses_attr_listeners.begin();
    for (; sa != _ses_attr_listeners.end(); sa++)
        ret.push_back(dynamic_pointer_cast<sl::java::util::EventListener>(*sa));

    return ret;
}

servlet::ServletContext& DefaultContext::getServletContext()
{
    return *_servlet_context;
}

std::vector<si::webxml::security_constraint>
DefaultContext::getSecurityConstraint() const
{
    return _web_xml->getSecurityConstraint();
}

si::webxml::login_config DefaultContext::getLoginConfig() const
{
    return _web_xml->getLoginConfig();
}

si::webxml::session_config DefaultContext::getSessionConfig() const
{
    return _web_xml->getSessionConfig();
}

time_t DefaultContext::invokeTime()
{
    return _invoke_time;
}


list<string> DefaultContext::welcomeFileList() const
{
    return _web_xml->getWelcomeFileList().list();
}


string DefaultContext::getErrorPage(int code) const
{
    return _web_xml->getErrorPage().find(code);
}


string DefaultContext::getErrorPage(const string& e) const
{
    return _web_xml->getErrorPage().find(e);
}


si::webxml::jsp_config DefaultContext::getJspConfig() const
{
    return _web_xml->getJspConfig();
}


void DefaultContext::init()
{
    if (!getLoader())
        setLoader(getParent()->getLoader());

    //
    // Logger ¸ߤ Logger ɤ.
    //
    if (config().exist("Logger")) {
        si::interface::Config c = config().get("Logger");
        c.attr("home", home());

        shared_ptr<Logger> logger;
        try {
            logger = loadLogger(c);
            logger->open();
        } catch (std::exception& e) {
            logger = shared_ptr<Logger>();
            unloadLogger();

            string msg("DefaultServer::init - \"Logger\" failed in opening.");
            msg = msg + "reason is '" + e.what() + "'.";
            throw runtime_error(msg);
        }
    } else {
        setLogger(getParent()->getLogger());
    }

    shared_ptr<Logger> logger = getLogger();

    logger->log("DefaultContext::init - Beginning of initialization.",
                Logger::INFO_LOG);

    //
    // Context ǻꤵƤ°ɤ߹.
    //
    setupAttribute();

    logger->log("DefaultContext::init - path       --- " + _path,
                Logger::INFO_LOG);
    logger->log("DefaultContext::init - docBase    --- " + _doc_base,
                Logger::INFO_LOG);
    logger->log("DefaultContext::init - reloadable --- " + _reloadable,
                Logger::INFO_LOG);

    //
    // WEBץꥱ web.xml ɤ߹.
    //
    string def_web_xml = home() + "/" + DEFAULT_WEB_XML;
    string app_web_xml = home() + "/" + _doc_base + WEB_INF_WEB_XML;

    if (!fs::exists(app_web_xml) || fs::is_directory(app_web_xml)) {
        string msg("DefaultContext::init \"" + app_web_xml + "\" doesn't exist.");
        logger->log(msg, Logger::ERROR_LOG);
        throw std::runtime_error(msg);
    }

    string a = "\"web.xml\" file is read. Passing is \"" + def_web_xml + "\".";
    logger->log(a, Logger::DEBUG_LOG);

    _web_xml = new si::webxml::ApplicationConfig(home(), def_web_xml);

    //
    // ǥեȤ Servlet ξ web.xml ɤ߹.
    //
    string d = "\"web.xml\" file is read. Passing is \"" + app_web_xml + "\".";
    logger->log(d, Logger::DEBUG_LOG);

    si::webxml::ApplicationConfig app(home() + "/" +
                                    _doc_base + WEB_INF_CLASSES, app_web_xml);
    _web_xml->merge(app);

    //
    // WEB-INF/lib ۲Υɥ֥Ȥɤ.
    //
    string libpath = home() + "/" + _doc_base + WEB_INF_LIB;
    _libloader.load(libpath);

    //
    // ServletContext 󥹥󥹤 
    //  FilterConfig  ServletWrapper ⻲Ȥ뤿ޤ.
    //
    createServletContext();

    // web.xml  listener Ǥ¸ߤ륯饹ɤ.
    loadAllListener();

    //
    // ե륿 ServletConfig ͳ ServletContext 
    // ǽǤ뤿ᡢServletContext .
    //
    // XXXXX δؿ Filter::init ޤǸƤǤ.
    //
    loadAllFilter();

    // Valve ˵Ҥ줿饹ɤ.
    loadValves(config().gets("Valve"));

    //
    // Manager 󥹥󥹤.
    // "Manager" ϥǥեȤ Manager 饹ɤ.
    //
    loadManager(config().get("Manager"));
    _manager->setParent(this);
    _manager->init();

    //
    // Realm ¸ߤ Realm ɤ.
    //
    if (config().exist("Realm")) {
        si::interface::Config c = config().get("Realm");
        c.attr("home", home());
        if (c.attr("className").empty())
            c.attr("className", DEFAULT_REALM);

        try {
            loadRealm(c);
            getRealm()->init();

        } catch (std::exception& e) {
            string msg("DefaultContext::loadRealm - "
                        "\"Reaml\" failed in loading.");
            getLogger()->log(msg, e);
        }
    } else {
        std::string msg("DefaultContext::init - "
                        "\"Realm\" element doesn't exist.");
        logger->log(msg, Logger::WARN_LOG);

        si::interface::Host* host = dynamic_cast<Host*>(getParent());
        if (host) {
            logger->log("DefaultContext::loadRealm - using parent Reaml",
                        Logger::WARN_LOG);
            setRealm(host->getRealm());
        }
    }

    loadAllWrapper();

    wrapper_list_t::iterator i = _wrapper_list.begin();
    for (; i != _wrapper_list.end(); i++) {
        try {
            (*i)->setParent(this);
            (*i)->setManager(_manager);
            (*i)->setLoader(getLoader());
            (*i)->setLogger(getLogger());
            (*i)->setServletContext(*_servlet_context);
            (*i)->home(home());
            (*i)->docBase(_doc_base);
            (*i)->init();
        } catch(std::runtime_error& e) {
            getLogger()->log("Context :", e);
        } catch(std::exception& e) {
            getLogger()->log("Context :", e);
        }
    }

    // ValveContext κԤȤϿƤ.
    setBase(this);

    //
    // ֥åȥѤΥåɵư.
    //
    if (_reloadable == "true") {
        logger->log("DefaultContext::init - \"reloadable\" is true.",
                    Logger::INFO_LOG);
        logger->log("DefaultContext::init - Start of reload confirmation "
                    "thread for Servlet.", Logger::DEBUG_LOG);

        _reload = true;
        _reload_thread =
            new boost::thread(boost::bind(&DefaultContext::reloadCheck, this));
    }

    _invoke_time = time(0);
}


void DefaultContext::destroy()
{
    std::string msg("DefaultContext::destroy - Beginning of termination. ");
    getLogger()->log(msg + "\"" + docBase() + "\".", Logger::INFO_LOG);

// XXXXX
#if 1
    {
        boost::mutex::scoped_lock lock(_service_mutex);
        _is_service = true;
    }
#endif

    // XXXXX
    if (_reloadable == "true") {
        {
            boost::mutex::scoped_lock lock(_reload_mutex);
            _reload = false;
            _reload_cond.notify_all();
        }
        _reload_thread->join();
        delete _reload_thread;
    }

    //
    // ServletWrapper Ф destroy дؿư
    // λ˥饹򥢥ɤ.
    //
    vector<string>    nlist;

    wrapper_list_t::iterator i = _wrapper_list.begin();
    for (; i != _wrapper_list.end(); i++) {
        nlist.push_back((*i)->className());
        (*i)->destroy();
    }
    _wrapper_list.clear();

    vector<string>::iterator j = nlist.begin();
    for (; j != nlist.end(); j++)
        unloadClass(*j);

    //
    // Realm 󥹥󥹤.
    //
    if (getRealm()) {
        getRealm()->destroy();
        unloadRealm();
    }

    //
    // Manager 󥹥󥹤.
    //
    if (_manager) {
        std::string class_name = _manager->className();
        _manager->destroy();
        _manager.reset();
        unloadClass(class_name);
    }

    //
    // Filter ڤ Listener ɤ.
    //
    unloadAllListener();
    unloadAllFilter();

    // XXXXX
    // ServletContextImpl 󥹥󥹤.
    //
//    _servlet_context->destroy();
    delete _servlet_context;

    //
    // Ĥä饹󥹥󥹤.
    //
    delete _web_xml;

    //
    // DefaultPipeline ݻƤƤ Valve ƻȤߤ
    // Ѥ nlist  Valve Υ饹̾¸Ƥ. 
    //
    nlist.clear();

    vector<shared_ptr<Valve> > valve_list = getValve();
    vector<shared_ptr<Valve> >::iterator v = valve_list.begin();
    for (; v != valve_list.end(); v++) {
        removeValve(*v);
        nlist.push_back((*v)->config().attr("className"));
    }
    valve_list.clear();

    for (j = nlist.begin(); j != nlist.end(); j++)
        unloadClass(*j);

    //
    // WEB-INF/lib ۲ΥɺѤߥɥ֥Ȥ򥢥ɤ.
    //
    _libloader.unload();

// XXXXX
#if 1
    {
        boost::mutex::scoped_lock lock(_service_mutex);
        _is_service = false;
    }
#endif
    unloadLogger();
}

void DefaultContext::loadAllFilter()
{
    FilterChainImpl* filter_chain = new FilterChainImpl();
    std::vector<si::webxml::filter> filters = _web_xml->getFilters();


    std::vector<si::webxml::filter>::iterator i = filters.begin();
    for (; i != filters.end(); i++) {
"XXXXX";
        std::string class_base_dir = _doc_base + WEB_INF_CLASSES;
        if (chdir(class_base_dir.c_str()) != 0)
            continue;

        std::string class_name = i->filter_class();
        boost::shared_ptr<servlet::Filter> instance;
        try {
            instance = loadClass<servlet::Filter>(class_name);
        } catch (std::runtime_error& e) {
            getLogger()->log("DefaultContext::loadAllFilter \"Filter\" loading failed.", e);
            continue;
        }
        if (!instance)
            continue;
"XXXXX";
        if (chdir(home().c_str()) != 0) {
            getLogger()->log("DefaultContext::loadAllFilter failed in chdir()");
            continue;
        }

        FilterConfigImpl *conf = new FilterConfigImpl();

        conf->setServletContext(_servlet_context);
        conf->setFilterName(i->filter_name());

        std::vector<std::string> p = i->init_param_names();
        std::vector<std::string>::iterator s = p.begin();
        for (; s != p.end(); s++)
            conf->addInitParameter(*s, i->find_init_param(*s));

        /* Filter::init() ¹ */
        instance->init(*conf);

        si::webxml::filter_mapping filter_map = _web_xml->getFilterMapping();

        FilterMapping mapping(i->filter_name());
        mapping.urlPattern(filter_map.url_pattern(i->filter_name()));
        mapping.dispatcher("REQUEST");

        filter_chain->addFilter(mapping, instance);

        _filter_list.push_back(instance.get());
        _filter_config_list.push_back(conf);
    }
    _filter_chain = filter_chain;
}

void DefaultContext::loadAllListener()
{
    // <listener> (listener-class).
    si::webxml::listener listeners = _web_xml->getListener();

    // <listener>  <listener-class> Υꥹ.
    std::list<std::string> class_list = listeners.list();
    std::list<std::string>::iterator i = class_list.begin();
    for (; i != class_list.end(); i++) {

"XXXXX";
        std::string class_base_dir = _doc_base + WEB_INF_CLASSES;
        if (chdir(class_base_dir.c_str()) != 0) {
            continue;
        }

        std::string class_name = *i;

        shared_ptr<sl::java::util::EventListener> l;
        try {
            l = loadClass<sl::java::util::EventListener>(class_name);
        } catch (std::runtime_error& e) {
            getLogger()->log("DefaultContext::loadAllListener - "
                             "\"Listener\" loading failed.", e);
        }

        if (!l)
            continue;
"XXXXX";
        if (chdir(home().c_str()) != 0) {
            getLogger()->log("DefaultContext::loadAllListener failed in chdir()");
            continue;
        }

        _listener_names.push_back(class_name);

        //
        // Listener ηѾ饹Ĵ̡¸Ƥ.
        //
        shared_ptr<servlet::http::HttpSessionAttributeListener> sb =
            dynamic_pointer_cast<servlet::http::HttpSessionAttributeListener>(l);
        if (sb) {
            _ses_attr_listeners.push_back(sb);
            continue;
        }

        shared_ptr<servlet::http::HttpSessionListener> sl =
                dynamic_pointer_cast<servlet::http::HttpSessionListener>(l);
        if (sl) {
            _ses_listeners.push_back(sl);
            continue;
        }

        shared_ptr<servlet::ServletContextAttributeListener> ca =
            dynamic_pointer_cast<servlet::ServletContextAttributeListener>(l);
        if (ca) {
            _ctx_attr_listeners.push_back(ca);
            continue;
        }

        shared_ptr<servlet::ServletContextListener> sc =
                dynamic_pointer_cast<servlet::ServletContextListener>(l);
        if (sc) {
            _ctx_listeners.push_back(sc);
            continue;
        }

        shared_ptr<servlet::ServletRequestAttributeListener> ra =
            dynamic_pointer_cast<servlet::ServletRequestAttributeListener>(l);
        if (ra) {
            _req_attr_listeners.push_back(ra);
            continue;
        }

        shared_ptr<servlet::ServletRequestListener> sr =
                dynamic_pointer_cast<servlet::ServletRequestListener>(l);
        if (sr) {
            _req_listeners.push_back(sr);
            continue;
        }

        getLogger()->log("DefaultContext:: instance of unknown Listener",
                         Logger::ERROR_LOG);
    }

    //
    // XXXXX
    // ServletContextListener Ф contextInitialized дؿ
    // ƤӽФԤʤ.
    // Ǥ Filter  Servlet εư˼¹Ԥ뤳Ȥˤ.
    // 
    servlet::ServletContextEvent e(*_servlet_context);
    ctx_listeners_t::iterator l = _ctx_listeners.begin();
    for (; l != _ctx_listeners.end(); l++)
        (*l)->contextInitialized(e);

    //
    // ʹ ServletContext °ѹФꥹʡؤΤ
    // ServletContextImpl ǤΤ c->setListener Ƥ
    // ContextAttributeListener  ServletContextImpl Ϥ.
    //
    ctx_attr_listeners_t::iterator m = _ctx_attr_listeners.begin();
    for (; m != _ctx_attr_listeners.end(); m++)
        _servlet_context->addAttributeListener(m->get());
}

void DefaultContext::unloadAllFilter()
{
    getLogger()->log("DefaultContext::unloadAllFilter Enter",
                     Logger::DEBUG_LOG);

    //
    // Filter::destroy дؿ¹Ԥ.
    //
    vector<servlet::Filter*>::iterator f = _filter_list.begin();
    for (; f != _filter_list.end(); f++)
        (*f)->destroy();

    //
    // FilterChainImpl ݻ Filter 󥹥󥹤. 
    //
    while (_filter_chain->size())
        _filter_chain->popFilter();

    delete _filter_chain;

    //
    // FilterConfigImpl .
    //
    vector<servlet::FilterConfig*>::iterator c = _filter_config_list.begin();
    for (; c != _filter_config_list.end(); c++)
        delete *c;

    //
    // ɤ Filter 󥹥󥹤̾ΥꥹȤ˥饹򥢥ɤ.
    //
#if 1
    std::vector<si::webxml::filter> filters = _web_xml->getFilters();
    std::vector<si::webxml::filter>::iterator i = filters.begin();
    for (; i != filters.end(); i++) {
        unloadClass(_doc_base + WEB_INF_CLASSES + "/" + i->filter_class());
    }
#else
    vector<string>::iterator i = _filter_names.begin();
    for (; i != _filter_names.end(); i++)
        unloadClass(*i);
#endif
}

void DefaultContext::unloadAllListener()
{
    getLogger()->log("DefaultContext::unloadAllListener Enter",
                     Logger::DEBUG_LOG);

    //
    // ServletContextListener Ф contextDestroyed дؿ
    // ƤӽФԤʤ.
    //
    servlet::ServletContextEvent e(*_servlet_context);
    ctx_listeners_t::iterator l = _ctx_listeners.begin();
    for (; l != _ctx_listeners.end(); l++)
        (*l)->contextDestroyed(e);

    //
    // Listener ݻƥʤ򥯥ꥢơ
    // ٤ƤλȤ򳰤.
    //
    _ctx_listeners.clear();
    _ctx_attr_listeners.clear();
    _req_listeners.clear();
    _req_attr_listeners.clear();
    _ses_listeners.clear();
    _ses_attr_listeners.clear();

    //
    // ɤ Listener 󥹥󥹤̾ΥꥹȤ˥饹
    // ɤ.
    //
    vector<string>::iterator i = _listener_names.begin();
    for (; i != _listener_names.end(); i++)
        unloadClass(*i);
}

void DefaultContext::loadAllWrapper()
{
    // <servlet> (servlet-name  servlet-class  map ).
    std::vector<si::webxml::servlet> servlets = _web_xml->getServlet();
    si::webxml::servlet_mapping mapping = _web_xml->getServletMapping();

    // <servlet>  <servlet-name> Υꥹ.
    std::vector<si::webxml::servlet>::iterator i = servlets.begin();
    for (; i != servlets.end(); i++) {
        si::webxml::servlet target = *i;
        shared_ptr<ServletWrapper> p = loadWrapper("");
        if (p) {
            p->name(target.servlet_name());
            p->servletClassName(target.servlet_class());
            p->servletClassPath(target.servlet_path());
            p->addMapping(mapping.url_pattern(target.servlet_name()));

            std::vector<std::string> params = target.init_param_names();
            std::vector<std::string>::iterator j = params.begin();
            for (; j != params.end(); j++)
                p->addInitParam(*j, target.find_init_param(*j));
        
            _wrapper_list.push_back(p);
        }
    }
}


shared_ptr<ServletWrapper> DefaultContext::loadWrapper(const string &name)
{
    string class_name = name;
    if (name.empty())
        class_name = DEFAULT_WRAPPER;

    shared_ptr<ServletWrapper> p;
    try {
        p = loadClass<ServletWrapper>(class_name);
        p->className(class_name);

    } catch (sl::java::lang::ClassNotFoundException& e) {
        getLogger()->log("DefaultContext::loadWrapper - "
                         "\"ServletWrapper\" loading failed.", e);
    }
    return p;
}

void DefaultContext::unloadWrapper(shared_ptr<ServletWrapper>& p)
{
    string class_name = p->className();
    p = shared_ptr<ServletWrapper>();
    try {
        unloadClass(class_name);
    } catch (sl::java::lang::ClassNotFoundException& e) {
        getLogger()->log("\tTarget class is not found.", e);
    }
}

void DefaultContext::loadManager(const si::interface::Config& c)
{
    string class_name = c.attr("className");

    if (class_name.empty())
        class_name = DEFAULT_MANAGER;

    shared_ptr<Manager> p;
    try {
        p = loadClass<Manager>(class_name);
        p->className(class_name);
        p->config(c);
        p->setParent(this);
        setManager(p);

    } catch(sl::java::lang::ClassNotFoundException &e) {
        getLogger()->log("DefaultContext::loadManager - "
                         "\"Manager\" loading failed.", e);
    }

    //
    // ʹߥåФꥹʡؤΤ Manager ǤΤǡ
    // SessionAttributeListener  Manager Ϥ.
    //
    ses_listeners_t::iterator s = _ses_listeners.begin();
    for (; s != _ses_listeners.end(); s++)
        p->addSessionListener(s->get());

    //
    // ʹߥå°ѹФꥹʡؤΤ Manager Ǥ.
    //
    ses_attr_listeners_t::iterator a = _ses_attr_listeners.begin();
    for (; a != _ses_attr_listeners.end(); a++)
        p->addSessionAttributeListener(a->get());
}


void DefaultContext::createServletContext()
{
    using namespace sl::java::lang;

    ServletContextImpl *context = new ServletContextImpl(this, home(), _doc_base);

    /*
     * DefaultServlet  CGIServlet μʤΤ
     * WEB ץꥱΥ롼ȥǥ쥯ȥ°Ȥ¸Ƥ
     */
    context->setAttribute(SERVLET_CONTEXT_TEMPDIR, this->getWorkDir());
    context->setAttribute(SERVLET_CONTEXT_PATH, home() + "/" + this->docBase());

    si::interface::Host* host = dynamic_cast<Host*>(getParent());
    if (host)
        context->setAttribute(SERVLET_WEBAPPS_PATH, home() + "/" + host->appBase());

    context->setServerInfo(si::info());
    context->setMajorVersion(si::api_major_version());
    context->setMinorVersion(si::api_minor_version());

    std::vector<std::string> names;
    std::vector<std::string>::iterator i;

    /* web.xml  context-param  Context ɤ߹ */
    si::webxml::context_param params = _web_xml->getContextParam();
    names = params.names();
    for (i = names.begin(); i != names.end(); i++)
        context->addInitParameter(*i, params.find(*i));

    /* web.xml  mime-mapping  Context ɤ߹ */
    si::webxml::mime_mapping mime = _web_xml->getMimeMapping();
    names = mime.names();
    for (i = names.begin(); i != names.end(); i++)
        context->addMimeType(*i, mime.mime_type(*i));

    _servlet_context = context;
}


void DefaultContext::reloadCheck()
{
    time_t i = _reload_interval;

    getLogger()->log("Context :reloadCheck " + lexical_cast<string>(_path) +
                     ". interval time is " + lexical_cast<string>(i) + " sec.",
                     Logger::INFO_LOG);

    for (;;) {
        {
            mutex::scoped_lock lock(_reload_mutex);
            if (!_reload)
                break;

            xtime xt;
            xtime_get(&xt, boost::TIME_UTC);
            xt.sec += i;

            if (_reload_cond.timed_wait(lock, xt))
                continue;
        }
        {
            boost::mutex::scoped_lock lock(_service_mutex);
            if (_is_service)
                continue;
        }

        wrapper_list_t::iterator i = _wrapper_list.begin();
        for (; i != _wrapper_list.end(); i++) {
            string file = home() + "/" +
                         (*i)->servletClassPath() + "/" + (*i)->servletClassName();
            replace_all(file, "//", "/");

#ifdef DEBUG
    cout << "reloadCheck:" << file << endl;
#endif
            time_t last;
            try {
                last = fs::last_write_time(file + sl::config::dso_suffix);
             } catch(fs::filesystem_error &e) {
#ifdef DEBUG
                cout << "reloadCheck :" << e.what() << endl;
#endif
                continue;
            }

#ifdef DEBUG
    cout << "reloadCheck :Invoke time is " << (*i)->invokeTime() << ", ";
    cout << "last access time is " << last << endl;
#endif
            if ((*i)->invokeTime() < last) {
                getLogger()->log("Context :reloaded servlet `" +
                                 (*i)->servletClassName() + "`", Logger::DEBUG_LOG);

                boost::mutex::scoped_lock lock(_service_mutex);

                (*i)->unloadInstance();
                (*i)->loadInstance();
            }
        }
    }
}

void DefaultContext::loadValves(const vector<si::interface::Config>& c)
{
    vector<si::interface::Config>::const_iterator i = c.begin();
    for (; i != c.end(); i++) {
        std::string class_name = i->attr("className");

        getLogger()->log("DefaultContext::loadValve - name of " + class_name,
                        Logger::DEBUG_LOG);
        try {
            shared_ptr<Valve> p = loadClass<Valve>(class_name);
            if (p) {
                p->config(*i);
                addValve(p);
            }
        } catch (std::exception& e) {
            getLogger()->log("DefaultContext::loadValve - "
                             "\"Valve\" loading failed.", e);
            // throw;
        }
    }

    try {
        shared_ptr<Valve> p = loadClass<Valve>(DEFAULT_VALVE);
        if (p) {
            si::interface::Config c;
            c.attr("className", DEFAULT_VALVE);
            p->config(c);
            addValve(p);
        }
    } catch (std::exception& e) {
        getLogger()->log("DefaultContext::loadValve - "
                         "\"Valve\" loading failed.", e);
        throw;
    }
}


void DefaultContext::setupAttribute()
{
    shared_ptr<Logger> logger = getLogger();

    //
    // Context Ǥɬפ°Ĵ٤.
    //
    if (_path.empty()) {
        _path = config().attr("path");
        if (_path.empty()) {
            string msg("DefaultContext::init: - \"path\" doesn't exist "
                        "in the attribute value.");
            logger->log(msg, Logger::ERROR_LOG);
            throw std::runtime_error(msg);
        }
    }

    if (_doc_base.empty()) {
        _doc_base = config().attr("docBase");
        if (_doc_base.empty()) {
            string msg("DefaultContext::init: - \"docBase\" doesn't exist "
                        "in the attribute value.");
            logger->log(msg, Logger::ERROR_LOG);
            throw std::runtime_error(msg);
        }
    }

    //
    // docBase Ǥ `/' ǳϤURLϡƥȤΥ롼ȤǤϤʤ
    // ФΥ롼ȤˤХѥȤƲᤵ.
    // docBase ° `/' ǳϤƤϤΤޤ޻Ѥ
    // `/' ǳϤƤʤ Host  appBase °ղä.
    //
    if (_doc_base[0] == '/') {
        home("");
    } else {
        Host* p = dynamic_cast<Host*>(getParent());
        if (!p) {
            string msg("DefaultContext::init - Parent types are not \"Host.\"");
            logger->log(msg, Logger::ERROR_LOG);
            throw std::runtime_error(msg);
        }
        _doc_base = p->appBase() + "/" + _doc_base;
    }

    //
    // ¾°ФƤ̤ǤХǥեȤͤꤹ.
    //
    if (_reloadable.empty()) {
        string reloadable = config().attr("reloadable");
        if (reloadable == "true" || reloadable == "false")
            _reloadable = reloadable;
        else
            _reloadable = DEFAULT_RELOAD;
    }

    if (_reloadable == "true") {
        try {
            _reload_interval =lexical_cast<int>(config().attr("checkInterval"));
        } catch (bad_lexical_cast) {
            _reload_interval = DEFAULT_INTERVAL;
        }
    }
}
