/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef SHIBAINU_DEFAULT_COMPONENT_H
#define SHIBAINU_DEFAULT_COMPONENT_H

#include <iostream>
#include <string>
#include <boost/shared_ptr.hpp>
#include <sl/java/lang/ClassLoader.h>

#include <si/interface/Config.h>
#include <si/interface/Component.h>
#include <si/interface/Loader.h>

#include "DefaultPipeline.h"
using namespace si::interface;


// namespace modules {
// namespace core {

class DefaultComponent
    : virtual public Component, virtual public DefaultPipeline {
public :
    DefaultComponent() { }

    virtual ~DefaultComponent() { }

    virtual bool operator!()
    { return !_available; }

    virtual void config(const Config &config)
    { _config = config; }

    virtual Config config() const
    { return _config; }

    virtual void name(const std::string &name)
    { _name = name; }

    virtual std::string name() const
    { return _name; }

    virtual void className(const std::string& name)
    { _class_name = name; }

    virtual std::string className() const
    { return _class_name; }

    void home(const std::string& home)
    { _home = home; }

    std::string home() const
    { return _home; }

    virtual void setParent(Component* parent)
    { _parent = parent; }

    virtual Component* getParent() const
    { return _parent; }

    virtual void setLoader(boost::shared_ptr<Loader> loader)
    { _loader = loader; }

    virtual boost::shared_ptr<Loader> getLoader() const
    { return _loader; }

    virtual void setLogger(boost::shared_ptr<Logger> logger)
    { _logger = logger; }

    virtual boost::shared_ptr<Logger> getLogger() const
    { return _logger; }

    virtual void setRealm(boost::shared_ptr<Realm> realm)
    { _realm = realm; }

    virtual boost::shared_ptr<Realm> getRealm() const
    { return _realm; }

    virtual void init() { }

    virtual void destroy() { }

    virtual void service(Connection*) { }

    template <class Type>
    boost::shared_ptr<Type> loadClass(const std::string& class_path)
    {
        boost::shared_ptr<Type> p;

        if (class_path.empty())
            throw sl::java::lang::ClassNotFoundException("path is empty");

        try {
            sl::java::lang::Class class_loader = getLoader()->loadClass(class_path);

            p = class.template newInstance<Type>(class_loader);

        } catch (sl::java::lang::ClassNotFoundException& e) {
            throw;
        }
        return p;
    }

    void unloadClass(const std::string& class_file)
    {
        try {
            getLoader()->unloadClass(class_file);
        } catch (sl::java::lang::ClassNotFoundException& e) {
            std::cerr << e.what() << std::endl;
        }
    }

protected :
    volatile bool    _available;

private :
    std::string _name;
    std::string _class_name;
    std::string _home;
    Config _config;
    Component* _parent;
    boost::shared_ptr<Loader> _loader;
    boost::shared_ptr<Logger> _logger;
    boost::shared_ptr<Realm> _realm;
};

// } // namespace core
// } // namespace modules::core

#endif // SHIBAINU_DEFAULT_COMPONENT_H

