/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef SHIBAINU_DEFAULT_DABE_H
#define SHIBAINU_DEFAULT_DABE_H

#include <iostream>
#include <string>
#include <boost/shared_ptr.hpp>
#include <sl/java/lang/ClassLoader.h>

#include <si/interface/Config.h>
#include <si/interface/Base.h>
#include <si/interface/Loader.h>
#include <si/interface/Logger.h>
#include <si/interface/Realm.h>

#include "DefaultPipeline.h"

namespace modules {
namespace core {

class DefaultBase
        : virtual public si::interface::Base, virtual public DefaultPipeline {
public :
    DefaultBase() { }

    virtual ~DefaultBase() { }

    virtual bool operator!() const
    {
        return !_available;
    }

    virtual void init() { }

    virtual void destroy() { }

    virtual void service(si::interface::Connection*) { }


    // --- setter and getter to parameter ---

    virtual void config(const si::interface::Config& c)
    {
        _config = c;
    }

    virtual si::interface::Config config() const
    {
        return _config;
    }

    virtual void name(const std::string &name)
    {
        _name = name;
    }

    virtual std::string name() const
    {
        return _name;
    }

    virtual void className(const std::string& name)
    {
        _class_name = name;
    }

    virtual std::string className() const
    {
        return _class_name;
    }

    void home(const std::string& home)
    {
        _home = home;
    }

    std::string home() const
    {
        return _home;
    }


    // --- setter and getter to class ---

    virtual void setParent(Base* parent)
    {
        _parent = parent;
    }

    virtual Base* getParent() const
    {
        return _parent;
    }

    virtual void setLoader(const boost::shared_ptr<si::interface::Loader>& loader)
    {
        _loader = loader;
    }

    virtual boost::shared_ptr<si::interface::Loader> getLoader() const
    {
        return _loader;
    }

    virtual void setLogger(const boost::shared_ptr<si::interface::Logger>& logger)
    {
        _logger = logger;
    }

    virtual boost::shared_ptr<si::interface::Logger> getLogger() const
    {
        return _logger;
    }

    void setRealm(const boost::shared_ptr<si::interface::Realm>& realm)
    {
        _realm = realm;
    }

    boost::shared_ptr<si::interface::Realm> getRealm() const
    {
        return _realm;
    }

    // --- loading and unloading ---

    boost::shared_ptr<si::interface::Logger> loadLogger(const si::interface::Config& c)
    {
        _logger = load<boost::shared_ptr<si::interface::Logger> >(_logger, c);
        return _logger;
    }

    void unloadLogger()
    {
        unload<boost::shared_ptr<si::interface::Logger> >(_logger);
    }

    boost::shared_ptr<si::interface::Realm> loadRealm(const si::interface::Config& c)
    {
        _realm = load<boost::shared_ptr<si::interface::Realm> >(_realm, c);
        return _realm;
    }

    void unloadRealm()
    {
        unload<boost::shared_ptr<si::interface::Realm> >(_realm);
    }

    template <class T>
    T load(T& instance, const si::interface::Config& c)
    {
        if (instance)
            unload<T>(instance);

        instance = loadClass<typename T::element_type>(c.attr("className"));
        instance->config(c);

        return instance;
    }

    template <class T>
    void unload(T& instance, const si::interface::Config& c)
    {
        if (!instance || !instance.unique())
            return;

        std::string class_name = c.attr("className");
        instance = boost::shared_ptr<typename T::element_type>();
        unloadClass(class_name);
    }

    template <class T>
    void unload(T& instance)
    {
        if (!instance || !instance.unique())
            return;
        unload<T>(instance, instance->config());
    }

    template <class T>
    boost::shared_ptr<T> loadClass(const std::string& target)
    {
        if (!getLoader()) {
            std::string msg("DefaultBase::loadClass "
                            "The instance of \"Loader\" doesn't exist.");
            throw std::runtime_error(msg);
        }

        if (target.empty())
            throw std::runtime_error("DefaultBase::loadClass Target name is empty");

        boost::shared_ptr<T> p;
        try {
            sl::java::lang::Class class_loader = getLoader()->loadClass(target);
            p = class_loader.template newInstance<T>();
            if (getLogger())
                getLogger()->log("DefaultBase::loadClass " +
                                 target + " was loaded.",
                                 si::interface::Logger::DEBUG_LOG);
        } catch (sl::java::lang::ClassNotFoundException& e) {
            std::string msg("DefaultBase::loadClass ");
            throw std::runtime_error(msg + e.what());
        }
        return p;
    }

    void unloadClass(const std::string& target)
    {
        if (!getLoader()) {
            std::string msg("DefaultBase::loadClass "
                            "The instance of \"Loader\" doesn't exist.");
            throw std::runtime_error(msg);
        }

        try {
            getLoader()->unloadClass(target);
            if (getLogger())
                getLogger()->log("DefaultBase::unloadClass " +
                                 target + " was unloaded.",
                                 si::interface::Logger::DEBUG_LOG);
        } catch (sl::java::lang::ClassNotFoundException& e) {
            std::string msg("DefaultComponent::unloadClass ");
            throw std::runtime_error(msg + e.what());
        }
    }

protected :
    volatile bool _available;

private :
    std::string _name;
    std::string _class_name;
    std::string _home;
    si::interface::Config _config;
    si::interface::Base* _parent;
    boost::shared_ptr<si::interface::Loader> _loader;
    boost::shared_ptr<si::interface::Logger> _logger;
    boost::shared_ptr<si::interface::Realm> _realm;
};

} // namespace core
} // namespace modules

#endif // SHIBAINU_DEFAULT_DABE_H

