/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: SSLRequest.cpp,v 1.16 2006/10/29 03:18:49 cvsuser Exp $
 */

#include <boost/algorithm/string/case_conv.hpp>
#include <boost/algorithm/string/split.hpp>
#include <boost/algorithm/string/trim.hpp>
#include <boost/algorithm/string/predicate.hpp>
#include <boost/lexical_cast.hpp>
using namespace std;
using namespace boost;

#include "SSLRequest.h"


//
// Constructor/Destructor.
//

SSLRequest::SSLRequest()
{ }


SSLRequest::~SSLRequest()
{ }


//
// Member functions.
//

string SSLRequest::method() const
{
	return _request->method();
}


string SSLRequest::uri() const
{
	return _request->uri();
}


string SSLRequest::protocol() const
{
	return _request->version();
}


string SSLRequest::header(const string &name) const
{
	return _request->msg_header(name);
}


void SSLRequest::cleanup()
{
	_request->cleanup();
}


istream &SSLRequest::copy_from(istream &is)
{
	string buffer(is.peek(), char_traits<char>::eof());
	_request->parse(buffer);

	return is;
}


ostream &SSLRequest::copy_to(ostream &os)
{
	os << _request->text();

	return os;
}


void SSLRequest::parent(Connection *parent)
{
	_parent = parent;
}


void SSLRequest::set(sl::net::http::request& req)
{
	_request = &req;
}


void SSLRequest::content(const string &s)
{
	_request->msg_body(s);
}

string SSLRequest::text()
{
	return _request->text();
}

std::string SSLRequest::getParameter(const std::string &s)
{
	sl::net::http::ordermap_t params;
	if (sl::net::http::has_parameter(*_request))
		params = _request->parameter();
	else
		params = _request->query();

	sl::net::http::ordermap_t::const_iterator i = params.begin();
	for (; i != params.end(); i++) {
		if (s == i->first)
			return sl::net::http::urldecode(i->second);
	}
	return "";
}

std::vector<std::string> SSLRequest::getParameterNames()
{
	std::vector<std::string> result;

	sl::net::http::ordermap_t params;
	if (sl::net::http::has_parameter(*_request))
		params = _request->parameter();
	else
		params = _request->query();

	sl::net::http::ordermap_t::const_iterator i = params.begin();
	for (; i != params.end(); i++)
		result.push_back(sl::net::http::urldecode(i->first));
	return result;
}

std::vector<std::string> SSLRequest::getParameterValues(const std::string& s)
{
	std::vector<std::string> result;

	sl::net::http::ordermap_t params;
	if (sl::net::http::has_parameter(*_request))
		params = _request->parameter();
	else
		params = _request->query();

	sl::net::http::ordermap_t::const_iterator i = params.begin();
	for (; i != params.end(); i++) {
		if (i->first == s)
			result.push_back(sl::net::http::urldecode(i->second));
	}
	return result;
}

std::multimap<std::string, std::string> SSLRequest::getParameterMap()
{
	std::multimap<std::string, std::string> result;

	sl::net::http::ordermap_t params;
	if (sl::net::http::has_parameter(*_request))
		params = _request->parameter();
	else
		params = _request->query();

	sl::net::http::ordermap_t::const_iterator i = params.begin();
	for (; i != params.end(); i++)
		result.insert(std::make_pair(i->first, i->second));
	return result;
}

std::string SSLRequest::getCharacterEncoding()
{
	return _request->msg_header("Character-Encoding");
}

int SSLRequest::getContentLength()
{
	return sl::net::http::content_length(*_request);
}

std::string SSLRequest::getContentType()
{
	return _request->msg_header("Content-Type");
}

std::string SSLRequest::getProtocol()
{
	return _request->version();
}

std::string SSLRequest::getScheme()
{
	return "https";
}

bool SSLRequest::isSecure()
{
	return true;
}

std::string SSLRequest::getServerName()
{
	return _parent->serverHost();
}

int SSLRequest::getServerPort()
{
	return _parent->serverPort();
}

std::string SSLRequest::getLocalAddr()
{
	return _parent->localAddr();
}

std::string SSLRequest::getLocalName()
{
	return _parent->localHost();
}

int SSLRequest::getLocalPort()
{
	return _parent->localPort();
}

std::string SSLRequest::getRemoteAddr()
{
	return _parent->remoteAddr();
}

std::string SSLRequest::getRemoteHost()
{
	return _parent->remoteHost();
}

int SSLRequest::getRemotePort()
{
	return _parent->remotePort();
}

std::string SSLRequest::getAuthType()
{
	return "";
}

std::vector<servlet::http::Cookie> SSLRequest::getCookies()
{
	vector<servlet::http::Cookie> result;

	sl::net::http::multimap_t headers = _request->msg_headers();
	sl::net::http::multimap_t::const_iterator i = headers.begin();
	for (; i != headers.end(); i++) {
		if (equals("cookie", i->first, is_iequal())) {
			vector<string> cookies;
			algorithm::split(cookies, i->second, is_any_of(";"));

			for (vector<string>::iterator j = cookies.begin();
				 j != cookies.end(); j++)
			{
				vector<string> c;
				algorithm::split(c, *j, is_any_of("="));
				if (c.size() == 2) {
					result.push_back(servlet::http::Cookie(
										trim_copy(c[0]), trim_copy(c[1])));
				}
			}
		}
	}
	return result;
}

std::string SSLRequest::getHeader(const std::string &s)
{
	return _request->msg_header(s);
}

std::vector<std::string> SSLRequest::getHeaders(const std::string &s)
{
	std::vector<std::string> v;

	sl::net::http::multimap_t headers = _request->msg_headers();
	sl::net::http::multimap_t::const_iterator i = headers.begin();
	for (; i != headers.end(); i++) {
		if (equals(s, i->first, is_iequal()))
			v.push_back(i->second);
	}
	return v;
}

std::vector<std::string> SSLRequest::getHeaderNames()
{
	std::vector<std::string> v;

	sl::net::http::multimap_t headers = _request->msg_headers();
	sl::net::http::multimap_t::const_iterator i = headers.begin();
	for (; i != headers.end(); i++)
		v.push_back(i->first);
	return v;
}

std::string SSLRequest::getMethod()
{
	return _request->method();
}

std::string SSLRequest::getRequestURL()
{
	std::string host = _parent->targetHost();
	if (host.empty())
		return "";
	return "https://" + _parent->targetHost() + _request->uri();
}

std::string SSLRequest::getRequestURI()
{
	return _request->uri();
}

std::string SSLRequest::getQueryString()
{
	return _request->query_string();
}

std::string SSLRequest::getContent()
{
	return _request->msg_body();
}

std::istream& SSLRequest::getInputStream()
{
	if (_request->msg_header("Content-Type") ==
										"application/x-www-form-urlencoded") {

		_stream = new std::istream(0);
		_sstream.str(_request->msg_body());
		_stream->rdbuf(_sstream.rdbuf());

		return *_stream;
	} else
		return _request->stream();
}

