/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: SSLConnector.cpp,v 1.30 2008/03/01 03:37:10 cvsuser Exp $
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <unistd.h>

#include <cctype>
#include <cerrno>
#include <csignal>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <string>
#include <boost/bind.hpp>
#include <boost/thread.hpp>
#include <boost/lexical_cast.hpp>
using namespace std;
using namespace boost;

#include <openssl/err.h>
#include <openssl/ssl.h>

#include "interface/Logger.h"
using namespace si::interface;

#include "SSLConnector.h"
#include "SSLConnection.h"


#include <sl/sys/class_loader.h>
REGIST_LOADABLE_CLASS(SSLConnector)


//
// Constructor/Destructor
//

SSLConnector::SSLConnector()
	: _max_thread(100),
	  _min_spare_thread(10),
	  _max_spare_thread(50),
	  _port(0),
	  _accept_count(10),
	  _timeout(20000),
	  _current_accept(0)
{
#ifdef DEBUG
	cerr << "SSLConnector::SSLConnector" << endl;
#endif
}


SSLConnector::~SSLConnector()
{
#ifdef DEBUG
	cerr << "SSLConnector::~SSLConnector" << endl;
#endif
}


//
// Member functions.
//

void SSLConnector::setPort(int port)
{
	_port = port;
}

int SSLConnector::getPort()
{
	return _port;
}

void SSLConnector::setAcceptCount(int count)
{
	_accept_count = count;
}

int SSLConnector::getAcceptCount() const
{
	return _accept_count;
}

void SSLConnector::setRedirectPort(int port)
{
	_redirect_port = port;
}

int SSLConnector::getRedirectPort() const
{
	return _redirect_port;
}

void SSLConnector::setTimeout(int time)
{
	_timeout = time;
}

int SSLConnector::getTimeout() const
{
	return _timeout;
}

void SSLConnector::setDisableUploadTimeout(bool timeout)
{
	_disable_to = timeout;
}

bool SSLConnector::getDisableUploadTimeout() const
{
	return _disable_to;
}

void SSLConnector::setExecuteThread(Connection* p)
{
	recursive_mutex::scoped_lock lock(_execute_mutex);
	_execute_thread.push_back(p);
}

void SSLConnector::removeExecuteThread(Connection* p)
{
	recursive_mutex::scoped_lock lock(_execute_mutex);
	_execute_thread.erase(std::remove(_execute_thread.begin(),
									   _execute_thread.end(),
									   p),
						   _execute_thread.end());
}

void SSLConnector::setSpareThread(Connection* p)
{
	recursive_mutex::scoped_lock lock(_spare_mutex);
	_spare_thread.push_back(p);
}

void SSLConnector::removeSpareThread(Connection* p)
{
	recursive_mutex::scoped_lock lock(_spare_mutex);
	_spare_thread.erase(std::remove(_spare_thread.begin(),
									_spare_thread.end(),
									p),
						_spare_thread.end());
}
void SSLConnector::init()
{
	if (!getLogger())
		setLogger(getParent()->getLogger());

	getLogger()->log("SSLConnector::init - Beginning of initialization.",
					Logger::INFO_LOG);

	try {
		_port = lexical_cast<int>(config().attr("port"));
	} catch(bad_lexical_cast) {
		throw std::invalid_argument("invalid port number.");
	}

	try {
		_redirect_port = lexical_cast<int>(config().attr("redirectPort"));
	} catch(bad_lexical_cast) { }

	try {
		_accept_count = lexical_cast<int>(config().attr("acceptCount"));
	} catch(bad_lexical_cast) { }

	try {
		_max_thread = lexical_cast<size_t>(config().attr("maxThreads"));
	} catch (bad_lexical_cast) { }

	try {
		_max_spare_thread=lexical_cast<size_t>(config().attr("maxSpareThread"));
	} catch (bad_lexical_cast) { }

	try {
		_min_spare_thread=lexical_cast<size_t>(config().attr("minSpareThread"));
	} catch (bad_lexical_cast) { }

	// default : true.
	_disable_to = !(config().attr("disableUploadTimeout") == "false");

	// default : false.
	_lookup = config().attr("enableLookups") == "true";

	struct sockaddr_in  saddr;
	if ((_socket = socket(AF_INET, SOCK_STREAM, 0)) < 0)
		throw std::runtime_error(
						string("The error occurred by 'socket' function. ") +
						strerror(errno));

	int ret;
	int opt = 1;
	ret = setsockopt(_socket, SOL_SOCKET, SO_REUSEADDR, &opt, sizeof(opt));
	if (ret != 0)
		throw std::runtime_error(string("The error occurred by 'setsockopt' "
								 "function. ") + strerror(errno));

	setsockopt(_socket, SOL_SOCKET, SO_KEEPALIVE, &opt, sizeof(opt));
	if (ret != 0)
		throw std::runtime_error(string("The error occurred by 'setsockopt' "
								 "function. ") + strerror(errno));

	opt = 15;
	setsockopt(_socket, SOL_SOCKET, SO_LINGER, &opt, sizeof(opt));
	if (ret != 0)
		throw std::runtime_error(string("The error occurred by 'setsockopt' "
								 "function. ") + strerror(errno));

	setsockopt(_socket, IPPROTO_TCP, TCP_NODELAY, &opt, sizeof(opt));
	if (ret != 0)
		throw std::runtime_error(string("The error occurred by 'setsockopt' "
								 "function. ") + strerror(errno));

	memset((char *)&saddr, 0x00, sizeof(saddr));
	saddr.sin_family = AF_INET;
	saddr.sin_addr.s_addr = INADDR_ANY;
	saddr.sin_port = htons(getPort());

	if (::bind(_socket, (struct sockaddr *)&saddr, sizeof(saddr)) < 0)
		throw std::runtime_error(string("The error occurred by 'bind' "
								 "function. ") + strerror(errno));

#ifdef LINUX
	if(listen(_socket, SOMAXCONN) < 0)
#else
	if(listen(_socket, 128) < 0)
#endif
		throw std::runtime_error(string("The error occurred by 'listen' "
								 "function. ") + strerror(errno));

	_addr_length = sizeof(_addr);

	// initialization SSL libraries.
	SSL_library_init();
	SSL_load_error_strings();

	for (size_t i = 0; i < _min_spare_thread; i++)
		setSpareThread(new SSLConnection(this, _socket));

	_accepted = true;

}

void SSLConnector::destroy()
{
	getLogger()->log("SSLConnector::destroy - Beginning of termination.",
					Logger::INFO_LOG);

	{
		recursive_mutex::scoped_lock lock1(_execute_mutex);
		recursive_mutex::scoped_lock lock2(_spare_mutex);
		_accepted = false;
	}

	::close(_socket);

	_wait_thread->join();
	delete _wait_thread;

	deque<Connection *>::iterator i = _execute_thread.begin();
	for (; i != _execute_thread.end(); i++)
		delete *i;

	for (i = _spare_thread.begin(); i != _spare_thread.end(); i++)
		delete *i;
}

void SSLConnector::acceptHandler(handler_t handler)
{
	_handler = handler;
}

Connector::handler_t SSLConnector::acceptHandler()
{
	return _handler;
}

void SSLConnector::accept()
{
	_wait_thread = new thread(boost::bind(&SSLConnector::wait, this));
}


//
// private -------------------------------------------------------------------
//

void SSLConnector::wait()
{
#ifdef DEBUG
	std::cerr << "SSLConnector::wait Enter" << std::endl;
#endif
	int n = 0;

	while (_accepted) {
		struct timeval t;
		t.tv_sec = 0;
		t.tv_usec= 50 * 1000;

		if (select(FD_SETSIZE, NULL, NULL, NULL, &t) < 0)
			throw std::runtime_error("select() "
									 "error occurred while executing it.");

		if (_spare_thread.size() < _min_spare_thread / 2 &&
			_execute_thread.size() < _max_thread)
		{
			setSpareThread(new SSLConnection(this, _socket));
			continue;
		}
#if 0
std::cerr << "spare size  :" << _spare_thread.size() << std::endl;
std::cerr << "execute size:" << _execute_thread.size() << std::endl;
#endif
		if (n++ != 10)
			continue;

		if (_spare_thread.size() > _min_spare_thread) {
			Connection* p;
			{
				recursive_mutex::scoped_lock lock(_spare_mutex);
				p = _spare_thread.back();
				_spare_thread.pop_back();
			}
			removeExecuteThread(p);
			removeSpareThread(p);

#ifdef DEBUG
	std::cerr << "Connector delete Connection*" << std::endl;
#endif
			delete p;
		}
		n = 0;
	}
#ifdef DEBUG
	std::cerr << "SSLConnector::wait Enter" << std::endl;
#endif
}
