/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <boost/lexical_cast.hpp>
#include <servlet/ServletContext.h>
#include <servlet/RequestDispatcher.h>
#include <servlet/http/Cookie.h>
#include <sl/inet/types.h>

#include <si/core/ErrorPage.h>
#include <si/core/HttpServletRequestImpl.h>
#include <si/core/HttpServletResponseImpl.h>
#include <si/interface/Request.h>

#include "LaConnection.h"
#include "LaResponse.h"
using namespace modules::connector;


//
// Interface/Destructor.
//

LaResponse::LaResponse()
{ }


LaResponse::~LaResponse()
{ }


//
// Member functions.
//
void LaResponse::init(LaConnection* parent, sl::http::http_response& res)
{
    _parent = parent;
    _response = &res;
    _is_forwarding = false;
    _cookies.clear();
}

std::ostream &LaResponse::copy_to(std::ostream &os)
{
    return os;
}


std::istream &LaResponse::copy_from(std::istream &is)
{
    return is;
}


void LaResponse::protocol(const std::string &protocol)
{
    _response->version(protocol);
}


void LaResponse::status(int code, const std::string &msg)
{
    _response->status_code(code);
    _response->reason_phrase(msg);
}


void LaResponse::header(const std::string &n, const std::string &v)
{
    _response->msg_header(n, v);
}


void LaResponse::content(const std::string &content)
{
    _response->msg_body(content);
}

void LaResponse::flush()
{
    _parent->writer().commit(*_response);
}

void LaResponse::flushHeader()
{
    // 쥹ݥ󥹥ɤ̤ξ 200 OK ꤷƤ.
    if (_response->status_code() == 0)
        _response->status_code(sl::http::status_code::OK);

    if (_response->reason_phrase().empty())
        _response->reason_phrase(
            sl::http::status_code::to_string(_response->status_code()));

    // å򥳥ԡ
    std::vector<servlet::http::Cookie>::iterator i = _cookies.begin();
    for (; i != _cookies.end(); i++) {
        std::string path  = !i->getPath().empty() ? "; Path=" + i->getPath() : "";
        _response->msg_header("Set-Cookie",
                              i->getName() + "=" + i->getValue() + path);
    }

    _parent->writer().flush_headers(*_response);
}

void LaResponse::flushBody()
{
    _parent->writer().flush_content(*_response);
}

void LaResponse::cleanup()
{
    _response->reset();
}

std::string LaResponse::text()
{
    return _response->text();
}


std::string LaResponse::content()
{
    return _response->msg_body();
}


void LaResponse::eraseHeader(const std::string &name)
{
    return _response->erase_header(name);
}

void LaResponse::setContext(const boost::shared_ptr<si::interface::Context>& c)
{
    _context = c;
}

boost::shared_ptr<si::interface::Context> LaResponse::getContext() const
{
    return _context;
}

void LaResponse::setContentType(const std::string &s)
{
    _response->msg_header("Content-Type", s);
}

void LaResponse::setContentLength(int i)
{
    std::string length;
    try {
        length = boost::lexical_cast<std::string>(i);
    } catch (...) { }

    _response->msg_header("Content-Length", length);
}

std::string LaResponse::getCharacterEncoding()
{
    return _response->msg_header("Character-Encoding");
}

std::string LaResponse::getContentType()
{
    return _response->msg_header("Content-Type");
}

void LaResponse::setContent(const std::string& s)
{ 
    _response->msg_body(s);
}

void LaResponse::addCookie(const servlet::http::Cookie& c)
{
    _cookies.push_back(c);
}

void LaResponse::addHeader(const std::string& s, const std::string& v)
{
    _response->msg_header(s, v);
}

void LaResponse::setHeader(const std::string& s, const std::string& v)
{
    _response->msg_header(s, v);
}

std::string LaResponse::getHeader(const std::string &s)
{
    return _response->msg_header(s);
}

bool LaResponse::containsHeader(const std::string& s)
{
    return !_response->msg_header(s).empty();
}

void LaResponse::sendError(int c)
{
    sendError(c, sl::http::status_code::to_string(c));
}

void LaResponse::sendError(int c, const std::string& s)
{
    _response->status_code(c);
    _response->reason_phrase(sl::http::status_code::to_string(c));

    bool success = false;

    eraseHeader("Content-Length");

    /*
     * sendError  servlet::RequestDispatcher::forward Ǽ¸Ƥ뤿ᡢ
     * forward  sendError ¹Ԥ̵¥롼פˤʤäƤޤΤ
     * ʣμ¹ԤϤʤ褦ˤƤ
     */

    if (!_is_forwarding && _context) {
        _is_forwarding = true;

        try {
            std::string path = _context->getErrorPage(c);
            if (!path.empty()) {
                using namespace si::core;

                HttpServletRequestImpl  req(*_parent, *_context);
                HttpServletResponseImpl res(*_parent, *_context);

                std::auto_ptr<servlet::RequestDispatcher> r =
                    _context->getServletContext().getRequestDispatcher(path);
                if (*r) {
                    r->forward(req, res);
                    success = true;
                    res.flushBuffer();
                }
            }
        } catch (std::exception& e) {
            std::cerr << e.what() << std::endl;
        }
    }

    if (!success) {
        setContentType("text/html;charset=iso-8859-1");
        std::string error = ErrorPage().create(c, "", "", s);
        setContent(error);

        addHeader("Content-Length",
                  boost::lexical_cast<std::string>(error.length()));
        flushHeader();
        flushBody();
    }
    flush();
}

void LaResponse::sendRedirect(const std::string& s)
{
    _response->status_code(307);
    _response->reason_phrase(sl::http::status_code::to_string(307));
    addHeader("Location", s);
    addHeader("Content-Length", "0");
    flushHeader();
    flush();
}

void LaResponse::setStatus(int c)
{
    _response->status_code(c);
    _response->reason_phrase(sl::http::status_code::to_string(c));
}

void LaResponse::reset()
{
    std::multimap<std::string, std::string> headers = _response->msg_headers();
    std::multimap<std::string, std::string>::const_iterator i = headers.begin();
    for (; i != headers.end(); i++)
        _response->erase_header(i->second);
}
