/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <boost/algorithm/string/case_conv.hpp>
#include <boost/algorithm/string/split.hpp>
#include <boost/algorithm/string/trim.hpp>
#include <boost/algorithm/string/predicate.hpp>
#include <boost/lexical_cast.hpp>

#include "LaConnection.h"
#include "LaRequest.h"
using namespace modules::connector;


//
// Constructor/Destructor.
//

LaRequest::LaRequest()
    : _parent(0), _request(0), _stream(0)
{ }


LaRequest::~LaRequest()
{ }


//
// Member functions.
//

void LaRequest::init(LaConnection* parent, sl::http::http_request& req)
{
    _parent = parent;
    _request= &req;
}


std::string LaRequest::method() const
{
    return _request->method();
}


std::string LaRequest::uri() const
{
    return _request->abs_path();
}


std::string LaRequest::protocol() const
{
    return _request->version();
}


std::string LaRequest::header(const std::string &name) const
{
    return _request->msg_header(name);
}


void LaRequest::cleanup()
{
    _request->reset();

    if (_stream) {
        delete _stream;
        _stream = 0;
    }
}


std::istream &LaRequest::copy_from(std::istream &is)
{
    return is;
}


std::ostream &LaRequest::copy_to(std::ostream &os)
{
    os << _request->text();

    return os;
}


void LaRequest::content(const std::string& s)
{
    _request->msg_body(s);
}

std::string LaRequest::text()
{
    return _request->text();
}

std::string LaRequest::getParameter(const std::string& s)
{
    const sl::http::parameter_map& params = _request->parameter();

    sl::http::parameter_map::const_iterator i =
        std::find_if(params.begin(), params.end(),
                        sl::http::parameter_first_equal(s));
    if (i != params.end())
        return sl::http::urldecode(i->second);
    return std::string();
}


std::vector<std::string> LaRequest::getParameterNames()
{
    const sl::http::parameter_map& params = _request->parameter();

    std::vector<std::string> result;
    std::transform(params.begin(), params.end(),
                    std::back_inserter(result),
                    sl::http::url_decoder(sl::http::parameter_first()));
    return result;
}

std::vector<std::string> LaRequest::getParameterValues(const std::string& s)
{
    sl::http::parameter_map params = _request->parameter();

    std::vector<std::string> result;
    std::transform(params.begin(), params.end(),
                    std::back_inserter(result),
                    sl::http::url_decoder(sl::http::parameter_second()));
    return result;
}

std::multimap<std::string, std::string> LaRequest::getParameterMap()
{
    return sl::http::parameter_to_multimap(_request->parameter());
}

std::string LaRequest::getCharacterEncoding()
{
    return _request->msg_header("Character-Encoding");
}

int LaRequest::getContentLength()
{
    return _request->content_length();
}

std::string LaRequest::getContentType()
{
    return _request->msg_header("Content-Type");
}

std::string LaRequest::getProtocol()
{
    return _request->version();
}

std::string LaRequest::getScheme()
{
    return _parent->scheme();
}

bool LaRequest::isSecure()
{
    return false;
}

std::string LaRequest::getServerName()
{
    return _parent->serverHost();
}

int LaRequest::getServerPort()
{
    return _parent->serverPort();
}

std::string LaRequest::getLocalAddr()
{
    return _parent->localAddr();
}

std::string LaRequest::getLocalName()
{
    return _parent->localHost();
}

int LaRequest::getLocalPort()
{
    return _parent->localPort();
}

std::string LaRequest::getRemoteAddr()
{
    return _parent->remoteAddr();
}

std::string LaRequest::getRemoteHost()
{
    return _parent->remoteHost();
}

int LaRequest::getRemotePort()
{
    return _parent->remotePort();
}

std::string LaRequest::getAuthType()
{
    return "";
}

std::vector<servlet::http::Cookie> LaRequest::getCookies()
{
    typedef std::multimap<std::string, std::string> multimap_type;

    std::vector<servlet::http::Cookie> result;

    multimap_type headers = _request->msg_headers();
    multimap_type::const_iterator i = headers.begin();
    for (; i != headers.end(); i++) {
        if (boost::equals("cookie", i->first, boost::is_iequal())) {
            std::vector<std::string> cookies;
            boost::split(cookies, i->second, boost::is_any_of(";"));

            for (std::vector<std::string>::iterator j = cookies.begin();
                 j != cookies.end(); j++)
            {
                std::vector<std::string> c;
                boost::split(c, *j, boost::is_any_of("="));
                if (c.size() == 2) {
                    result.push_back(servlet::http::Cookie(
                            boost::trim_copy(c[0]), boost::trim_copy(c[1])));
                }
            }
        }
    }
    return result;
}


std::string LaRequest::getHeader(const std::string &s)
{
    return _request->msg_header(s);
}

std::vector<std::string> LaRequest::getHeaders(const std::string &s)
{
    typedef std::multimap<std::string, std::string> multimap_type;

    std::vector<std::string> v;

    multimap_type headers = _request->msg_headers();
    multimap_type::const_iterator i = headers.begin();
    for (; i != headers.end(); i++) {
        if (boost::equals(s, i->first, boost::is_iequal()))
            v.push_back(i->second);
    }
    return v;
}

std::vector<std::string> LaRequest::getHeaderNames()
{
    typedef std::multimap<std::string, std::string> multimap_type;

    std::vector<std::string> v;
    multimap_type headers = _request->msg_headers();
#if 1
    multimap_type::const_iterator i = headers.begin();
    for (; i != headers.end(); i++)
        v.push_back(i->first);
#else
    transform(headers.begin(), headers.end(),
                std::back_inserter(v), sl::http::parameter_first());
#endif
    return v;
}

std::string LaRequest::getMethod()
{
    return _request->method();
}

std::string LaRequest::getRequestURL()
{
    std::string s = getScheme() + "://" + _parent->serverHost();
    if (_parent->serverPort() != 0)
        s += ":" + boost::lexical_cast<std::string>(_parent->serverPort());
    return s + _request->uri();
}

std::string LaRequest::getRequestURI()
{
    return _request->abs_path();
}

std::string LaRequest::getQueryString()
{
    return _request->query();
}

std::string LaRequest::getContent()
{
    return _request->msg_body();
}

std::istream& LaRequest::getInputStream()
{
    /**
     * Servlet Ϥ륹ȥ꡼
     *
     * Servlet νǥȥ꡼򤹤٤ɤߤȥȥ꡼ξ֤
     * eof + fail ξ֤ˤʤäƤޤ.
     * ưΤåȤȤƤ iostream ȤƼƤ뤿
     * istream ȤƻѤ eof + fail ȤʤäǤäƤ
     * ostream ȤƻѤư쥹ݥ󥹤ʤ.
     * ΤᡢХåեʬ sockstreambuf 򤽤Τޤ޻Ѥ
     * ȥ꡼(Ǥistream) ϿΤѰդ.
     */
    _stream = new std::istream(0);

    if (_request->msg_header("Content-Type") ==
                                 "application/x-www-form-urlencoded")
    {
        _sstream.str(_request->msg_body());
        _stream->rdbuf(_sstream.rdbuf());
    } else
        _stream->rdbuf(_request->stream().rdbuf());

    return *_stream;
}
