/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <boost/bind.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/thread.hpp>

#include <si/interface/Logger.h>

#include "LaConnector.h"
using namespace modules::connector;


//
// Constructor/Destructor
//

LaConnector::LaConnector() { }

LaConnector::~LaConnector() { }


//
// Member functions.
//

void LaConnector::setPort(int port)
{
    _port = port;
}

int LaConnector::getPort()
{
    return _port;
}

void LaConnector::init()
{
    if (!getLogger())
        setLogger(getParent()->getLogger());

    getLogger()->log("LaConnector::init - Beginning of initialization.",
                     si::interface::Logger::INFO_LOG);

    size_t max_t;
    size_t max_s;
    size_t min_s;

    try {
        _port = boost::lexical_cast<int>(config().attr("port"));
        _port_s = config().attr("port");

        if (config().attr("redirectPort").empty())
            _redirect_port = 0;
        else
            _redirect_port =
                    boost::lexical_cast<int>(config().attr("redirectPort"));

        if (config().attr("acceptCount").empty())
            _accept_count  = 0;
        else
            _accept_count  =
                boost::lexical_cast<int>(config().attr("acceptCount"));

        max_t = boost::lexical_cast<size_t>(config().attr("maxThreads"));
        max_s = boost::lexical_cast<size_t>(config().attr("maxSpareThread"));
        min_s = boost::lexical_cast<size_t>(config().attr("minSpareThread"));

    } catch (boost::bad_lexical_cast& e) {
        throw std::invalid_argument("Invalid configuretion.");
    }

    // default : true.
    _disable_to = !(config().attr("disableUploadTimeout") == "false");

    // default : false.
    _lookup = config().attr("enableLookups") == "true";

    // default : false.
    _ssl = config().attr("enableSSL") == "true";


    time_t  ct = 300; // ³ǡޤǤΥॢȻ.
    time_t  pt = 30;  // ֤ΥॢȻ.
    time_t  kt = ct;  // keep-alive ॢȻ.
    size_t  ks = 30;  // keep-alive ³ݻ.

    if (_ssl) {
        _cert_file    = config().attr("certificateFile");
        _private_file = config().attr("privateKeyFile");

        _ssl_connector_impl =
            new ssl_connector_impl_type(
                boost::bind(&LaConnector::ssl_access_handle, this, _1));

        _ssl_connector_impl->init(
            ssl_connector_impl_type::connector_args(ks),
            ssl_connector_impl_type::acceptor_args(_port_s, _cert_file, _private_file),
            ssl_connector_impl_type::connection_args(ct, pt, kt),
            ssl_connector_impl_type::pool_args(max_t, max_s, min_s));
    } else {
        _http_connector_impl =
            new http_connector_impl_type(
                boost::bind(&LaConnector::http_access_handle, this, _1));
        _http_connector_impl->init(
            http_connector_impl_type::connector_args(ks),
            http_connector_impl_type::acceptor_args(_port_s),
            http_connector_impl_type::connection_args(ct, pt, kt),
            http_connector_impl_type::pool_args(max_t, max_s, min_s));
    }
}

void LaConnector::destroy()
{
    getLogger()->log("LaConnector::destroy - Beginning of termination.",
                     si::interface::Logger::INFO_LOG);

    if (_ssl)
        _ssl_connector_impl->destroy();
    else
        _http_connector_impl->destroy();

    std::vector<boost::shared_ptr<boost::thread> >::iterator i =
        _accept_thread.begin();
    for (; i != _accept_thread.end(); i++)
        (*i)->join();

    _accept_thread.clear();
}

void LaConnector::acceptHandler(handler_t handler)
{
    _handler = handler;
}

void LaConnector::accept()
{
    if (_ssl) {
        for (int i = 0; i < 10; i++)
            _accept_thread.push_back(
                boost::shared_ptr<boost::thread>(
                    new boost::thread(
                        boost::bind(&ssl_connector_impl_type::accept,
                                      _ssl_connector_impl))));
    } else {
        for (int i = 0; i < 10; i++)
            _accept_thread.push_back(
                boost::shared_ptr<boost::thread>(
                    new boost::thread(
                        boost::bind(&http_connector_impl_type::accept,
                                      _http_connector_impl))));
    }
}

void LaConnector::ssl_access_handle(ssl_connector_impl_type::connection_type& c)
{
    LaSSLConnection& con = dynamic_cast<LaSSLConnection&>(c);
    con.init(_lookup);
    _handler(con);
}

void LaConnector::http_access_handle(http_connector_impl_type::connection_type& c)
{
    LaHttpConnection& con = dynamic_cast<LaHttpConnection&>(c);
    con.init(_lookup);
    _handler(con);
}
