/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: HttpResponse.cpp,v 1.24 2008/03/01 03:37:10 cvsuser Exp $
 */

#include <boost/lexical_cast.hpp>
using namespace std;
using namespace boost;

#include <sl/net/http/http_types.h>
using namespace sl::net::http;

#include <servlet/ServletContext.h>
#include <servlet/RequestDispatcher.h>
#include <servlet/http/Cookie.h>
using namespace servlet;
using namespace servlet::http;

#include "core/ErrorPage.h"
#include "interface/Request.h"

#include "HttpResponse.h"


//
// Interface/Destructor.
//

HttpResponse::HttpResponse()
{ }


HttpResponse::~HttpResponse()
{ }


//
// Member functions.
//

ostream &HttpResponse::copy_to(ostream &os)
{
	os << _response.text();

	return os;
}


istream &HttpResponse::copy_from(istream &is)
{
	string buffer(is.peek(), char_traits<char>::eof());
	_response.parse(buffer);

	return is;
}


void HttpResponse::protocol(const string &protocol)
{
	_response.version(protocol);
}


void HttpResponse::status(int code, const string &msg)
{
	_response.status_code(code);
	_response.reason_phrase(msg);
}


void HttpResponse::header(const string &n, const string &v)
{
	_response.msg_header(n, v);
}


void HttpResponse::content(const string &content)
{
	_response.msg_body(content);
}

void HttpResponse::flush()
{
	_parent->flush();
}

void HttpResponse::flushHeader()
{
#ifdef DEBUG
	std::cerr << "HttpResponse::flushHeader" << std::endl;
#endif
	// 쥹ݥ󥹥ɤ̤ξ 200 OK ꤷƤ.
	if (_response.status_code() == 0)
		_response.status_code(status_code::OK);

	if (_response.reason_phrase().empty())
		_response.reason_phrase(status_code::to_string(_response.status_code()));

	// 괺 HTTP/1.1 Ǹ.
	if (_response.version().empty())
		_response.version("HTTP/1.1");

	// å򥳥ԡ
	vector<servlet::http::Cookie>::iterator i = _cookies.begin();
	for (; i != _cookies.end(); i++) {
		string path  = !i->getPath().empty() ? "; Path=" + i->getPath() : "";
		_response.msg_header("Set-Cookie",
							  i->getName() + "=" + i->getValue() + path);
	}

	_parent->flushHeader();
}

void HttpResponse::flushBody()
{
#ifdef DEBUG
	std::cerr << "HttpResponse::flushBody" << std::endl;
#endif
	_parent->flushBody();
}

void HttpResponse::cleanup()
{
	_response.cleanup();
}

void HttpResponse::parent(Connection *parent)
{
	_parent = parent;
}


string HttpResponse::text()
{
	return _response.text();
}


string HttpResponse::content()
{
	return _response.msg_body();
}


void HttpResponse::erase_header(const string &name)
{
	return _response.erase_header(name);
}

void HttpResponse::setContext(const boost::shared_ptr<Context>& c)
{
	_context = c;
}

boost::shared_ptr<Context> HttpResponse::getContext() const
{
	return _context;
}

void HttpResponse::setContentType(const std::string &s)
{
	_response.msg_header("Content-Type", s);
}

void HttpResponse::setContentLength(int i)
{
	std::string length;
	try {
		length = lexical_cast<std::string>(i);
	} catch (...) { }

	_response.msg_header("Content-Length", length);
}

std::string HttpResponse::getCharacterEncoding()
{
	return _response.msg_header("Character-Encoding");
}

std::string HttpResponse::getContentType()
{
	return _response.msg_header("Content-Type");
}

void HttpResponse::setContent(const std::string& s)
{ 
	_response.msg_body(s);
}

void HttpResponse::addCookie(const servlet::http::Cookie& c)
{
	_cookies.push_back(c);
}

void HttpResponse::addHeader(const std::string& s, const std::string& v)
{
	_response.msg_header(s, v);
}

void HttpResponse::setHeader(const std::string& s, const std::string& v)
{
	_response.msg_header(s, v);
}

std::string HttpResponse::getHeader(const std::string &s)
{
    return _response.msg_header(s);
}

bool HttpResponse::containsHeader(const std::string& s)
{
	return !_response.msg_header(s).empty();
}

void HttpResponse::sendError(int c)
{
	sendError(c, "");
}

void HttpResponse::sendError(int c, const std::string& s)
{
	_response.status_code(c);
	_response.reason_phrase(status_code::to_string(c));

	bool success = false;
	if (_context) {
		string path = _context->getErrorPage(c);
		if (!path.empty()) {
			ServletContext& sc = _context->getServletContext();
			RequestDispatcher& r = sc.getRequestDispatcher(path);
			if (r) {
				//r.forward(
				success = true;
			}
		}
	}

	if (!success) {
		setContentType("text/html;charset=iso-8859-1");
		flushHeader();

		string error = ErrorPage().create(c, "", "", s);
		setContent(error);
		flushBody();

		flush();
	}
}

void HttpResponse::sendRedirect(const std::string& s)
{
	_response.status_code(307);
	_response.reason_phrase(status_code::to_string(307));
	addHeader("Location", s);
	flushHeader();
	flush();
}

void HttpResponse::setStatus(int c)
{
	_response.status_code(c);
}

void HttpResponse::reset()
{
	sl::net::http::multimap_t headers = _response.msg_headers();
	sl::net::http::multimap_t::const_iterator i = headers.begin();
	for (; i != headers.end(); i++)
		_response.erase_header(i->second);
}

sl::net::http::response& HttpResponse::get()
{
	return _response;
}
