/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: HttpRequest.cpp,v 1.28 2006/10/29 03:18:49 cvsuser Exp $
 */

#include <boost/algorithm/string/case_conv.hpp>
#include <boost/algorithm/string/split.hpp>
#include <boost/algorithm/string/trim.hpp>
#include <boost/algorithm/string/predicate.hpp>
#include <boost/lexical_cast.hpp>
using namespace std;
using namespace boost;

#include "HttpRequest.h"


//
// Constructor/Destructor.
//

HttpRequest::HttpRequest()
	: _stream(0)
{ }


HttpRequest::~HttpRequest()
{ }


//
// Member functions.
//

string HttpRequest::method() const
{
	return _request->method();
}


string HttpRequest::uri() const
{
	return _request->uri();
}


string HttpRequest::protocol() const
{
	return _request->version();
}


string HttpRequest::header(const string &name) const
{
	return _request->msg_header(name);
}


void HttpRequest::cleanup()
{
	_request->cleanup();

	if (_stream) {
		delete _stream;
		_stream = 0;
	}
}


istream &HttpRequest::copy_from(istream &is)
{
	string buffer(is.peek(), char_traits<char>::eof());
	_request->parse(buffer);

	return is;
}


ostream &HttpRequest::copy_to(ostream &os)
{
	os << _request->text();

	return os;
}


void HttpRequest::parent(Connection *parent)
{
	_parent = parent;
}


void HttpRequest::set(sl::net::http::request& req)
{
	_request = &req;
}


void HttpRequest::content(const string &s)
{
	_request->msg_body(s);
}

string HttpRequest::text()
{
	return _request->text();
}

std::string HttpRequest::getParameter(const std::string &s)
{
	sl::net::http::ordermap_t params;
	if (sl::net::http::has_parameter(*_request))
		params = _request->parameter();
	else
		params = _request->query();

	sl::net::http::ordermap_t::const_iterator i = params.begin();
	for (; i != params.end(); i++) {
		if (s == i->first)
			return sl::net::http::urldecode(i->second);
	}
	return "";
}

std::vector<std::string> HttpRequest::getParameterNames()
{
	std::vector<std::string> result;

	sl::net::http::ordermap_t params;
	if (sl::net::http::has_parameter(*_request))
		params = _request->parameter();
	else
		params = _request->query();

	sl::net::http::ordermap_t::const_iterator i = params.begin();
	for (; i != params.end(); i++)
		result.push_back(sl::net::http::urldecode(i->first));
	return result;
}

std::vector<std::string> HttpRequest::getParameterValues(const std::string& s)
{
	std::vector<std::string> result;

	sl::net::http::ordermap_t params;
	if (sl::net::http::has_parameter(*_request))
		params = _request->parameter();
	else
		params = _request->query();

	sl::net::http::ordermap_t::const_iterator i = params.begin();
	for (; i != params.end(); i++) {
		if (i->first == s)
			result.push_back(sl::net::http::urldecode(i->second));
	}
	return result;
}

std::multimap<std::string, std::string> HttpRequest::getParameterMap()
{
	std::multimap<std::string, std::string> result;

	sl::net::http::ordermap_t params;
	if (sl::net::http::has_parameter(*_request))
		params = _request->parameter();
	else
		params = _request->query();

	sl::net::http::ordermap_t::const_iterator i = params.begin();
	for (; i != params.end(); i++)
		result.insert(std::make_pair(i->first, i->second));
	return result;
}

std::string HttpRequest::getCharacterEncoding()
{
	return _request->msg_header("Character-Encoding");
}

int HttpRequest::getContentLength()
{
	return sl::net::http::content_length(*_request);
}

std::string HttpRequest::getContentType()
{
	return _request->msg_header("Content-Type");
}

std::string HttpRequest::getProtocol()
{
	return _request->version();
}

std::string HttpRequest::getScheme()
{
	return "http";
}

bool HttpRequest::isSecure()
{
	return false;
}

std::string HttpRequest::getServerName()
{
	return _parent->serverHost();
}

int HttpRequest::getServerPort()
{
	return _parent->serverPort();
}

std::string HttpRequest::getLocalAddr()
{
	return _parent->localAddr();
}

std::string HttpRequest::getLocalName()
{
	return _parent->localHost();
}

int HttpRequest::getLocalPort()
{
	return _parent->localPort();
}

std::string HttpRequest::getRemoteAddr()
{
	return _parent->remoteAddr();
}

std::string HttpRequest::getRemoteHost()
{
	return _parent->remoteHost();
}

int HttpRequest::getRemotePort()
{
	return _parent->remotePort();
}

std::string HttpRequest::getAuthType()
{
	return "";
}

std::vector<servlet::http::Cookie> HttpRequest::getCookies()
{
	vector<servlet::http::Cookie> result;

	sl::net::http::multimap_t headers = _request->msg_headers();
	sl::net::http::multimap_t::const_iterator i = headers.begin();
	for (; i != headers.end(); i++) {
		if (equals("cookie", i->first, is_iequal())) {
			vector<string> cookies;
			algorithm::split(cookies, i->second, is_any_of(";"));

			for (vector<string>::iterator j = cookies.begin();
				 j != cookies.end(); j++)
			{
				vector<string> c;
				algorithm::split(c, *j, is_any_of("="));
				if (c.size() == 2) {
					result.push_back(servlet::http::Cookie(
										trim_copy(c[0]), trim_copy(c[1])));
				}
			}
		}
	}
	return result;
}

std::string HttpRequest::getHeader(const std::string &s)
{
	return _request->msg_header(s);
}

std::vector<std::string> HttpRequest::getHeaders(const std::string &s)
{
	std::vector<std::string> v;

	sl::net::http::multimap_t headers = _request->msg_headers();
	sl::net::http::multimap_t::const_iterator i = headers.begin();
	for (; i != headers.end(); i++) {
		if (equals(s, i->first, is_iequal()))
			v.push_back(i->second);
	}
	return v;
}

std::vector<std::string> HttpRequest::getHeaderNames()
{
	std::vector<std::string> v;

	sl::net::http::multimap_t headers = _request->msg_headers();
	sl::net::http::multimap_t::const_iterator i = headers.begin();
	for (; i != headers.end(); i++)
		v.push_back(i->first);
	return v;
}

std::string HttpRequest::getMethod()
{
	return _request->method();
}

std::string HttpRequest::getRequestURL()
{
	std::string host = _parent->targetHost();
	if (host.empty())
		return "";
	return "http://" + _parent->targetHost() + _request->uri();
}

std::string HttpRequest::getRequestURI()
{
	return _request->uri();
}

std::string HttpRequest::getQueryString()
{
	return _request->query_string();
}

std::string HttpRequest::getContent()
{
	return _request->msg_body();
}

std::istream& HttpRequest::getInputStream()
{
	if (_request->msg_header("Content-Type") ==
										"application/x-www-form-urlencoded") {

		_stream = new std::istream(0);
		_sstream.str(_request->msg_body());
		_stream->rdbuf(_sstream.rdbuf());

		return *_stream;
	} else
		return _request->stream();
}
