/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: HttpConnection.cpp,v 1.59 2008/03/01 03:37:09 cvsuser Exp $
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <netinet/in.h>
#include <unistd.h>

#include <boost/algorithm/string/predicate.hpp>
#if 0
#include <boost/iostreams/copy.hpp>
#include <boost/iostreams/filter/gzip.hpp>
#include <boost/iostreams/filtering_streambuf.hpp>
#include <boost/lexical_cast.hpp>
#else
#include <boost/lexical_cast.hpp>
#endif
using namespace std;
using namespace boost;

#include <sl/io/fdstream.h>
#include <sl/net/http/date.h>
#include <sl/net/http/http_function.h>
#include <sl/sys/print_trace.h>
#include <sl/sys/clock.h>

#include "include/si.h"
#include "HttpConnector.h"
#include "HttpConnection.h"
#include "HttpRequest.h"
#include "HttpResponse.h"

#ifndef HOST_NAME_MAX
#define HOST_NAME_MAX 1024
#endif

const int	ENCODE_GZIP = 0;
const int	ENCODE_ZLIB = 1;

#ifdef debug
#undef debug
#endif

#if 0
#define debug fprintf
#else
#define debug(...)
#endif

//
// Constructor/Destructor
//


HttpConnection::HttpConnection(HttpConnector *parent, int sock)
	: _available(false), _reading(false), _thread_loop(true)
{
#ifdef DEBUG
	cout << "HttpConnection::HttpConnection " << hex << this << dec << endl;
#endif
	_parent = parent;
	_accept_socket = sock;

	_request.parent(this);
	_response.parent(this);

	_compress = parent->config().attr("compression") == "true";
	_lookup   = parent->config().attr("enableLookups") == "true";
	_timeout  = parent->getTimeout();

	_wait_thread = new thread(boost::bind(&HttpConnection::wait, this));
}


HttpConnection::~HttpConnection()
{
#ifdef DEBUG
	cout << "HttpConnection::~HttpConnection " << hex << this << dec << endl;
#endif
	_thread_loop = false;

	this->close();
	_wait_thread->join();
	delete _wait_thread;
}



//
// Operators.
//

bool HttpConnection::operator!()
{
	return !_available;
}


//
// Member functions.
//

void HttpConnection::wait()
{
	while (_thread_loop) {
		fd_set  fds;
		FD_ZERO(&fds);
		FD_SET(_accept_socket, &fds);

		struct timeval t;
		t.tv_sec = 1;
		t.tv_usec= 0;

		if (select(_accept_socket + 1, &fds, NULL, NULL, &t) < 0)
			return;

		if (!FD_ISSET(_accept_socket, &fds))
			continue;

		int s = ::accept(_accept_socket,
						 (struct sockaddr *)&_parent->_addr,
						 &_parent->_addr_length);
		sl::sys::clock::start();

		if (s == -1) {
#ifdef DEBUG
			std::cerr << "HttpConnection::wait - accept error" << std::endl;
#endif
			return;
		}

		_parent->removeSpareThread(this);
		_parent->setExecuteThread(this);
		_socket = s;
		_available = true;

		debug(stdout, "reset %f\n", sl::sys::clock::elapsed());

		_iostream.reset(_socket);
		_iostream.clear();

		debug(stdout, "reset end %f\n", sl::sys::clock::elapsed());

		Connector::handler_t handler = _parent->acceptHandler();
		while (init()) {
			debug(stdout, "handler start %f\n", sl::sys::clock::elapsed());
			handler(*this);
			debug(stdout, "handler end %f\n", sl::sys::clock::elapsed());
		}

		if (available())
			this->close();
		_parent->removeExecuteThread(this);
		_parent->setSpareThread(this);
	}
}

bool HttpConnection::init()
{
#ifdef DEBUG
	cout << "HttpConnection::init Enter " << hex << this << dec << endl;
	cout << "available:" << available() << endl;
#endif
	if (!available())
		return false;

	debug(stdout, "init %f\n", sl::sys::clock::elapsed());

	// ॢȤ30ä
	// ɤ߹².
	_iostream.timeout(30);
	_iostream.rlength(-1);

	int ret = sl::net::http::read_message(_iostream, _real_msg);

	if (ret <= 0) {
#ifdef DEBUG
	cerr << "HttpConnection::init Error:" << ret << endl;
#endif
		_real_msg.cleanup();
		return false;
	}

	server_info(_socket);
	remote_info(_socket);
	local_info(_socket);

	int length = sl::net::http::content_length(_real_msg);
#ifdef DEBUG
	cerr << "CONTENT-LENGTH:" << length << std::endl;
#endif
	if (length > 0)
		_iostream.rlength(length);

	_request.set(_real_msg);

	debug(stdout, "init end %f\n", sl::sys::clock::elapsed());

#ifdef DEBUG
	cout << "HttpConnection::init Exit " << hex << this << dec << endl;
#endif
	return true;
}


void HttpConnection::flushHeader()
{
#ifdef DEBUG
	cout << "HttpConnection::flashHeader Enter reading:" << _reading << endl;
#endif
	debug(stdout, "flush header %f\n", sl::sys::clock::elapsed());

	// Server إå
	_response.header("Server", si::server());
	_response.header("Date", sl::net::http::date::current_time());

	// Connection إå
	_chunked = false;
	if (_request.protocol() == "HTTP/1.1" &&
		equals(_request.header("Connection"), "Keep-Alive", is_iequal()))
	{
		_response.header("Connection", "Keep-Alive");

		if (!_response.containsHeader("Content-Length")) {
			_response.header("Transfer-Encoding", "chunked");
			_chunked = true;
		}
	} else
		_response.header("Connection", "close");

	sl::net::http::write_headers(_iostream, _response.get());

#ifdef DEBUG
//	std::cerr << _response.get().text_header() << std::endl;
#endif
	debug(stdout, "flush header end %f\n", sl::sys::clock::elapsed());
}


void HttpConnection::flushBody()
{
#ifdef DEBUG
	std::cerr << "BODY-LENGTH:" << _response.get().msg_body().length()
			  << std::endl;
	std::cerr << "BODY-LENGTH:" << std::hex << _response.get().msg_body().length()
			  << std::dec << "" << std::endl;
#endif
	debug(stdout, "flush body %f\n", sl::sys::clock::elapsed());
	if (_response.get().msg_body().empty())
		return;

	if (_chunked) {
		std::string chunk = _response.get().msg_body();

		std::ostringstream ss;
		ss << std::hex << chunk.length() << std::dec;

		chunk.insert(0, "\r\n", 2);
		chunk.insert(0, ss.str());
		chunk.append("\r\n", 2);

		_response.get().msg_body(chunk);
	}

	sl::net::http::write_body(_iostream, _response.get());
}


void HttpConnection::flush()
{
#ifdef DEBUG
	cout << "HttpConnection::flash Enter reading:" << _reading << endl;
#endif
	debug(stdout, "flush %f\n", sl::sys::clock::elapsed());

	if (!available())
		return;

	sl::net::http::read_unnecessary(_socket, 0);

	if (!equals(_response.getHeader("Connection"), "Keep-Alive", is_iequal()))
		this->close();

	if (_chunked) {
		_response.get().msg_body("\r\n0\r\n\r\n");
		sl::net::http::write_body(_iostream, _response.get());
	}

	_request.cleanup();
	_response.cleanup();

	debug(stdout, "flush end %f\n", sl::sys::clock::elapsed());
#ifdef DEBUG
	cout << "HttpConnection::flash Exit reading:" << _reading << endl;
#endif
}


void HttpConnection::close()
{
#ifdef DEBUG
	cout << "HttpConnection::close Enter " << hex << this << dec << endl;
	cerr << "HttpConnection::close reading:" << _reading << endl;
#endif
	if (available()) {
		_available = false;

		if (::shutdown(_socket, SHUT_RDWR) == -1)
			perror("shutdown");
		if (::close(_socket) == -1)
			perror("close");
	}
#ifdef DEBUG
	cout << "HttpConnection::close Exit " << hex << this << dec << endl;
#endif
}


string HttpConnection::targetHost()
{
	std::string host = _request.header("Host");
	if (host.empty()) {
		try {
			std::string port = boost::lexical_cast<std::string>(serverPort());
			host = serverHost() + ":" + port;
		} catch(...) { }
	}
	return host;
}


Request &HttpConnection::request()
{
	return _request;
}


Response &HttpConnection::response()
{
	return _response;
}


bool HttpConnection::available() const
{ 
	return _available;
}


string HttpConnection::localHost() const
{
	return _local_name;
}


string HttpConnection::localAddr() const
{
	return _local_addr;
}


int HttpConnection::localPort() const
{
	return _local_port;
}


string HttpConnection::remoteHost() const
{
	return _remote_name;
}


string HttpConnection::remoteAddr() const
{
	return _remote_addr;
}


int HttpConnection::remotePort() const
{
	return _remote_port;
}


string HttpConnection::serverHost() const
{
	return _server_name;
}


string HttpConnection::serverAddr() const
{
	return _server_addr;
}


int HttpConnection::serverPort() const
{
	return _server_port;
}


//
// --- private --------------------------------------------------------------
//

std::string HttpConnection::body_encode(const std::string& s, int type)
{
#if 0
	namespace io = boost::iostreams;

	stringstream is(s);
	stringstream os;

	io::filtering_streambuf<io::input> filter;
	if (type == ENCODE_GZIP)
		filter.push(io::gzip_compressor());
	else if (type == ENCODE_ZLIB)
		filter.push(io::zlib_compressor());
	filter.push(is);

	io::copy(filter, os);
	return os.str();
#else
	return s;
#endif
}

bool HttpConnection::remote_info(int sock)
{
	_remote_port = 0;

	struct sockaddr_in	addr;
	socklen_t			len = sizeof(addr);

	if (getpeername(sock, (struct sockaddr *)&addr, &len) < 0)
		return false;

	_remote_addr = inet_ntoa(addr.sin_addr);
	_remote_port = ntohs(addr.sin_port);

	if (_lookup) {
		struct hostent *host = gethostbyaddr((char *)&addr.sin_addr.s_addr,
											 sizeof(addr.sin_addr), AF_INET);
		if (!host)
			return false;

		_remote_name = host->h_name;
	} else {
		_remote_name = _remote_addr;
	}

	return true;
}


bool HttpConnection::server_info(int /* sock */)
{
	char buffer[HOST_NAME_MAX];
	if (gethostname(buffer, HOST_NAME_MAX) < 0)
		return false;

	_server_name = std::string(buffer);
	_server_port = _parent->getPort();

	return true;
}

bool HttpConnection::local_info(int sock)
{
	_local_port = 0;

	struct sockaddr_in	addr;
	socklen_t			len = sizeof(addr);

	if (getsockname(sock, (struct sockaddr *)&addr, &len) < 0)
		return false;

	_local_addr = inet_ntoa(addr.sin_addr);
	_local_port = ntohs(addr.sin_port);

	if (_lookup) {
		struct hostent *host = gethostbyaddr((char *)&addr.sin_addr.s_addr,
											 sizeof(addr.sin_addr), AF_INET);
		if (!host)
			return false;
		_local_name = host->h_name;
	} else
		_local_name = _local_addr;
	return true;
}
