/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: Ajp13Request.cpp,v 1.24 2006/10/29 03:18:49 cvsuser Exp $
 */

#include <iostream>
#include <boost/algorithm/string/case_conv.hpp>
#include <boost/algorithm/string/split.hpp>
#include <boost/algorithm/string/trim.hpp>
#include <boost/algorithm/string/predicate.hpp>
#include <boost/lexical_cast.hpp>
using namespace std;
using namespace boost;

#include <sl/net/ajp13/function.h>

#include "Ajp13Request.h"
using namespace modules::connector;

//
// Interface/Destructor.
//

Ajp13Request::Ajp13Request()
{ }

Ajp13Request::~Ajp13Request()
{ }


//
// Member functions.
//

string Ajp13Request::method() const
{
	return _headers.method();
}


string Ajp13Request::uri() const
{
	return _headers.req_uri();
}


string Ajp13Request::protocol() const
{
	return "AJP/1.3";
}


void Ajp13Request::cleanup()
{
	_headers.clear();
	_body_chunk.clear();
}


string Ajp13Request::header(const string &name) const
{
	return _headers.header(name);
}


istream &Ajp13Request::copy_from(istream &is)
{
#ifdef DEBUG
	cerr << "Ajp13Request::" << __FUNCTION__ << endl;
#endif

	using namespace sl::net::ajp13;
	try {
		read_packet(is, _headers);
		if (_headers.content_length() > 0)
			read_packet(is, _body_chunk);

	} catch(ajp13_error &e) {
		cerr << e.what() << endl;
		cerr << "It faild in function " << __FUNCTION__ << endl;
	}

	return is;
}


ostream &Ajp13Request::copy_to(ostream &os)
{
#ifdef DEBUG
	cerr << "Ajp13Request::" << __FUNCTION__ << endl;
#endif

	using namespace sl::net::ajp13;
	try {
		write_packet(os, _headers);
		if (_headers.content_length() > 0)
			write_packet(os, _body_chunk);

	} catch(ajp13_error &e) {
		cerr << e.what() << endl;
		cerr << "It faild in function " << __FUNCTION__ << endl;
	}

	return os;
}

void Ajp13Request::parent(Connection *parent)
{
	_parent = parent;
}


void Ajp13Request::read(int fd)
{
	try {
		sl::net::ajp13::read_packet(fd, _headers);
		if (_headers.content_length() > 0)
			read_packet(fd, _body_chunk);
	} catch(sl::net::ajp13::ajp13_error& e) {
#ifdef DEBUG
	cerr << e.what() << endl;
#endif
		throw std::runtime_error(e.what());
	}
}

std::string Ajp13Request::getParameter(const std::string &s)
{
	typedef multimap<string, string>	parameter_t;

	parameter_t params = getParameterMap();
	parameter_t::iterator i = params.find(s);
	return i!= params.end() ? i->second : "";
}

std::vector<std::string> Ajp13Request::getParameterNames()
{
	typedef multimap<string, string>	parameter_t;

	std::vector<std::string> result;

	parameter_t params = getParameterMap();
	parameter_t::iterator i = params.begin();
	for (; i != params.end(); i++)
		result.push_back(i->first);
	return result;
}

std::vector<std::string>
Ajp13Request::getParameterValues(const std::string& s)
{
	typedef multimap<string, string>	parameter_t;

	std::vector<std::string> result;

	parameter_t params = getParameterMap();
	parameter_t::iterator i = params.begin();
	for (; i != params.end(); i++) {
		if (i->first == s)
			result.push_back(i->second);
	}
	return result;
}

multimap<string, string> Ajp13Request::getParameterMap()
{
	multimap<string, string>	result;

	string method = _headers.method();
	string type   = _headers.header("Content-Type");

	string param;
	if (method == "GET" || method == "HEAD")
		param = _headers.attributes()["query_string"];
	else if (method == "POST" && type == "application/x-www-form-urlencoded")
		param = _body_chunk.body();

	if (!param.empty()) {
		vector<string> sp;
		algorithm::split(sp, param, is_any_of("=&"));

		vector<string>::iterator i = sp.begin();
		for (; i != sp.end() && (i + 1) != sp.end(); i+=2)
			result.insert(make_pair(*i, *(i + 1)));
	}
	return result;
}

std::string Ajp13Request::getCharacterEncoding()
{
	return _headers.header("Character-Encoding");
}

int Ajp13Request::getContentLength()
{
	int length = -1;
	try {
		length = lexical_cast<int>(_headers.header("Content-Length"));
	} catch (...) { }
	return length;
}

std::string Ajp13Request::getContentType()
{
	return _headers.header("Content-Type");
}

std::string Ajp13Request::getProtocol()
{
	return _headers.protocol();
}

std::string Ajp13Request::getScheme()
{
	return "http";
}

bool Ajp13Request::isSecure()
{
	return false;
}

std::string Ajp13Request::getServerName()
{
	return _headers.server_name();
}

int Ajp13Request::getServerPort()
{
	return _headers.server_port();
}

std::string Ajp13Request::getLocalAddr()
{
	return _parent->localAddr();
}

std::string Ajp13Request::getLocalName()
{
	return _parent->localHost();
}

int Ajp13Request::getLocalPort()
{
	return _parent->localPort();
}

std::string Ajp13Request::getRemoteAddr()
{
	return _headers.remote_addr();
}

std::string Ajp13Request::getRemoteHost()
{
	return _headers.remote_host();
}

int Ajp13Request::getRemotePort()
{
	return -1; // _headers.remote_port();
}

std::string Ajp13Request::getAuthType()
{
	return "";
}

std::vector<servlet::http::Cookie> Ajp13Request::getCookies()
{
	vector<servlet::http::Cookie> result;

	multimap<string, string> headers = _headers.headers();
	multimap<string, string>::iterator i = headers.begin();
	for (; i != headers.end(); i++) {
		if (equals("cookie", i->first, is_iequal())) {
			vector<string> cookies;
			algorithm::split(cookies, i->second, is_any_of(";"));

			for (vector<string>::iterator j = cookies.begin();
				 j != cookies.end(); j++)
			{
				vector<string> c;
				algorithm::split(c, *j, is_any_of("="));
				if (c.size() == 2) {
					result.push_back(servlet::http::Cookie(
									trim_copy(c[0]), trim_copy(c[1])));
				}
			}
		}
	}
	return result;
}

std::string Ajp13Request::getHeader(const std::string &s)
{
	multimap<string, string> headers = _headers.headers();
	multimap<string, string>::const_iterator i = headers.find(s);
	return (i != headers.end()) ? i->second : "";
}

std::vector<std::string> Ajp13Request::getHeaders(const std::string &s)
{
	std::vector<std::string> v;
	multimap<string, string> headers = _headers.headers();

	multimap<string, string>::const_iterator i = headers.begin();
	for (; i != headers.end(); i++) {
		if (equals(s, i->first, is_iequal()))
			v.push_back(i->second);
	}
	return v;
}

std::vector<std::string> Ajp13Request::getHeaderNames()
{
	std::vector<std::string> v;
	multimap<string, string> headers = _headers.headers();

	multimap<string, string>::const_iterator i = headers.begin();
	for (; i != headers.end(); i++)
		v.push_back(i->first);
	return v;
}

std::string Ajp13Request::getMethod()
{
	return _headers.method();
}

std::string Ajp13Request::getRequestURL()
{
	//return "http://" + _parent->targetHost() + _headers.req_uri();
	return "";
}

std::string Ajp13Request::getRequestURI()
{
	return _headers.req_uri();
}

std::string Ajp13Request::getQueryString()
{
	return _headers.attributes()["query_string"];
}

std::string Ajp13Request::getContent()
{
	return _body_chunk.body();
}

std::istream& Ajp13Request::getInputStream()
{
//	if (_headers.header("Content-Type") ==
//										"application/x-www-form-urlencoded") {

		_stream = new std::istream(0);
		_sstream.str(_body_chunk.body());
		_stream->rdbuf(_sstream.rdbuf());

		return *_stream;
//	} else
//		return _headers.stream();
}
