/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: Ajp13Connector.cpp,v 1.23 2008/05/06 16:26:11 cvsuser Exp $
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <unistd.h>

#include <cerrno>
#include <string>
#include <boost/bind.hpp>
#include <boost/thread.hpp>
#include <boost/lexical_cast.hpp>
using namespace std;
using namespace boost;

#include <si/interface/Logger.h>
using namespace si::interface;

#include "Ajp13Connector.h"
#include "Ajp13Connection.h"
using namespace modules::connector;

#include <sl/class_loader/register.hpp>
SL_REGIST_LOADABLE_CLASS(modules::connector::Ajp13Connector);

const int DEFAULT_MAX_THREAD = 10;

//
// Constructor/Destructor
//

Ajp13Connector::Ajp13Connector()
	: _port(0)
{ }


Ajp13Connector::~Ajp13Connector()
{ }


//
// Member functions.
//

void Ajp13Connector::setPort(int port)
{
	_port = port;
}


int Ajp13Connector::getPort()
{
	return _port;
}


void Ajp13Connector::acceptHandler(boost::function<void (Connection&)> handler)
{
	_handler = handler;
}

void Ajp13Connector::init()
{
#ifdef DEBUG
	cout << "Ajp13Connector::init" << endl;
#endif
    try {
        _port = lexical_cast<int>(config().attr("port"));
    } catch (boost::bad_lexical_cast&) {
        throw std::invalid_argument("invalid port number.");
    }

	struct sockaddr_in  saddr;

	if ((_socket = socket(AF_INET, SOCK_STREAM, 0)) < 0)
		throw runtime_error(
			string("The error occurred by 'socket' function. ") +
						strerror(errno));

	int flag = 1;
	setsockopt(_socket, SOL_SOCKET, SO_REUSEADDR, &flag, sizeof(flag));

	//
	// ʤɲä®٤夬...
	//
	setsockopt(_socket, IPPROTO_TCP, TCP_NODELAY, &flag, sizeof(flag));

	memset((char *)&saddr, 0x00, sizeof(saddr));

	saddr.sin_family = AF_INET;
	saddr.sin_addr.s_addr = INADDR_ANY;
	saddr.sin_port = htons(_port);

	if (::bind(_socket, (struct sockaddr *)&saddr, sizeof(saddr)) < 0)
		throw std::runtime_error(string("The error occurred by 'bind' "
								 "function. ") + strerror(errno));

#ifdef LINUX
	if(listen(_socket, SOMAXCONN) < 0)
#else
	if(listen(_socket, 128) < 0)
#endif
		throw std::runtime_error(string("The error occurred by 'listen' "
								 "function. ") + strerror(errno));

	_addr_length = sizeof(_addr);

	try {
		_max_thread = boost::lexical_cast<int>(config().attr("maxThreads"));
	} catch (boost::bad_lexical_cast& e) {
		_max_thread = DEFAULT_MAX_THREAD;
	}
}


void Ajp13Connector::destroy()
{
#ifdef DEBUG
	cout << "Ajp13Connector::destroy" << endl;
#endif
	{
		recursive_mutex::scoped_lock lock(_mutex);
		::close(_socket);
		_accepted = false;
		_empty_cond.notify_all();
	}

	thread_list_t::iterator i = _queue_threads.begin();
	for (i = _queue_threads.begin(); i != _queue_threads.end(); i++) {
		(*i)->join();
		delete *i;
	}
#ifdef DEBUG
	cerr << "Ajp13Connector::destroy queue() threads was ended." << endl;
#endif

	socket_queue_t::iterator s = _socket_queue.begin();
	for (; s != _socket_queue.end(); s++) {
		::shutdown(_socket, SHUT_RDWR);
		::close(*s);
	}
#ifdef DEBUG
	cerr << "Ajp13Connector::destroy remaining connection was closed." << endl;
#endif

	for (i = _proc_threads.begin(); i != _proc_threads.end(); i++) {
		(*i)->join();
		delete *i;
	}
#ifdef DEBUG
	cerr << "Ajp13Connector::destroy proc() threads was ended." << endl;
#endif
}


void Ajp13Connector::accept()
{
	_accepted = true;

	for (int i = 0; i < _max_thread; i++) {
		thread* thr = new thread(boost::bind(&Ajp13Connector::queue, this));
		_queue_threads.push_back(thr);
	}
}


void Ajp13Connector::queue()
{
	while (_accepted) {
		fd_set	fds;
		FD_ZERO(&fds);
		FD_SET(_socket, &fds);

		struct timeval t;
		t.tv_sec = 1;
		t.tv_usec= 0;

		if (select(FD_SETSIZE, &fds, NULL, NULL, &t) < 0)
			break;
			//throw std::runtime_error("select() "
									// "error occurred while executing it.");

		if (!FD_ISSET(_socket, &fds))
			continue;

		int s = ::accept(_socket, (struct sockaddr *)&_addr, &_addr_length);
		if (s < 0)
			continue;
		_socket_queue.push_back(s);

		Ajp13Connection* p = new Ajp13Connection(this, s);
        thread* thr = new thread(boost::bind(&Ajp13Connector::proc, this, p));
        _proc_threads.push_back(thr);
	}
#ifdef DEBUG
	cerr << "Ajp13Connector::queue Exit" << endl;
#endif
}

void Ajp13Connector::proc(Connection* p)
{
	for (;;) {
		try {
			p->init();
			_handler(*p);
		} catch(...) { break; }
	}

	delete p;
}
