/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef SERVLET_SERVLET_REQUEST_H
#define SERVLET_SERVLET_REQUEST_H

#include <map>
#include <string>
#include <vector>

#include <sl/java/io/BufferedReader.h>
#include <sl/java/lang/Object.h>
#include <sl/java/util/Enumeration.h>
#include <sl/java/util/Locale.h>
#include <servlet/ServletException.h>
#include <servlet/RequestDispatcher.h>

namespace servlet {

class ServletInputStream;

/**
 * @interface ServletRequest.
 * 饤Ȥ׵˴ޤޤƼ Servlet 黲Ȥ٤
 * 󥿡եޤ.
 *
 * ServletRequest 饹μ֤ Servlet ƥʤˤäƺ졢
 * Servlet::service дؿΰˤä Servlet ˰Ϥޤ.
 */
class ServletRequest {
public :
    /** Destructor. */
    virtual ~ServletRequest()=0;

    /**
     *  name ˤäƻꤵ줿̾°֤ޤ.
     *
     * °̾ͤΥڥ ServletRequest ڤ Servlet 
     *  Servlet ƥʡ⤷ Servlet Ȥˤäꡢ
     * Ԥʤޤ.
     * ޤб°¸ߤʤ null 󥹥󥹤֤ޤ.
     *
     * @param    name    °̾.
     * @return    sl::java::lang::Object ̾ȥڥȤʤ°.
     * @note    java.*, javax.*, sun.* ˳̾ Servlet API    ͤ
     *            ͽ󤵤Ƥ뤽ʤΤǡǤϻѤƤޤ.
     */
    virtual sl::java::lang::Object getAttribute(const std::string& name)=0;

    /**
     *  obj   name ˤäƻꤵ줿̾°Ȥ
     * ꤷޤ.
     *
     * ˻ꤵ줿̾°ꤷ֤ޤ.
     *
     * ͤȤ null ݻ obj Ϥ줿ϡremoveAttribute 
     * дؿθƤӽФˤʤޤ.
     *
     * ǤдϢꥹʡ֥Ȥ¸ߤŬڤ
     * Τ٤ǤǤϥꥹʡϢνϼƤޤ.
     *
     * @param    name    ꤹ°̾.
     * @param    obj        ꤹ sl::java::lang::Object °.
     * @note    java.*, javax.*, sun.* ˳̾ Servlet API    ͤ
     *            ͽ󤵤Ƥ뤽ʤΤǡǤϻѤƤޤ.
     */
    virtual void
    setAttribute(const std::string& name, sl::java::lang::Object obj)=0;

    /**
     *  name ˤäƻꤵ줿̾°ޤ.
     *
     * @c removeAttribute дؿˤäƺ줿°Ф
     * Ԥʤä null 󥹥󥹤֤ޤ.
     * 
     * @param    name    ꤹ°̾.
     */
    virtual void removeAttribute(const std::string& name)=0;

    /**
     * ServletContext ݻ°̾ΥꥹȤ֤ޤ.
     *
     * @return        °̾ΥꥹȤݻ sl::java::util::Enumeration 
     *                󥹥.
     */
    virtual sl::java::util::Enumeration getAttributeNames()=0;

    /**
     * ׵Υѥ᡼ꤵ줿֤̾ͤޤ.
     *
     * ׵Υѥ᡼ȤϥꥯȤդƤɲþǤꡢ
     * ̾ @c HTTP ץȥΥ꡼ʸ @c HTML  @c FORM 
     *  @c POST ꥯȤΥåܥǥʬޤ.
     *
     * ꤵ줿̾ĥѥ᡼¸ߤʤ϶ʸ
     * std::string 󥹥󥹤֤ޤ.
     *
     * @c getParameter ϻꤷ̾ʣ¸ߤƤ
     * ׵ꤵ줿ѥ᡼Ƭ֤ͤޤ.
     *
     * <pre>
     *    POST / HTTP/1.0
     *    Content-Length: 23
     * 
     *    value=a\&value=b\&value=c
     * </pre>
     *
     * Τ褦ʥꥯȤ硢
     * @c ServletRequest::getParameter( @c "value" ) ֤ͤ
     * @c "a" Ȥʤޤ.
     *
     * @param    name    ׵ѥ᡼̾.
     * @return     name б׵ѥ᡼.    
     */
    virtual std::string getParameter(const std::string& name)=0;

    /**
     * ׵˴ޤޤѥ᡼̾ΥꥹȤ֤ޤ.
     *
     * @return    ׵ѥ᡼̾Υꥹ.
     */
    virtual sl::java::util::Enumeration getParameterNames()=0;

    /**
     * ꤵ줿̾׵Υѥ᡼ƤǼ
     * @c std::vector<@c std::string> 󥹥󥹤֤ޤ.
     *
     * @param    name    ׵ѥ᡼̾.
     * @return     name б׵ѥ᡼ͤݻ
     *            @c std::vector<@c std::string> 󥹥.
     */
    virtual std::vector<std::string>
    getParameterValues(const std::string& name)=0;

    /**
     * ׵Υѥ᡼ƤǼ
     * @c std::map<@c std::string, @c std::string> 󥹥󥹤֤ޤ.
     *
     * @return    @c std::multimap<@c std::string, @c std::string>
     *             󥹥.
     */
    virtual std::multimap<std::string, std::string> getParameterMap()=0;

    /**
     * ꥯȤΥåܥǥ˻ѤƤʸ󥳡ǥ
     * ֤ޤ.ʸ󥳡ǥ󥰤ꤵƤʤ϶
     * std::string 󥹥󥹤֤ޤ.
     *
     * @return    ʸ󥳡ǥ󥰤򼨤 std::string 󥹥.
     */
    virtual std::string getCharacterEncoding()=0;

    /**
     * ꥯȤΥåܥǥ˻ѤƤʸ󥳡ǥ
     * ѹޤ.
     *
     * @param    s    ʸ󥳡ǥ󥰤򼨤 std::string 󥹥.
     * @warning     setCharacterEncoding дؿϼƤޤ.
     */
    virtual void setCharacterEncoding(const std::string& s)=0;

    /**
     * ꥯȤΥåܥǥ뤤ϡϥȥ꡼फɤ߽Ф
     * ǽʥХĹ֤ޤ.
     * Хȿʾ -1 ֤ޤ.
     *
     * @return    ɤ߽ФǽʥХĹ.
     */
    virtual int getContentLength()=0;

    /**
     * ꥯȤΥåܥǥФ MIME פ֤ޤ.
     * ʾ϶ std::string 󥹥󥹤֤ޤ.
     *
     * @return    MIME .
     */
    virtual std::string getContentType()=0;

    /**
     * ꥯȤΥåܥǥɤ߹٤ @c ServletInputStream 
     * ֤ޤ.
     *
     * @c getInputStream дؿƤӽФ
     * @c getReader дؿƤӽФԤʤ㳰ȯޤ.
     *
     * @return    åܥǥɤ߹٤
     *            @c ServletInputStream 󥹥.
     * @throw    sl::java::lang::IllegalStateException.
     */
    virtual ServletInputStream& getInputStream()=0;

    /**
     * ꥯȤΥåܥǥʸȤɤ߹٤
     * @c sl::java::io::BufferedReader ֤ޤ.
     *
     * @c getInputStream дؿƤӽФ
     * @c getReader дؿƤӽФԤʤ㳰ȯޤ.
     *
     * @return    åܥǥɤ߹٤
     *            @c BufferedReader 󥹥.
     * @throw    sl::java::lang::IllegalStateException.
     */
    virtual sl::java::io::BufferedReader getReader()=0;

    /**
     * Accept-Language إå򸵤ˡ饤Ȥ׵ˤ Locale
     * ֤ޤ.
     *
     * @warning     getLocale дؿϼƤޤ.
     */
    virtual sl::java::util::Locale getLocale()=0;

    /**
     * Accept-Language إå򸵤ˡ饤Ȥ׵ˤ Locale
     * ꥹȤ֤ޤ.
     *
     * @warning     getLocales дؿϼƤޤ.
     */
    virtual sl::java::util::Enumeration getLocales()=0;

    /**
     * ꥯȤΥץȥ̾ȥС֤ޤ.
     *
     * @return    ץȥ̾ȥСޤ std::string.
     */
    virtual std::string getProtocol()=0;

    /**
     * http/https ΥꥯȤΥ֤ޤ.
     *
     * @return    ޤ std::string.
     */
    virtual std::string getScheme()=0;

    /**
     * HTTPS Τ褦ʥ奢̿ˤꥯȤɤȽ̤
     * ֤ޤ.
     *
     * @return    奢̿ɤ򼨤 bool .
     */
    virtual bool isSecure()=0;

    /**
     * ꥯȤäФΥۥ֤̾ޤ.
     *
     * @return    ꥯȤ̾ޤ std::string.
     */
    virtual std::string getServerName()=0;

    /**
     * ꥯȤäФΥݡֹ֤ޤ.
     *
     * @return    ꥯȤݡֹ.
     */
    virtual int getServerPort()=0;

    /**
     * ꥯȤ饤ȤIPɥ쥹֤ޤ.
     *
     * @return    Servlet ƥʤưƤIPɥ쥹.
     */
    virtual std::string getLocalAddr()=0;

    /**
     * Servlet ƥʤưƤۥȤ֤̾ޤ.
     *
     * ۥ̾β˼Ԥ硢⤷ϥۥ̾βԤʤʤ
     *  IP ɥ쥹֤ޤ.
     * 
     * @return    Servlet ƥʤưƤۥȤ̾.
     */
    virtual std::string getLocalName()=0;

    /**
     * Servlet ƥʤưƤۥȤΥݡֹ֤ޤ.
     *
     * @return    Servlet ƥʤưƤۥȤΥݡֹ.
     */
    virtual int getLocalPort()=0;

    /**
     * ꥯȤ饤ȤIPɥ쥹֤ޤ.
     *
     * @return    饤ȤIPɥ쥹.
     */
    virtual std::string getRemoteAddr()=0;

    /**
     * ꥯȤ饤ȤFQDN֤ޤ.
     * ۥ̾β˼Ԥ硢⤷ϥۥ̾βԤʤʤ
     *  饤ȤIPɥ쥹֤ޤ.
     * 
     * @return    饤ȤΥۥ̾.
     */
    virtual std::string getRemoteHost()=0;

    /**
     * ꥯȤ饤ȤΥݡֹ֤ޤ.
     *
     * @return    饤ȤΥݡֹ.
     */
    virtual int getRemotePort()=0;

    /**
     * ꤵ줿ѥб꥽ɽ RequestDispatcher
     * 󥹥󥹤֤ޤ.
     *
     * ʥѥꤵ줿 null 󥹥󥹤֤ޤ.
     *
     * @param    s    ꥽ؤΥѥ.
     * @return    б RequestDispatcher 󥹥.
     */
    virtual std::auto_ptr<RequestDispatcher>
    getRequestDispatcher(const std::string& s)=0;
};

} // namespace servlet

#endif    // SERVLET_SERVLET_REQUEST_H
