/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <fstream>
#include <iostream>
#include <set>
#include <sstream>
#include <string>
#include <boost/filesystem/exception.hpp>
#include <boost/filesystem/operations.hpp>
#include <boost/filesystem/path.hpp>
using namespace std;
using namespace boost;
using namespace boost::filesystem;

#include <sl/object.hpp>
#include <sl/xml.h>

#include <sl/java/lang/Object.h>
#include <servlet/Servlet.h>
#include <servlet/http/HttpServlet.h>
#include <servlet/http/HttpServletRequest.h>
#include <servlet/http/HttpServletResponse.h>
#include <servlet/jsp/HttpJspPage.h>
using namespace servlet;
using namespace servlet::http;
using namespace servlet::jsp;

#include <si/interface/ServletWrapper.h>
#include <si/core/ErrorPage.h>
#include <si/core/ServletContextImpl.h>
using namespace si::interface;
using namespace si::core;

#include "JspServlet.h"
using namespace modules::servlets;

#include "JspLoader.h"
#include "JspContext.h"
#include "JspWrapper.h"
using namespace sj;

#include "jsp2c/jsp_compiler.h"
#include "jsp2c/util.h"
#include "runtime/JspRuntime.h"


void JspServlet::init() throw(ServletException)
{
    JspRuntime::start();

    try {
        _jsp_loader = boost::shared_ptr<JspLoader>(new JspLoader());

        _jsp_context = new JspContext();
        _jsp_context->setLoader(_jsp_loader);

        char *e = getenv("SHIBAINU_HOME");
        if (e)
            _jsp_context->home(e);

        _jsp_context->init();

    } catch (...) {
        throw ServletException("JspServlet::init faild.");
    }

    std::string debug = getInitParameter("debug");
    _debug = 0;
    if (!debug.empty()) {
        try {
            _debug = boost::lexical_cast<int>(debug);
        } catch (boost::bad_lexical_cast& e) {
            getServletContext().log("JspServlet::init failed. "
                                    "\"debug\" attribute not an integer.",
                                    servlet::ServletException(e.what()));
        }
    }

    string file = _jsp_context->home()+"/conf/" + getInitParameter("setupFile");

    // core::ServletContextImpl ͳǥƥ Context 
    try {
        ServletContextImpl& c =
                        dynamic_cast<ServletContextImpl&>(getServletContext());
        _context = c.getContainerContext();
    } catch (std::bad_cast& e) {
        throw ServletException("container is unknown implementation.");
    }

    /*
     * WEB ץꥱΥ롼ȥǥ쥯ȥȰѥǥ쥯ȥ
     * ѿ¸Ƥ.
     */
    sl::java::lang::Object o;

    o = getServletContext().getAttribute("javax.servlet.context.tempdir");
    if (o) {
        try {
            _tempdir_path = sl::object_cast<string>(o);
        } catch(...) {
            throw ServletException("tempdir is not found");
        }
    } else
        throw ServletException("servlet.context.tempdir is not found");

    o = getServletContext().getAttribute("javax.servlet.context.path");
    if (o) {
        try {
            _context_path = sl::object_cast<string>(o);
        } catch(bad_cast &e) {
            throw ServletException("servlet.context.path is not found");
        }
    } else
        throw ServletException("context.path is not found");

    _context_name = _context_path.substr(_context_path.rfind("/"));

    si::interface::Config config;
    try {
        using namespace sl;

        xml_parser xml(file);
        xml_tag compile_tag = xml.get();

        _CC = compile_tag.child("compiler").attr("name");

        tag_list_t tags = compile_tag.get("options.value");
        for (tag_list_t::iterator i = tags.begin(); i != tags.end(); i++)
            _options.push_back(i->value());

        tags = compile_tag.get("include.path");
        for (tag_list_t::iterator i = tags.begin(); i != tags.end(); i++)
            _include.push_back(i->value());

        tags = compile_tag.get("library.path");
        for (tag_list_t::iterator i = tags.begin(); i != tags.end(); i++)
            _library.push_back(i->value());

        tags = compile_tag.get("library.module");
        for (tag_list_t::iterator i = tags.begin(); i != tags.end(); i++)
            _shared.push_back(i->value());

    } catch (sl::xml_parser_error &e) {
        std::cerr << e.what() << std::endl;
    }

    _include.push_back(_context_path + "/WEB-INF/classes");

}


void JspServlet::destroy() throw()
{
}


void JspServlet::service(HttpServletRequest &req, HttpServletResponse &res)
    throw(ServletException)
{
    if (_debug > 0)
        getServletContext().log("JspServlet::service");

    std::string jsp_path;
    std::string jsp_file;

    /* include.request_uri °̵ͭ include() ͳݤȽꤹ */
    sl::java::lang::Object obj;
    if ((obj = req.getAttribute("javax.servlet.include.request_uri"))) {
        try {
            std::string URI = sl::object_cast<std::string>(obj);
            std::string servlet_path = sl::object_cast<std::string>(
                             req.getAttribute("javax.servlet.include.servlet_path"));
            std::string path_info = sl::object_cast<std::string>(
                             req.getAttribute("javax.servlet.include.path_info"));

            jsp_path = getServletContext().getRealPath(servlet_path + path_info);
            jsp_file = path_info;

        } catch(std::bad_cast &e) {
            throw ServletException("including, invalid attribute");
        }
    } else {
       jsp_path = req.getPathTranslated();
       jsp_file = req.getPathInfo();
    }

    /* JSP ե뤫оݤȤʤServlet Υ饹̾ */
    std::string class_name;
    std::string namespace_dir;
    string::size_type pos;
    if ((pos = jsp_file.rfind("/")) != std::string::npos) {
        class_name = sj::jsp_to_class(jsp_file.substr(pos + 1));
        namespace_dir  = jsp_file.substr(0, pos);
    } else {
        class_name = sj::jsp_to_class(jsp_file);
    }

    if (namespace_dir[0] == '/')
        namespace_dir.erase(0, 1);

    /*
     * jsp_file  PathInfo Υѥޤ .jsp ե̾
     * jsp_path  롼ȥǥ쥯ȥ꤫Υѥޤ .jsp ե̾
     * class_name ϥǥ쥯ȥѥޤޤʤ .jsp ե̾
     * .jsp ե̾Υ饹ѤѴʸ
     * namespace_dir  PathInfo  .jsp ե̾ǥ쥯ȥ
     *
     * http://xxxx.com/${context}/path/to/file.jsp
     * jsp_file  : /path/to/file.jsp
     * jsp_path  : ${home}/${context}/path/to/file.jsp
     * class_name: __si_jsp_file_period_jsp  (̾Ѵ)
     * namespace : path/to ('/''::'Ѵƥ͡ॹڡȤƻ)
     */

    /* ե뤬̵ 404 Not Found */
    if (!filesystem::exists(jsp_path)) {
        res.sendError(HttpServletResponse::SC_NOT_FOUND);
        return;
    }
    time_t last = last_write_time(jsp_path);

    std::string class_path;
    if (!namespace_dir.empty())
        class_path = namespace_dir + "/" + class_name;
    else
        class_path = class_name;

    boost::shared_ptr<si::interface::ServletWrapper> p;
    {
        recursive_mutex::scoped_lock lock(_mutex);
        p = _jsp_context->getWrapper(class_path);
        if (!p) {
            recursive_mutex::scoped_lock lock(_mutex);

            doJspCompile(req, res, jsp_file, namespace_dir, class_name);

            p = _jsp_context->getWrapper(class_path);
            if (!p) {
                throw servlet::ServletException("JspServlet::service failed. "
                                                "can't loaded " + class_path);
            }
        }
    }

    if (p->invokeTime() < last) {
        doUnload(p, namespace_dir, class_name);
        doJspCompile(req, res, jsp_path, namespace_dir, class_name);
        p = _jsp_context->getWrapper(class_path);
    }

    if (!p)
        throw ServletException("JspServlet Instance is not found.");

    try {
        servlet::Servlet* s = p->instance();
        s->service(req, res);
    } catch (std::runtime_error& e) {
        throw ServletException(e.what());
    }
}


void JspServlet::doJspCompile(HttpServletRequest &req,
                              HttpServletResponse &res,
                              const string &jsp_file,
                              const string &namespace_dir,
                              const string &class_name)
    throw(ServletException)
{
    if (_debug > 0)
        getServletContext().log("JspServlet::doJspCompile");

    string output = _tempdir_path + "/" + namespace_dir;
    si::webxml::jsp_config jsp_conf = _context->getJspConfig();

    map<string, string> taguri = jsp_conf.get();
    set<string> dep_libs;

    if (_debug > 0) {
    getServletContext().log("*** JspServlet::doJspCompile");
    getServletContext().log("jsp_file      :" + jsp_file);
    getServletContext().log("namespace_dir :" + namespace_dir);
    getServletContext().log("class_name    :" + class_name);
    getServletContext().log("_context_path :" + _context_path);
    getServletContext().log("output:" + output);
    }

    try {
        sj::jsp_compiler j2cc;
        j2cc.run(_jsp_context->home(),
                _context_path,
                jsp_file,
                output,
                taguri,
                _CC,
                _options,
                _include,
                _library,
                _shared,
                dep_libs,
                _debug);

        log("sj::JspServlet::doJspCompile :Load depend classes");

        set<string>::iterator j = dep_libs.begin();
        for (; j != dep_libs.end(); j++) {
            try {
                _jsp_loader->loadClass(*j);
                log("Load \"" + *j + "\" :Successed");
            } catch (std::exception& e) {
                log("Loading failed.", sl::java::lang::Exception(e.what()));
            }
        }
    } catch (std::exception& e) {
        throw ServletException(sj::encode_xml(e.what()));
    }
    
    boost::shared_ptr<ServletWrapper> p(new JspWrapper());
    try {
        p->home(_jsp_context->home());
        p->setParent(_jsp_context);
        p->setManager(_jsp_context->getManager());
        p->setLogger(_jsp_context->getLogger());
        p->setLoader(_jsp_context->getLoader());
        p->setServletContext(_jsp_context->getServletContext());
        p->servletClassPath(_tempdir_path);

                std::string class_path;
                if (!namespace_dir.empty())
                    class_path = namespace_dir + "/" + class_name;
                else
                    class_path = class_name;
        p->servletClassName(class_path);

        p->init();

    } catch (std::runtime_error& e) {
        throw ServletException(e.what());
    }

    _jsp_context->setWrapper(p);
}


void JspServlet::doUnload(boost::shared_ptr<ServletWrapper>& p,
                        const string &namespace_dir,
                        const string &class_name)
{
    if (_debug > 0)
        getServletContext().log("JspServlet::doUnload");

    _jsp_context->removeWrapper(namespace_dir + "/" + class_name);
    p->destroy();
    p = boost::shared_ptr<ServletWrapper>();
}
