/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <boost/regex.hpp>
using namespace std;
using namespace boost;

#include <sl/java/lang/Object.h>
#include <sl/java/lang/String.h>

#include <si/shibainu.h>
#include <si/interface/Config.h>
#include <si/interface/Connection.h>
#include <si/interface/Loader.h>
#include <si/interface/Logger.h>
#include <si/interface/Manager.h>
#include <si/interface/ServletWrapper.h>
#include <si/core/ServletContextImpl.h>
using namespace si::interface;
using namespace si::core;

#include "JspContext.h"
#include "JspLoader.h"
#include "JspLogger.h"
using namespace sj;


const string WEB_INF_CLASSES = "/WEB-INF/classes/";
const string WEB_INF_LIB     = "/WEB-INF/lib/";
const string WEB_INF_WEB_XML = "/WEB-INF/web.xml";

const string DEFAULT_WEB_XML = "conf/web.xml";
const string DEFAULT_CLASSES = "./";

//
// Constructor/Destructor.
//

JspContext::JspContext() : _available(false)
{ }


//
// Member functions.
//

bool JspContext::operator!() const
{
    return _available;
}

void JspContext::config(const interface::Config& c)
{
    _config = c;
}

Config JspContext::config() const
{
    return _config;
}

void JspContext::name(const string& name)
{
    _name = name;
}

string JspContext::className() const
{
    return _class_name;
}

void JspContext::className(const string& name)
{
    _class_name = name;
}

string JspContext::name() const
{
    return _name;
}

void JspContext::home(const string& home)
{
    _home = home;
}

string JspContext::home() const
{
    return _home;
}

void JspContext::path(const string &path)
{
    if (path.empty()) {
        cerr << "runtime_error :Path is empty" << endl;
        throw runtime_error("Path is empty");
    }

    _path = path;
}

string JspContext::path() const
{
    return _path;
}

void JspContext::docBase(const string &doc_base)
{
    if (doc_base.empty())

    _doc_base = doc_base;
}


string JspContext::docBase() const
{
    return _doc_base;
}

string JspContext::getWorkDir() const
{
    string work_dir = "work/localhost" + _path;
    return work_dir;
}

void JspContext::setWorkDir(const string &dir)
{

}

bool JspContext::match(const string &uri) const
{
    using namespace boost;

    string compare_path = _path;

    // ̤Ǥϥޥåʤ
    if (compare_path.empty())
        return false;

    if (compare_path[0] == '/')
        compare_path = "^" + compare_path;

    if (compare_path[0] == '*') {
        if (compare_path.length() >= 2 || compare_path[1] == '.') {
            compare_path.erase(0, 2);
        compare_path = ".*\\." + compare_path;
        } else
            compare_path = "." + compare_path;
    }

    if (compare_path[compare_path.length() - 1] != '*')
        compare_path.append("$");

    try {
        regex reg(compare_path);
        if (regex_search(uri, reg))
            return true;

    } catch(std::exception &e) {
        cerr << e.what() << endl;
    }

    return false;
}

 
void JspContext::setParent(Base *)
{ }


Base* JspContext::getParent() const
{
    return 0;
}


void JspContext::setManager(const boost::shared_ptr<Manager>& manager)
{
    _manager = manager;
}


boost::shared_ptr<Manager> JspContext::getManager() const
{
    return _manager;
}


void JspContext::setLogger(const boost::shared_ptr<Logger>& logger)
{
    _logger = logger;
}


boost::shared_ptr<Logger> JspContext::getLogger() const
{
    return _logger;
}


void JspContext::setLoader(const boost::shared_ptr<Loader>& loader)
{
    _loader = loader;
}


boost::shared_ptr<Loader> JspContext::getLoader() const
{
    return _loader;
}

boost::shared_ptr<interface::Logger>
JspContext::loadLogger(const interface::Config&)
{
    return boost::shared_ptr<interface::Logger>();
}

void JspContext::setWrapper(boost::shared_ptr<ServletWrapper> wrapper)
{
    _wrapper_list.push_back(wrapper);
}


boost::shared_ptr<ServletWrapper>
JspContext::getWrapper(const string &path) const
{
    wrapper_list_t::const_iterator i = _wrapper_list.begin();
    for (; i != _wrapper_list.end(); i++) {
        if ((*i)->match(path))
            return *i;
    }

    // ĤʤϥǥեȤ Servlet õ.
    i = _wrapper_list.begin();
    for (; i != _wrapper_list.end(); i++) {
        if ((*i)->match("/"))
            return *i;
    }

    return boost::shared_ptr<ServletWrapper>();
}


boost::shared_ptr<ServletWrapper>
JspContext::getNamedWrapper(const std::string& /* name */) const
{
    /* JspServlet ¦ǤפʤΤ㳰 */
    throw std::runtime_error("JspContext::getNamedWrapper is not implement.");
}


void JspContext::removeWrapper(const string &path)
{
    wrapper_list_t::iterator i = _wrapper_list.begin();
    for (; i != _wrapper_list.end(); i++) {
        if ((*i)->match(path)) {
            _wrapper_list.erase(i);
            return;
        }
    }
}

void JspContext::setRealm(const shared_ptr<Realm>& realm)
{
    _realm = realm;
}


shared_ptr<Realm> JspContext::getRealm() const
{
    return _realm;
}


servlet::ServletContext& JspContext::getServletContext()
{
    return *_servlet_context;
}

time_t JspContext::invokeTime()
{
    return 0;
}

void JspContext::init()
{
    if (_available == true)
        return;

    if (!_loader)
        _loader = boost::shared_ptr<Loader>(new JspLoader());

    if (!_logger)
        _logger = boost::shared_ptr<Logger>(new JspLogger());

    if (!_manager) {
        //  config ϤƥǥեȤ Manager 󥹥󥹤.
        loadManager(config());
    }

    //
    // ServletContext 󥹥󥹤 
    //
    using namespace sl::java::lang;

    ServletContextImpl *sc = new ServletContextImpl(this, _home, _doc_base);
    sc->setAttribute("javax.servlet.context.tempdir",
                      Object(new String(this->getWorkDir())));
    sc->setAttribute("javax.servlet.context.path",
                      Object(new String(this->path())));
    sc->setServerInfo(si::info());

    _servlet_context = sc;

    _available = true;
}


void JspContext::destroy()
{
#ifdef DEBUG
    cout << "JspContext::destory" << endl;
#endif
    if (_available == false)
        return;

    _available = false;
}


void JspContext::service(Connection*)
{ }

void JspContext::loadAllWrapper()
{
std::cerr << "AAAAAAAAAA" << std::endl;
    // <servlet> (servlet-name  servlet-class  map ).
#if 0
    interface::Config servlet_config = _app_config->servlet();
    interface::Config mapping_config = _app_config->servlet_mapping();
    interface::Config mime_config    = _app_config->mime_mapping();

    // <servlet>  <servlet-name> Υꥹ.
    vector<string> name_list = servlet_config.attr_names();
    vector<string>::iterator i = name_list.begin();
    for (; i != name_list.end(); i++) {
        boost::shared_ptr<ServletWrapper> p = loadWrapper();
        if (p) {
            p->name(*i);
            p->servletClassName(servlet_config.attr(*i));
            p->servletClassPath(_doc_base + WEB_INF_CLASSES);
            p->addMapping(mapping_config.attr(*i));

            _wrapper_list.push_back(p);
        }
    }
#endif
}


boost::shared_ptr<ServletWrapper> JspContext::loadWrapper()
{
    string class_file = "modules/lib/DefaultWrapper";
    string class_name = "DefaultWrapper";

    sl::java::lang::Class clazz = _loader->loadClass(class_file);
    return clazz.newInstance<ServletWrapper>();
}

void JspContext::loadManager(const interface::Config& c)
{
    string class_file = c.attr("classFile");
    string class_name = c.attr("className");

    if (class_file.empty() || class_name.empty()) {
        class_file = "modules/lib/DefaultManager";
        class_name = "DefaultManager";
    }

    try {
        sl::java::lang::Class clazz = _loader->loadClass(class_file);
        _manager = clazz.newInstance<Manager>();
    } catch (sl::java::lang::ClassNotFoundException) { }

    setManager(_manager);
}
