/*-
 * Copyright (c) 2005 osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <cctype>
#include <cmath>
#include <cstdlib>
#include <iostream>
#include <string>
#include <stdexcept>
#include <boost/algorithm/string/predicate.hpp>
#include <boost/algorithm/string/compare.hpp>
#include <boost/algorithm/string/classification.hpp>
#include <boost/algorithm/string/trim.hpp>
#include <boost/bind.hpp>
#include <boost/lexical_cast.hpp>
using namespace std;
using namespace boost;

#include <sl/java/lang/Object.h>
#include <sl/java/lang/String.h>
#include <sl/java/lang/Integer.h>
#include <sl/object.hpp>

#include "operators.h"
using namespace sj::el;


bool is_digit(const string& s)
{
    const char* nptr = s.c_str();
    char *endptr;
    double ret = strtod(nptr, &endptr);
    if (ret == HUGE_VAL || ret == -HUGE_VAL)
        return false;
    if (ret == 0 && nptr == endptr)
        return false;
    return true;
}

string remove_quate(const string& s)
{
    return string();
}


string invalid_expr(int line)
{
    return string("Invalid EL Expression. line at ") +
                  boost::lexical_cast<string>(line);
}

string invalid_expr(int line, const string& cause)
{
    return string("Invalid EL Expression. line at ") +
                  boost::lexical_cast<string>(line) +
                  ". exception: " + cause;
}

string invalid_expr(int line, const std::exception& e)
{
    return string("Invalid EL Expression. line at ") +
                  boost::lexical_cast<string>(line) +
                  ". exception: " + e.what();
}


op::add::add(operators* parent)
    : _parent(parent)
{ }

void op::add::operator()(value_type& lhs, const value_type& rhs) const
{
#ifdef DEBUG
    cerr << "sj::el::op::add " << __LINE__ << endl;
#endif
    if (!lhs) {
        lhs = rhs;
        return;
    }

    string& l = sl::object_cast<std::string>(lhs);
#ifdef DEBUG
    cerr << "lhs:" << l << std::endl;
#endif
    try {
        string r = sl::object_cast<std::string>(rhs);
#ifdef DEBUG
    cerr << "rhs:" << r << std::endl;
#endif
        if (is_digit(l) && is_digit(r)) {
            double result = boost::lexical_cast<double>(l) +
                            boost::lexical_cast<double>(r);
            string s = boost::lexical_cast<string>(result);
            lhs = s;
        } else {
            l += r;
        }
        return;

    } catch(std::exception& e) {
        cerr << "sj::el::op::add :" << invalid_expr(__LINE__, e) << endl;
    }

    throw runtime_error(invalid_expr(__LINE__));
}

void op::add::operator()(value_type& lhs, const bool rhs) const
{
#ifdef DEBUG
    cerr << "sj::el::op::add " << __LINE__ << endl;
#endif
    if (!lhs) {
        lhs = 1;
        return;
    }

    throw runtime_error(invalid_expr(__LINE__));
}

void op::add::operator()(value_type& lhs, const string::iterator& first,
                                          const string::iterator& last) const
{
    string s(first, last);
    boost::trim_if(s, boost::is_any_of("\""));
#ifdef DEBUG
    cerr << "sj::el::op::add " << __LINE__ << endl;
    cerr << s << endl;
#endif

    if (!_parent->_func_stack.empty()) {
        stack_e& e = _parent->_func_stack.top();
        e.second.push_back(s);
        return;
    }

    if (!lhs) {
        lhs = s;
        return;
    }

    try {
        string& l = sl::object_cast<std::string>(lhs);
        l += s;
        return;
    } catch(const std::exception& e) {
        cerr << "sj::el::op::add :" << invalid_expr(__LINE__, e) << endl;
    }

    throw runtime_error(invalid_expr(__LINE__));
}

void op::add::operator()(value_type& lhs, char rhs) const
{
#ifdef DEBUG
    cerr << "sj::el::op::add " << __LINE__ << endl;
    cerr << rhs << endl;
#endif
    if (!lhs) {
        lhs = string(rhs, 1);
        return;
    }

    try {
        string& l = sl::object_cast<std::string>(lhs);
        l += string(rhs, 1);
        return;
    } catch(const std::exception& e) {
        cerr << "sj::el::op::add :" << invalid_expr(__LINE__, e) << endl;
    }

    throw runtime_error(invalid_expr(__LINE__));
}

void op::add::operator()(value_type& lhs, double rhs) const
{
#ifdef DEBUG
    cerr << "sj::el::op::add " << __LINE__ << endl;
#endif
    if (!lhs) {
        lhs = boost::lexical_cast<string>(rhs);
        return;
    }

    try {
        string& l = sl::object_cast<std::string>(lhs);

        if (is_digit(l)) {
            double r = boost::lexical_cast<double>(l) + rhs;
            string s = boost::lexical_cast<string>(r);
            lhs = s;
        } else {
            string s = boost::lexical_cast<string>(rhs);
            l += s;
        }

    } catch(std::exception& e) {
        cerr << "sj::el::op::add :" << invalid_expr(__LINE__, e) << endl;
        throw runtime_error(invalid_expr(__LINE__));
    }
}


///////////////////////////////////////////////////////////////////////////////
// op::sub::operator()
//
void op::sub::operator()(value_type& lhs, const value_type& rhs) const
{
#ifdef DEBUG
    cerr << "sj::el::op::sub " << __LINE__ << endl;
#endif
    if (!lhs || !rhs)
        throw invalid_argument("Invalid Argument in Operator Sub.");

    try {
        string& l = sl::object_cast<std::string>(lhs);
        string r = sl::object_cast<std::string>(rhs);
        if (is_digit(l) && is_digit(r)) {
            double result = boost::lexical_cast<double>(l) -
                            boost::lexical_cast<double>(r);
            string s = boost::lexical_cast<string>(result);
            lhs = s;
        }
    } catch(std::exception& e) {
        cerr << "sj::el::op::sub :" << invalid_expr(__LINE__, e) << endl;
        throw runtime_error(invalid_expr(__LINE__));
    }
}


///////////////////////////////////////////////////////////////////////////////
// op::mul::operator()
//
void op::mul::operator()(value_type& lhs, const value_type& rhs) const
{
#ifdef DEBUG
    cerr << "sj::el::op::mul " << __LINE__ << endl;
#endif
    if (!lhs || !rhs)
        throw invalid_argument("Invalid Argument in Operator Sub.");

    try {
        string& l = sl::object_cast<string>(lhs);
        string r = sl::object_cast<string>(rhs);
        if (is_digit(l) && is_digit(r)) {
            double result = boost::lexical_cast<double>(l) *
                            boost::lexical_cast<double>(r);
            string s = boost::lexical_cast<string>(result);
            lhs = s;
        }
    } catch(std::exception& e) {
        cerr << "sj::el::op::mul :" << invalid_expr(__LINE__, e) << endl;
        throw runtime_error(invalid_expr(__LINE__));
    }
}


///////////////////////////////////////////////////////////////////////////////
// op::div::operator()
//
void op::div::operator()(value_type& lhs, const value_type& rhs) const
{
#ifdef DEBUG
    cerr << "sj::el::op::div " << __LINE__ << endl;
#endif
    if (!lhs || !rhs)
        throw invalid_argument("Invalid Argument in Operator Sub.");

    try {
        string& l = sl::object_cast<string>(lhs);
        string r = sl::object_cast<string>(rhs);
        if (is_digit(l) && is_digit(r)) {
            double r_value = boost::lexical_cast<double>(r);

            if (r_value != 0) {
                double result = boost::lexical_cast<double>(l) /
                                boost::lexical_cast<double>(r_value);
                string s = boost::lexical_cast<string>(result);
                lhs = s;
            } else {
                lhs = "infinity";
            }
        }
    } catch(std::exception& e) {
        cerr << "sj::el::op::div :" << invalid_expr(__LINE__, e) << endl;
        throw runtime_error(invalid_expr(__LINE__));
    }
}


///////////////////////////////////////////////////////////////////////////////
// op::mod::operator()
//
void op::mod::operator()(value_type& lhs, const value_type& rhs) const
{
#ifdef DEBUG
    cerr << "sj::el::op::mod " << __LINE__ << endl;
#endif
    if (!lhs || !rhs)
        throw invalid_argument("Invalid Argument in Operator Sub.");

    try {
        string& l = sl::object_cast<string>(lhs);
        string r = sl::object_cast<string>(rhs);
        if (is_digit(l) && is_digit(r)) {
            int r_value = boost::lexical_cast<int>(r);

            if (r_value != 0) {
                int result = boost::lexical_cast<int>(l) % r_value;
                string s = boost::lexical_cast<string>(result);
                lhs = s;
            } else {
                lhs = "infinity";
            }
        }
    } catch(std::exception& e) {
        cerr << "sj::el::op::mod :" << invalid_expr(__LINE__, e) << endl;
        throw runtime_error(invalid_expr(__LINE__));
    }
}
 

///////////////////////////////////////////////////////////////////////////////
// op::equal::operator()
//
void op::equal::operator()(value_type& lhs, const value_type& rhs) const
{
#ifdef DEBUG
    cerr << "sj::el::op::equal " << __LINE__ << endl;
#endif

    try {
        string& l = sl::object_cast<string>(lhs);
        string r = sl::object_cast<string>(rhs);
        l = (l == r) ? "1" : "0";
        return;
    } catch(...) { }

    try {
        int& l = sl::object_cast<int>(lhs);
        int r = sl::object_cast<int>(rhs);
        l = (l == r) ? 1 : 0;
        return;
    } catch(const std::exception& e) {
        cerr << "sj::el::op::equal :" << invalid_expr(__LINE__, e) << endl;
    }

    throw runtime_error(invalid_expr(__LINE__));
}


///////////////////////////////////////////////////////////////////////////////
// op::equal::operator()
//
void op::equal::operator()(string& lhs, const string::iterator& first,
                                     const string::iterator& last) const
{
#ifdef DEBUG
    cerr << "sj::el::op::equal " << __LINE__ << endl;
#endif
}

///////////////////////////////////////////////////////////////////////////////
// op::not_equal::operator()
//
void op::not_equal::operator()(value_type& lhs, const value_type& rhs) const
{
#ifdef DEBUG
    cerr << "sj::el::op::not_equal " << __LINE__ << endl;
#endif
    try {
        int& l = sl::object_cast<int>(lhs);
        int r = sl::object_cast<int>(rhs);
        l = (l != r) ? 1 : 0;
        return;
    } catch(const std::exception& e) {
        cerr << "sj::el::op::not_equal :" << invalid_expr(__LINE__, e) << endl;
    }

    try {
        string& l = sl::object_cast<string>(lhs);
        string r = sl::object_cast<string>(rhs);
        l = (l != r) ? "1" : "0";
        return;
    } catch(const std::exception& e) {
        cerr << "sj::el::op::not_equal :" << invalid_expr(__LINE__, e) << endl;
    }

    throw runtime_error(invalid_expr(__LINE__));
}

///////////////////////////////////////////////////////////////////////////////
// op::less_equal::operator()
//
void op::less_equal::operator()(value_type& lhs, const value_type& rhs) const
{
#ifdef DEBUG
    cerr << "sj::el::op::less_equal " << __LINE__ << endl;
#endif
    try {
        int& l = sl::object_cast<int>(lhs);
        int r = sl::object_cast<int>(rhs);
        l = (l <= r) ? 1 : 0;
        return;
    } catch(const std::exception& e) {
        cerr << "sj::el::op::less_equal :" << invalid_expr(__LINE__, e) << endl;
    }

    try {
        string& l = sl::object_cast<string>(lhs);
        string r = sl::object_cast<string>(rhs);
        l = (l <= r) ? "1" : "0";
        return;
    } catch(const std::exception& e) {
        cerr << "sj::el::op::less_equal :" << invalid_expr(__LINE__, e) << endl;
    }

    throw runtime_error(invalid_expr(__LINE__));
}

///////////////////////////////////////////////////////////////////////////////
// op::greater_equal::operator()
//
void op::greater_equal::operator()(value_type& lhs, const value_type& rhs) const
{
#ifdef DEBUG
    cerr << "sj::el::op::greator_equal " << __LINE__ << endl;
#endif
    try {
        int& l = sl::object_cast<int>(lhs);
        int r = sl::object_cast<int>(rhs);
        l = (l >= r) ? 1 : 0;
        return;
    } catch(const std::exception& e) {
        cerr << "sj::el::op::gre_equal :" << invalid_expr(__LINE__, e) << endl;
    }

    try {
        string& l = sl::object_cast<string>(lhs);
        string r = sl::object_cast<string>(rhs);
        l = (l >= r) ? "1" : "0";
        return;
    } catch(const std::exception& e) {
        cerr << "sj::el::op::gre_equal :" << invalid_expr(__LINE__, e) << endl;
    }

    throw runtime_error(invalid_expr(__LINE__));
}

///////////////////////////////////////////////////////////////////////////////
// op::less::operator()
//
void op::less::operator()(value_type& lhs, const value_type& rhs) const
{
#ifdef DEBUG
    cerr << "sj::el::op::less " << __LINE__ << endl;
#endif
    if (lhs.instanceof<int>() && rhs.instanceof<int>()) {
        int& l = sl::object_cast<int>(lhs);
        int r = sl::object_cast<int>(rhs);
        l = (l < r) ? 1 : 0;
        return;
    }

    if (lhs.instanceof<std::string>() && rhs.instanceof<std::string>()) {
        string& l = sl::object_cast<string>(lhs);
        string r = sl::object_cast<string>(rhs);
        l = (l < r) ? "1" : "0";
        return;
    }

    throw runtime_error(invalid_expr(__LINE__));
}

///////////////////////////////////////////////////////////////////////////////
// op::greater::operator()
//
void op::greater::operator()(value_type& lhs, const value_type& rhs) const
{
#ifdef DEBUG
    cerr << "sj::el::op::greator " << __LINE__ << endl;
#endif
    if (lhs.instanceof<int>() && rhs.instanceof<int>()) {
        int& l = sl::object_cast<int>(lhs);
        int r = sl::object_cast<int>(rhs);
        l = (l > r) ? 1 : 0;
        return;
    }

    if (lhs.instanceof<std::string>() && rhs.instanceof<std::string>()) {
        string& l = sl::object_cast<string>(lhs);
        string r = sl::object_cast<string>(rhs);
        l = (l > r) ? "1" : "0";
        return;
    }

    throw runtime_error(invalid_expr(__LINE__));
}

///////////////////////////////////////////////////////////////////////////////
// op::symbol::operator()
//
op::symbol::symbol(operators* parent, servlet::jsp::PageContext& v)
    : _parent(parent), _context(v)
{ }

class icomp : public unary_function<bool, map<string, string>::value_type> {
    std::string         _s;
public :
    icomp(const std::string s) : _s(s) { }

    bool operator()(const map<string, string>::value_type& l) const
    {
        return boost::equals(l.first, _s, is_iequal());
    }
};


void op::symbol::operator()(value_type& lhs, const string::iterator& first,
                                         const string::iterator& last) const
{
    string s(first, last);
#ifdef DEBUG
    cerr << "sj::el::op::symbol " << __LINE__ << " " << s << endl;
#endif
    string name, value;
    bool is_container = false;

    string::size_type pos1 = s.find("[");
    string::size_type pos2 = s.find("]", pos1);
    if (pos1 != string::npos && pos2 != string::npos) {
        name  = s.substr(0, pos1);
        value = s.substr(pos1 + 1, pos2 - pos1 - 1);
        trim_if(value, is_any_of("\""));
        is_container = true;
    } else {
        name = s;
    }

#ifdef DEBUG
    cerr << "name :" << name << endl;
    cerr << "index:" << value << endl;
#endif

    /*
     * ֥Ȥ¸ߤʤ "" ʸ
     */
    lhs = _context.getAttribute(name);
    if (!lhs) {
        lhs = sl::java::lang::Object("");

        if (!_parent->_func_stack.empty())
            _parent->_func_stack.top().second.push_back(lhs);

        return;
    }

    /*
     * ź '[]'  '.' ̵ñǻǤФǽλ.
     *
     * 
     * ΤȤEL Ǥα黻Ʒ̤ʸȤݻƤ뤿ᡢ
     *  object  EL Function ΰʳǻȤϡ
     *  object ͤĴ١ͤǤʸ󷿤ѹƤ.
     *
     * EL Function ΰȤƻѤ
     * EL Function μ¹Ի˳ؿΥѥ᡼η˥㥹Ȥ뤿ᡢ
     * Ǥϲ⤹ɬפϤʤ.
     *
     */
    if (!is_container) {
        if (!_parent->_func_stack.empty())
            _parent->_func_stack.top().second.push_back(lhs);

        try {
            int l = sl::object_cast<int>(lhs);
            lhs = sl::java::lang::Object(new string(
                                        boost::lexical_cast<string>(l)));
        } catch (...) {
            /* 㥹Ȥ˼ԤͤǤ̵Ჿ⤷ʤ */
        }

        return;
    }

    typedef map<string, string> str_map_t;
    typedef vector<string>        str_seq_t;

    /**
     * ν̲᤹ȤȤź '[]'  '.' ꤵƤ
     * ϤʤΤǡʥƥʷ˥㥹ȤƤ 
     * ˥㥹Ȥ֥Ȥ̤ lhs ꤹ. 
     */ 
    if (lhs.instanceof<str_map_t>()) {
        str_map_t m = sl::object_cast<str_map_t>(lhs);
        str_map_t::iterator i = find_if(m.begin(), m.end(), icomp(value));
        if (i != m.end())
            lhs = i->second;
        else
            lhs = "";

        if (!_parent->_func_stack.empty())
            _parent->_func_stack.top().second.push_back(lhs);
        return;
    }

    if (lhs.instanceof<std::vector<std::string> >()) {
        str_seq_t m = sl::object_cast<vector<string> >(lhs);
        string ret = m[boost::lexical_cast<int>(value)];
        lhs = ret;

        if (!_parent->_func_stack.empty())
            _parent->_func_stack.top().second.push_back(lhs);
        return;
    }

    lhs = "";
    if (!_parent->_func_stack.empty())
        _parent->_func_stack.top().second.push_back(lhs);
}


///////////////////////////////////////////////////////////////////////////////
// op::func::operator()
//
op::func::func(operators* parent, servlet::jsp::PageContext& v)
    : _parent(parent), _context(v)
{ }

void op::func::operator()(value_type& lhs, const value_type& rhs) const
{
#ifdef DEBUG
    cerr << "sj::el::op::func " << __LINE__ << endl;
#endif
    lhs = rhs;
}

void op::func::operator()(value_type& lhs, const char rhs) const
{
#ifdef DEBUG
    cerr << "sj::el::op::func " << __LINE__ << endl;
#endif
    try {
        string& l = sl::object_cast<std::string>(lhs);
        l.append(rhs, 1);
        return;
    } catch(const std::exception& e) {
        cerr << "sj::el::op::func :" << invalid_expr(__LINE__, e) << endl;
    }
    throw runtime_error(invalid_expr(__LINE__));
}

void op::func::operator()(value_type& lhs, const string::iterator& first,
                                       const string::iterator& last) const
{
    stack_e e(string(first, last), vector<sl::java::lang::Object>());
    _parent->_func_stack.push(e);
#ifdef DEBUG
    cerr << "sj::el::op::func " << __LINE__ << endl;
    cerr << string(first, last) << endl;
#endif
}


///////////////////////////////////////////////////////////////////////////////
// op::func_exec::operator()
//
op::func_exec::func_exec(operators* parent,
                    servlet::jsp::PageContext& v,
                    const servlet::jsp::el::FunctionMapper& mapper)
    : _parent(parent), _context(v), _mapper(mapper)
{ }

void op::func_exec::operator()(value_type& lhs, const value_type& rhs) const
{
#ifdef DEBUG
    cerr << "sj::el::op::func_exec " << __LINE__ << endl;
#endif

    if (_parent->_func_stack.empty())
        throw std::runtime_error("Functions stack is empty.");

    stack_e& e = _parent->_func_stack.top();
    string func = e.first;
    objects    objs= e.second;

    _parent->_func_stack.pop();

#ifdef DEBUG
    cerr << "func_exec:" << __LINE__ << endl;
    cerr << "__function:" << func << endl;
    cerr << "__objects :" << objs.size() << endl;
#endif
    try {
        std::string tmp = func;
        std::string::size_type pos;

        if ((pos = tmp.find(":")) == std::string::npos)
            throw std::runtime_error("Invalid Function");

        std::string prefix = tmp.substr(0, pos);
        std::string name   = tmp.substr(pos + 1);

#ifdef DEBUG
std::cerr << "prefix:" << prefix << std::endl;
std::cerr << "name  :" << name << std::endl;
#endif
        lhs = _mapper.resolveFunction(prefix, name, objs);

        if (!_parent->_func_stack.empty())
            _parent->_func_stack.top().second.push_back(lhs);

    } catch(std::exception& e) {
        cerr << "sj::el::op::func_exec :" << invalid_expr(__LINE__, e) << endl;
        throw runtime_error(invalid_expr(__LINE__, e.what()));
    }
}
