/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <iostream>
using namespace std;

#include <sl/object.hpp>
#include <sl/java/lang/Object.h>
#include <sl/java/util/Enumeration.h>
#include <servlet/ServletContext.h>
#include <servlet/RequestDispatcher.h>
#include <servlet/http/HttpServletRequest.h>
using namespace sl::java::lang;
using namespace sl::java::util;
using namespace servlet::http;

#include "BodyContentImpl.h"
#include "PageContextImpl.h"
#include "JspWriterImpl.h"
#include "ExpressionEvaluatorImpl.h"
#include "VariableResolverImpl.h"
using namespace servlet;
using namespace servlet::jsp;
using namespace sj;

PageContextImpl::PageContextImpl()
    : _is_initialize(false), _push_body_count(0)
{
    _servlet_context = 0;
    _servlet_config = 0;
    _servlet_request = 0;
    _servlet_response = 0;
    _session = 0;
    _writer = 0;
    _exception = 0;
    _expression_evaluator = new ExpressionEvaluatorImpl(*this);
    _variable_resolver    = new VariableResolverImpl(*this);
}

PageContextImpl::~PageContextImpl()
{
    delete _expression_evaluator;
    delete _variable_resolver;
}


//
// JspContext
//

Object PageContextImpl::findAttribute(const std::string& name)
{
    return getAttribute(name);
}

Object PageContextImpl::getAttribute(const std::string& name)
{
    sl::java::lang::Object obj;

    obj = getAttribute(name, PAGE_SCOPE);
    if (obj)
        return obj;

    obj = getAttribute(name, REQUEST_SCOPE);
    if (obj)
        return obj;

    if (_session) {
        obj = getAttribute(name, SESSION_SCOPE);
        if (obj)
            return obj;
    }

    obj = getAttribute(name, APPLICATION_SCOPE);
    if (obj)
        return obj;
    return obj;
}

Object PageContextImpl::getAttribute(const std::string& name, int scope)
{
    sl::java::lang::Object obj;

    if (scope == PAGE_SCOPE) {
        boost::recursive_mutex::scoped_lock lk(_mutex);
        std::map<std::string, sl::java::lang::Object>::iterator i =
                                                        _attributes.find(name);
        if (i != _attributes.end())
            return i->second;
        return sl::java::lang::Object();
    }

    switch (scope) {
    case REQUEST_SCOPE :
        if (_servlet_request)
            return _servlet_request->getAttribute(name);
         return Object();

    case SESSION_SCOPE :
        if (_session)
            return _servlet_context->getAttribute(name);
        return Object();

    case APPLICATION_SCOPE :
        if (_servlet_context)
            return _servlet_context->getAttribute(name);
        return Object();

    /* case PAGE_SCOPE : */
    default :
        throw std::runtime_error("Invalid Scope");
    }
    /* return obj; */
}

sl::java::util::Enumeration
PageContextImpl::getAttributeNamesInScope(int scope)
{
    std::vector<sl::java::lang::Object> objs;
    std::set<std::string>::const_iterator i;

    switch (scope) {
    case PAGE_SCOPE :
        for (i = _attr_names.begin(); i != _attr_names.end(); i++) {
            objs.push_back(sl::java::lang::Object(*i));
        }
        return sl::java::util::Enumeration(objs);

    case REQUEST_SCOPE :
        return _servlet_request->getAttributeNames();

    case SESSION_SCOPE :
        if (!_session)
            throw std::runtime_error("No Session");
        return _servlet_context->getAttributeNames();

    case APPLICATION_SCOPE :
        return _servlet_context->getAttributeNames();

    default :
        throw std::runtime_error("Invalid Scope");
    }
}

int PageContextImpl::getAttributesScope(const std::string& name)
{
    sl::java::lang::Object obj;

    if (getAttribute(name, PAGE_SCOPE))
        return PAGE_SCOPE;

    if (getAttribute(name, REQUEST_SCOPE))
        return REQUEST_SCOPE;

    if (_session) {
        if (getAttribute(name, SESSION_SCOPE))
            return SESSION_SCOPE;
    }

    if (getAttribute(name, APPLICATION_SCOPE))
        return APPLICATION_SCOPE;

    throw std::runtime_error("Not Found");
}

el::ExpressionEvaluator& PageContextImpl::getExpressionEvaluator()
{
    return *_expression_evaluator;
}

JspWriter& PageContextImpl::getOut()
{
    return *_writer;
}

el::VariableResolver& PageContextImpl::getVariableResolver()
{
    return *_variable_resolver;
}

JspWriter& PageContextImpl::popBody()
{
    return dynamic_cast<JspWriter&>(*_writer);
}

JspWriter& PageContextImpl::pushBody(const sl::java::io::Writer& writer)
{
#if 0
    _writer_map[_push_body_content_count] = writer;
    BodyCoutent* p = new BodyContentImpl(_writer);

    _body_c_map[_push_body_content_count] = p;
    ++_push_body_content_count;
#endif
    servlet::jsp::tagext::BodyContent* p = new BodyContentImpl(*_writer);
    return *p;
}

void PageContextImpl::removeAttribute(const std::string& name)
{
    removeAttribute(name, PAGE_SCOPE);
    removeAttribute(name, REQUEST_SCOPE);
    removeAttribute(name, APPLICATION_SCOPE);
    if (_session)
        removeAttribute(name, SESSION_SCOPE);
}

void PageContextImpl::removeAttribute(const std::string& name, int scope)
{
    switch (scope) {
    case PAGE_SCOPE :
        this->removeAttribute(name);
        break;

    case REQUEST_SCOPE :
        _servlet_request->removeAttribute(name);
        break;

    case SESSION_SCOPE :
        if (!_session)
            throw std::runtime_error("No Session");
        _servlet_context->removeAttribute(name);
        break;

    case APPLICATION_SCOPE :
        _servlet_context->removeAttribute(name);
        break;

    default :
        throw std::runtime_error("Invalid Scope");
    }
}

void PageContextImpl::setAttribute(const std::string& name,
                                   Object attribute)
{
    boost::recursive_mutex::scoped_lock lk(_mutex);
    _attr_names.insert(name);
    _attributes[name] = attribute; 
}

void PageContextImpl::setAttribute(const std::string& name,
                                   Object attribute, int scope)
{
    switch (scope) {
    case PAGE_SCOPE :
        this->setAttribute(name, attribute);
        break;

    case REQUEST_SCOPE :
        _servlet_request->setAttribute(name, attribute);
        break;

    case SESSION_SCOPE :
        if (!_session)
            throw std::runtime_error("No Session");
        _servlet_context->setAttribute(name, attribute);
        break;

    case APPLICATION_SCOPE :
        _servlet_context->setAttribute(name, attribute);
        break;

    default :
        throw std::runtime_error("Invalid Scope");
    }
}


//
// PageContext
//

void PageContextImpl::forward(const std::string& url_path)
{
    servlet::ServletContext& context = getServletContext();
    context.getRequestDispatcher(url_path)->forward(getRequest(), getResponse());
}

std::exception& PageContextImpl::getException()
{
    return *_exception;
}

Object PageContextImpl::getPage()
{
    return Object();
}

ServletRequest& PageContextImpl::getRequest()
{
    return *_servlet_request;
}

ServletResponse& PageContextImpl::getResponse()
{
    return *_servlet_response;
}

ServletConfig& PageContextImpl::getServletConfig()
{
    return *_servlet_config;
}

ServletContext& PageContextImpl::getServletContext()
{
    return *_servlet_context;
}

HttpSession& PageContextImpl::getSession()
{
    HttpServletRequest *r = dynamic_cast<HttpServletRequest*>(_servlet_request);
    return r->getSession();
}

void PageContextImpl::handlePageException(std::exception& e)
{
    _exception = &e;
}

void PageContextImpl::include(const std::string& url_path)
{
    servlet::ServletContext& context = getServletContext();
    context.getRequestDispatcher(url_path)->include(getRequest(), getResponse());
}

void PageContextImpl::initialize(Servlet& s,
                                ServletRequest& req,
                                ServletResponse& res,
                                const std::string& error_page_url,
                                bool needs_session,
                                int buffer_size,
                                bool auto_flush)
{
    _servlet_config  = &s.getServletConfig();
    _servlet_context = &_servlet_config->getServletContext();
    _servlet_response = &res;
    _servlet_request = &req;
    if (needs_session) {
        servlet::http::HttpServletRequest* req =
            dynamic_cast<servlet::http::HttpServletRequest*>(_servlet_request);
        _session = &req->getSession();
    } else
        _session = 0;

    _writer = new JspWriterImpl(&res.getWriter(), buffer_size, auto_flush);

    /*
     * EL ΰۤΥ֥ "header" 
     * PAGE_SCOPE ¸
     */
    HttpServletRequest* hreq = dynamic_cast<HttpServletRequest*>(&req);
    if (hreq) {
        map<string, string> header_map;
        Enumeration e = hreq->getHeaderNames();
        while (e.hasMoreElements()) {
            string name  = sl::object_cast<string>(e.nextElement());
            string value = hreq->getHeader(name);
            header_map.insert(make_pair(name, value));
        }

        setAttribute("header", header_map, PAGE_SCOPE);

        map<string, string> param_map;
        e = hreq->getParameterNames();
        while (e.hasMoreElements()) {
            string name  = sl::object_cast<string>(e.nextElement());
            string value = hreq->getParameter(name);
            param_map.insert(make_pair(name, value));
        }

        setAttribute("param", param_map, PAGE_SCOPE);
    }

    _is_initialize = true;
}


tagext::BodyContent& PageContextImpl::pushBody()
{
    return dynamic_cast<tagext::BodyContent&>(*_writer);
}

void PageContextImpl::release()
{
    if (_writer) {
        delete _writer;
        _writer = 0;
    }

    _is_initialize = false;
}
