/*-
 * Copyright (c) 2005 osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef SJ_JSP2C_UTIL_H
#define SJ_JSP2C_UTIL_H

#include <map>
#include <string>
#include <boost/algorithm/string/replace.hpp>
#include <boost/algorithm/string/erase.hpp>
#include <boost/lexical_cast.hpp>

namespace sj {

/** ư륯饹̾ѿ̾Υץեå. */
const std::string key_prefix = "__si_jsp_";

/**
 * @class ascii_string.
 *
 * JSP Υե̾饯饹̾뤿Υ饹.
 *
 * JSP ե̾Ѥƥ饹̾ˤä
 * 饹̾ȤȤʤʸ䵭ʸѴޤ.
 * 㤨ж `_space' Ѵꡢϥۥ `_hyphen' Ѵޤ.
 */
class ascii_string {
    typedef std::map<int, std::string> code_map_t;
    typedef code_map_t::value_type type;

public :
    static std::string code_to_string(int n)
    {
        code_map_t::const_iterator i = instance()._map.find(n);
        return i != instance()._map.end() ? i->second : "";
    }

    static int string_to_code(const std::string& s)
    {
        code_map_t::const_iterator i = instance()._map.begin();
        for (; i != instance()._map.end(); i++) {
            if (i->second == s)
                return i->first;
        }
        return 0xff;
    }

private :
    ascii_string()
    {
#if 1
        _map.insert(type(0x20, "_space_"));
        _map.insert(type(0x21, "_exclamation_mark_"));
        _map.insert(type(0x22, "_quotation_mark_"));
        _map.insert(type(0x23, "_number_sign_"));
        _map.insert(type(0x24, "_boller_sign_"));
        _map.insert(type(0x25, "_percent_sign_"));
        _map.insert(type(0x26, "_ampersand_"));
        _map.insert(type(0x27, "_apostrophe_"));
        _map.insert(type(0x28, "_left_parenthesis_"));
        _map.insert(type(0x29, "_right_parenthesis_"));
        _map.insert(type(0x2a, "_asterisk_"));
        _map.insert(type(0x2b, "_plus_sign_"));
        _map.insert(type(0x2c, "_comma_"));
        _map.insert(type(0x2d, "_hyphen_"));
        _map.insert(type(0x2e, "_period_"));
        _map.insert(type(0x2f, "_slash_"));
        _map.insert(type(0x3a, "_colon_"));
        _map.insert(type(0x3b, "_semicolon_"));
        _map.insert(type(0x3c, "_less_than_sign_"));
        _map.insert(type(0x3d, "_equals_sign_"));
        _map.insert(type(0x3e, "_greater_than_sign_"));
        _map.insert(type(0x3f, "_question_mark_"));
        _map.insert(type(0x40, "_at_sign_"));
        _map.insert(type(0x5b, "_left_square_bracket_"));
        _map.insert(type(0x5c, "_backslash_"));
        _map.insert(type(0x5d, "_right_square_bracket_"));
        _map.insert(type(0x5e, "_circumflex_accent_"));
        _map.insert(type(0x5f, "_underline_"));
        _map.insert(type(0x60, "_grave_accent_"));
        _map.insert(type(0x7b, "_left_curly_bracket_"));
        _map.insert(type(0x7c, "_vertical_line_"));
        _map.insert(type(0x7d, "_right_curly_bracket_"));
        _map.insert(type(0x7e, "_tilde_"));
#else
        _map.insert(type(0x20, "20"));
        _map.insert(type(0x21, "21"));
        _map.insert(type(0x22, "22"));
        _map.insert(type(0x23, "23"));
        _map.insert(type(0x24, "24"));
        _map.insert(type(0x25, "25"));
        _map.insert(type(0x26, "26"));
        _map.insert(type(0x27, "27"));
        _map.insert(type(0x28, "28"));
        _map.insert(type(0x29, "29"));
        _map.insert(type(0x2a, "2a"));
        _map.insert(type(0x2b, "2b"));
        _map.insert(type(0x2c, "2c"));
        _map.insert(type(0x2d, "_H"));
        _map.insert(type(0x2e, "_D"));
        _map.insert(type(0x2f, "_S"));
        _map.insert(type(0x3a, "3a"));
        _map.insert(type(0x3b, "3b"));
        _map.insert(type(0x3c, "3c"));
        _map.insert(type(0x3d, "3d"));
        _map.insert(type(0x3e, "3e"));
        _map.insert(type(0x3f, "3f"));
        _map.insert(type(0x40, "40"));
        _map.insert(type(0x5b, "5b"));
        _map.insert(type(0x5c, "5c"));
        _map.insert(type(0x5d, "5d"));
        _map.insert(type(0x5e, "5e"));
        _map.insert(type(0x5f, "_"));
        _map.insert(type(0x60, "60"));
        _map.insert(type(0x7b, "7b"));
        _map.insert(type(0x7c, "7c"));
        _map.insert(type(0x7d, "7d"));
        _map.insert(type(0x7e, "7e"));
#endif
    }

    static ascii_string& instance()
    {
        static ascii_string instance;
        return instance;
    }

    code_map_t    _map;
};

inline std::string class_to_jsp(const std::string& jsp_file)
{
    std::string s = jsp_file;

    if (s.length() < 2)
        return std::string();

    /* ƬΥץեå */
    s.erase(0, 2);
    return s;
}

inline std::string jsp_to_class(const std::string& jsp_file)
{
    std::string s = jsp_file;

    /* ƤʸɬפѴ */
    for (size_t i = 0; i < s.length(); i++) {
        if (isalnum(s[i]))
            continue;

        std::string c = ascii_string::code_to_string(s[i]);
        if (c.empty())
            c = boost::lexical_cast<std::string>(static_cast<int>(s[i]));

        s.replace(i, 1, c);
        i += c.length() - 1;
    }

    /* Ƭ˥ץեåղä */
    s.insert(0, sj::key_prefix);
    return s;
}

inline std::string quote_to_scalar(const std::string& line)
{
    std::string s(line);
    std::string::size_type pos;

    for (pos = 0; (pos = s.find('\t', pos)) != std::string::npos; pos += 2)
        s.replace(pos, 1, "\\t");

    for (pos = 0; (pos = s.find('\"', pos)) != std::string::npos; pos += 2)
        s.replace(pos, 1, "\\\"");

    /* EL  ${} 򥨥פƤ \${...} ȤʤΤ
     * '\' 
     */
    for (pos = 0; (pos = s.find("\\$", pos)) != std::string::npos; pos += 3)
        s.replace(pos, 2, "$");

    return s;
}

inline std::string insert_escape(const std::string& line)
{
    std::string s(line);
    std::string::size_type pos;

    for (pos = 0; (pos = s.find("\r\n", pos)) != std::string::npos; pos += 4)
        s.replace(pos, 2, "\\r\\n");

    for (pos = 0; (pos = s.find("\n", pos)) != std::string::npos; pos += 2)
        s.replace(pos, 1, "\\n");

    return s;
}

inline std::string scalar_to_lf(const std::string& line)
{
    std::string s(line);
    std::string::size_type pos;

    for (pos = 0; (pos = s.find("\\n", pos)) != std::string::npos; pos += 2)
        s.replace(pos, 2, "\n");

    return s;
}

inline std::string decode_xml(const std::string& s)
{
    std::string tmp(s);
    boost::replace_all(tmp, "&apos;", "'");
    boost::replace_all(tmp, "&quot;", "\"");
    boost::replace_all(tmp, "&lt;", "<");
    boost::replace_all(tmp, "&gt;", ">");
    boost::replace_all(tmp, "&amp;", "&");
    boost::replace_all(tmp, "<br />", "\n");
    return tmp;
}

inline std::string encode_xml(const std::string& s)
{
    std::string tmp(s);
    boost::replace_all(tmp, "&", "&amp;");
    boost::replace_all(tmp, "'", "&apos;");
    boost::replace_all(tmp, "\"", "&quot;");
    boost::replace_all(tmp, "<", "&lt;");
    boost::replace_all(tmp, ">", "&gt;");
    boost::replace_all(tmp, "\n", "<br />\n");
    return tmp;
}

inline std::string erase_qualifier(const std::string& s)
{
    std::string tmp(s);
    boost::erase_all(tmp, "const");
    boost::erase_all(tmp, "volatile");
    boost::erase_all(tmp, "&");
    return tmp;
}

/**
 * ѥ͡ॹڡ.
 *
 * @param s ѥޤJSPե̾.
 * @return ѥ "::" Ѵʸ.
 */
inline std::string get_namespace(const std::string& s)
{
    std::string tmp;

    std::string::size_type pos = s.rfind("/");
    if (pos != std::string::npos)
        tmp = s.substr(0, pos);

    if (tmp[0] == '.') tmp.erase(0, 1);
    if (tmp[0] == '/') tmp.erase(0, 1);
    pos = 0;
    while ((pos = tmp.find('/', pos)) != std::string::npos)
        tmp.replace(pos, 1, "::");

    return tmp;
}

} // namespace sj

#endif // SJ_JSP2C_UTIL_H
