/*-
 * Copyright (c) 2005 osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <iostream>
#include <stdexcept>
#include <boost/algorithm/string/classification.hpp>
#include <boost/algorithm/string/replace.hpp>
#include <boost/algorithm/string/trim.hpp>
using namespace std;
using namespace boost;

#include <sl/xml/xml_parser.h>
using namespace sl::xml;

#include "tld_parser.h"
using namespace sj;


bool tld_parser::start(const string& home,
                       const string& root,
                       const string_map_t& taguri,
                       const string_map_t& prefix_uri,
                       bool debug)
{
    if (debug) {
        std::clog << "*** tld_parser::start" << std::endl;
        std::clog << "home:" << home << std::endl;
        std::clog << "root:" << root << std::endl;
     
        string_map_t::const_iterator n = taguri.begin();
        for (; n != taguri.end(); n++) {
            std::clog << "taguri: first:[" << n->first << "] "
                 << "second:[" << n->second << "]" << std::endl;
        }
        for (n = prefix_uri.begin(); n != prefix_uri.end(); n++) {
            std::clog << "prefix_uri: first:[" << n->first << "] "
                 << "second:[" << n->second << "]" << std::endl;
        }
    }

    string_map_t::const_iterator i = prefix_uri.begin();
    for (; i != prefix_uri.end(); i++) {

        /* WEB-INF/classes/... */
        if (i->second[0] == '/') {
            string file_path = root + i->second;

            parse_tld(root + "/WEB-INF/classes/", i->first, file_path);
            _tag_map.insert(make_pair(file_path, parse_tag_list()));

            continue;
        }

        /* taglib ǥ쥯ƥ֤ uri ˥ޥå taguri 򸡺 */
        string_map_t::const_iterator t = taguri.find(i->second);
        if (t != taguri.end()) {
            string root_path;
            string file_path;
            if (t->second[0] == '/') {
                file_path = root + t->second;
                root_path = root + "/WEB-INF/classes";
            } else {
                file_path = home + "/" + t->second;
                root_path = home + "/common/classes";
            }
            parse_tld(root_path, i->first, file_path);
            _tag_map.insert(make_pair(file_path, parse_tag_list()));

        } else {
            string msg = "ϢTLDե뤬¸ߤޤ :";
            msg += i->first + ":" + i->second;
            cerr << msg << endl;
            msg = "URL:'" + i->first + "' is not found. "
                  "prefix to '" + i->second  +"'";
            throw std::runtime_error(msg);
        }
    }
    return true;
}


list<tld_tag> tld_parser::tld_tags(const string& prefix_uri) const
{
    tag_map_t::const_iterator i = _tag_map.find(prefix_uri);
    return i != _tag_map.end() ? i->second : list<tld_tag>();
}

list<tld_tag> tld_parser::tld_tags() const
{
    list<tld_tag>    tags;

    tag_map_t::const_iterator i = _tag_map.begin();
    for (; i != _tag_map.end(); i++)
        std::copy(i->second.begin(), i->second.end(), std::back_inserter(tags));

    return tags;
}

std::list<tld_function> tld_parser::tld_functions() const
{
    return _func_list;
}

bool tld_parser::parse_tld(const std::string& root,
                           const std::string& prefix,
                           const string& file)
{
#ifdef DEBUG
    cerr << "tld_parser::parse_tld " << file << endl;
#endif

    try {
        xml_parser xml(file);
        xml_tag taglib = xml.get();

        tag_list_t tags = taglib.get("tag");

        tag_list_t::const_iterator i = tags.begin();
        for (i = tags.begin(); i != tags.end(); i++) {
            /*
             * <tag>
             *   <name>hoge</name>
             *   <tag-class>hoge/ExampleTag</tag-class>
             *   <tei-class>hoge/ExampleTagExtraInfo</tei-class>
             *   <body-content>JSP</body-content>
             *   <description>description</description>
             *   <attribute>
             *     <name>examples</name>
             *     <required>false</required>
             *   </attribute>
             * </tag>
             */

            string name;
            string tag_class;
            string tei_class;
            string body_content;

            try {
                name = i->child("name").value();
                tag_class = i->child("tag-class").value();
            } catch (xml_parser_error& e) {
                cerr << e.what() << endl;
                throw std::runtime_error(e.what());
                return false;
            }

            try {
                tei_class = i->child("tei-class").value();
            } catch (...) { }

            try {
                body_content = i->child("body-content").value();
            } catch (...) { }

            string_map_t attr;
            tag_list_t attributes = i->get("attribute");

            tag_list_t::const_iterator j = attributes.begin();
            for (j = attributes.begin(); j != attributes.end(); j++) {
                string attr_name;
                string required;
                try {
                    attr_name = j->child("name").value();
                    required  = j->child("required").value();
                } catch (xml_parser_error& e) {
                    cerr << e.what() << endl;
                    cerr << "name of " << attr_name << endl;
                    throw std::runtime_error(e.what());
                    return false;
                }

                if (!attr_name.empty() && !required.empty())
                    attr.insert(make_pair(attr_name, required));
            }

            tld_tag info;
            info.root(root);
            info.prefix(prefix);
            info.name(name);
            info.tag_class(tag_class);
            info.tei_class(tei_class);
            info.body_content(body_content);
            info.attributes(attr);

            _tag_list.push_back(info);
        }

        tags = taglib.get("function");
        i = tags.begin();
        for (i = tags.begin(); i != tags.end(); i++) {
            /*
               * <function>
             *   <name>date</name>
             *   <function-class>sample/StringLength</function-class>
             *   <function-signature>int length(...);</function-signature>
             *  </function>
             */

            string name;
            string func_class;
            string signature;

            try {
                name       = i->child("name").value();
                func_class = i->child("function-class").value();
                signature  = i->child("function-signature").value();
            } catch (xml_parser_error& e) {
                cerr << e.what() << endl;
                throw std::runtime_error(e.what());
                return false;
            }

            tld_function info;
            info.root(root);
            info.prefix(prefix);
            info.name(name);
            info.function_class(func_class);
            info.function_signature(signature);

            _func_list.push_back(info);
        }

    } catch (sl::xml::xml_parser_error& e) {
        cerr << e.what() << endl;
        string msg("file:`" + file + "' Υѡ˥顼ȯޤ `");
        msg += e.what();
        msg += "'.";
        throw std::runtime_error(msg);
        return false;
    }

    return true;
}

std::list<tld_tag> tld_parser::parse_tag_list() const
{
    return _tag_list;
}
