/*-
 * Copyright (c) 2005 osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef SJ_COMPILER_TLD_FUNCTION_H
#define SJ_COMPILER_TLD_FUNCTION_H

#include <iostream>
#include <map>
#include <string>
#include <vector>
#include <boost/algorithm/string/classification.hpp>
#include <boost/algorithm/string/replace.hpp>
#include <boost/algorithm/string/trim.hpp>

#include "jsp_element.h"

namespace sj {

/**
 * @tld_function.
 *
 * TLD ˵Ҥ <function> ǤȼºݤΥ饹Ƥ
 * ݻ륯饹Ǥ.
 * дؿ prefix  tld_parser ˤä JSP βϷ̤ꤵ졢
 * ʳξϤ٤ <functions> ۲ǤƤǤ.
 */
class tld_function {
    typedef std::map<std::string, std::string> string_map_t;

public :
    /**
     * TLDեƤ륯饹μºݤΥѥ.
     * ̾ WEBץꥱ`ƥȥ롼+/WEB-INF/classes'
     * ˤʤޤ.
     *
     * @return    饹եΥ롼ȥǥ쥯ȥ.
     */
    std::string root() const
    {
        return _root;
    }

    /**
     * JSP ե줿 taglib ǥ쥯ƥ֤Υץեå̾.
     * 
     * @return    ץեå̾
     */
    std::string prefix() const
    {
        return _prefix;
    }

    /**
     * TLD ե <name> Ǥ.
     * 
     * @return    饹̾.
     */
    std::string name() const
    {
        return _name;
    }

    /**
     * TLD ե <function-class> Ǥ.
     * 
     * @return    <function-class> Ǥ.
     */
    std::string function_class() const
    {
        return _function_class;
    }

    /**
     * TLD ե <function-signature> Ǥ.
     * 
     * @return    <function-signature> Ǥ.
     */
    std::string function_signature() const
    {
        return _function_signature;
    }

    /**
     * <function-signature>Ƥؿΰη̾Υꥹ.
     * 
     * ºݤͤ TLD ե뤫ǤϤʤtag_class_parser 饹ˤä
     * 饹(̾*.hե뤫)ľɤ߹ͤˤʤޤ.
     *
     * @return    ̾Υꥹ.
     */
    std::vector<std::string> parameters() const
    {
        return _parameters;
    }

    /**
     * TLD եˤä륯饹Υ롼ȥѥꤷޤ.
     * ̾ WEBץꥱ`ƥȥ롼+/WEB-INF/classes'
     * ˤʤޤ.
     *
     * @param    s    饹Υ롼ȥѥ.
     */
    void root(const std::string& s)
    {
        _root = s;
    }

    /**
     * JSP ե줿 taglib ǥ쥯ƥ֤Υץեå̾.
     * 
     * @param    s    ץեå̾
     */
    void prefix(const std::string& s)
    {
        _prefix = s;
    }

    /**
     * TLD ե <name> Ǥ.
     * 
     * @param    s    饹̾.
     */
    void name(const std::string& s)
    {
        _name = s;
    }

    /**
     * TLD ե <function-class> Ǥ.
     * 
     * @param    s    <function-class> Ǥ.
     */
    void function_class(const std::string& s)
    {
        _function_class = s;
        boost::replace_all(_function_class, ".", "/");

        std::string tmp = _function_class;
        std::string::size_type pos = tmp.rfind("/");
        if (pos == std::string::npos) {
            _class = tmp;
            return;
        }

        _namespace = tmp.substr(0, pos);
        _class = tmp.substr(pos + 1);

        boost::replace_all(_namespace, "/", "::");
    }

    /**
     * TLD ե <function-signature> Ǥ.
     * 
     * @param    s    <function-signature> Ǥ.
     */
    void function_signature(const std::string& s)
    {
        _function_signature = s;
        boost::replace_all(_function_signature, ".", "::");

        std::string tmp = _function_signature;

        /* signature ̾͡ʬ */
        std::string::size_type pos = tmp.find("(");
        if (pos == std::string::npos)
            return;

        /* ʬ¸ */
        _f_arg = tmp.substr(pos);
        boost::trim_if(_f_arg, boost::is_any_of("()"));

        /* ʬΤǴؿ̾ȰγϤ '(' δ֤ζ)
         */
        tmp.erase(pos);
        boost::trim(tmp);

        pos = tmp.rfind(" ");
        if (pos == std::string::npos)
            return;

        _f_name = boost::trim_copy(tmp.substr(pos));
        _f_ret  = boost::trim_copy(tmp.substr(0, pos));
    }

    /**
     * <function-signature>Ƥؿΰη̾Υꥹ.
     * 
     * ºݤͤ TLD ե뤫ǤϤʤtag_class_parser 饹ˤä
     * 饹(̾*.hե뤫)ľɤ߹ͤˤʤޤ.
     *
     * @params    ̾Υꥹ.
     */
    void parameters(const std::vector<std::string>& params)
    {
        _parameters = params;
    }

    /**
     * 饹Ƥ namespace ޤ.
     *
     * @return    namespace.
     */
    std::string real_namespace() const
    {
        return _namespace;
    }

    /**
     * namespace ºݤΥ饹̾.
     *
     * @return    饹̾.
     */
    std::string real_class() const
    {
        return _class;
    }

    /**
     * ͤؿ̾.
     *
     * @return    ؿ̾.
     */
    std::string func_name() const
    {
        return _f_name;
    }

    /**
     * ؿ̾.
     *
     * @return    .
     */
    std::string func_ret() const
    {
        return _f_ret;
    }

    /**
     * ͡ؿ̾ʬʸ.
     *
     * @return    ʸ.
     */
    std::string func_arg() const
    {
        return _f_arg;
    }


    void debug_dump() const
    {
        std::clog << "*** tld_function Configuration" << std::endl;
        std::clog << "prefix:" << _prefix << std::endl;
        std::clog << "name:" << _name << std::endl;
        std::clog << "function_class:" << _function_class << std::endl;
        std::clog << "function_signature:" << _function_signature << std::endl;
        std::clog << "_f_name:" << _f_name << std::endl;
        std::clog << "_f_ret:" << _f_ret << std::endl;
        std::clog << "_f_arg:" << _f_arg << std::endl;
        std::clog << "_class:" << _class << std::endl;
        std::clog << "_namespace:" << _namespace << std::endl;

        std::vector<std::string>::const_iterator i = _parameters.begin();
        for (; i != _parameters.end(); ++i)
            std::clog << "_parameters:" << *i << std::endl;
    }

private :
    std::string _root;
    std::string _prefix;
    std::string _name;
    std::string _function_class;
    std::string _function_signature;
    std::vector<std::string> _parameters;

    std::string _namespace;
    std::string _class;
    std::string _f_name;
    std::string _f_ret;
    std::string _f_arg;
};

inline bool operator<(const tld_function& l, const tld_function& r)
{
    return (l.prefix() + l.name()) < (r.prefix() + r.name());
}

} // namespace sj

#endif // SJ_COMPILER_TLD_FUNCTION_H
