/*-
 * Copyright (c) 2005 osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <sys/types.h>
#include <sys/stat.h>

#include <iostream>
#include <fstream>
#include <set>
#include <string>
#include <boost/filesystem/operations.hpp>
using namespace std;

#include "compiler.h"
#include "generator.h"
#include "jsp_compiler.h"
#include "jsp_element.h"
#include "jsp_parser.h"
#include "tag_class_parser.h"
#include "tld_parser.h"
#include "util.h"
using namespace sj;


#if defined(__CYGWIN__)
const std::string dso_suffix = ".dll";
#else
const std::string dso_suffix = ".so";
#endif

bool jsp_compiler::run(const string& home,
                        const string& context_root,
                        const string& jsp_file,
                        const string& outputdir,
                        const string_map_t& taguri,
                        const string& CC,
                        const options_t& opt,
                        const options_t& inc,
                        const options_t& lib,
                        const options_t& shared,
                        set<string> &dep_libs,
                        bool debug)
{
    if (debug) {
        std::clog << "*** bool jsp_compiler::run() Enter" << std::endl;
        std::clog << "home         :" << home << std::endl;
        std::clog << "context_root :" << context_root << std::endl;
        std::clog << "jsp_file     :" << jsp_file << std::endl;
        std::clog << "outputdir    :" << outputdir << std::endl;
    
        std::map<std::string, std::string>::const_iterator m = taguri.begin();
        for (; m != taguri.end(); m++)
            std::clog << "taguri.first:" << m->first << " second:" << m->second << std::endl;
    
        options_t::const_iterator n;
        for (n = opt.begin(); n != opt.end(); n++)
            clog << "option :" << *n << std::endl;
        for (n = inc.begin(); n != inc.end(); n++)
            clog << "include:" << *n << std::endl;
        for (n = lib.begin(); n != lib.end(); n++)
            clog << "library:" << *n << std::endl;
        for (n = shared.begin(); n != shared.end(); n++)
            clog << "shared:" << *n << std::endl;
    }

    if (context_root.empty())
        throw std::runtime_error("jsp_compiler::run failed. context_root is empty");

    /*
     * JSPե뤫 饹̾  ե̾Ȥʤʸ.
     * 
     * jsp_file  context_root ХѥդʸʤΤ
     * ե̾ȥѥڤФƥ饹̾͡ॹڡȤƻѤ.
     */
    std::string jsp_file_name;
    std::string::size_type pos = jsp_file.rfind("/");
    if (pos != std::string::npos)
        jsp_file_name = jsp_file.substr(pos + 1);
    else
        jsp_file_name = jsp_file;

    std::string class_name    = jsp_to_class(jsp_file_name);
    std::string jsp_file_path = context_root + "/" + jsp_file;

    /* ѥ͡ॹڡ */
    std::string jsp_name_space = sj::get_namespace(jsp_file);

    /*
     * ʹߤνɬפˤʤѥ̾+եƤ.
     */
    std::string output_object_file = outputdir + "/" + class_name + dso_suffix;
    std::string output_header_file = outputdir + "/" + class_name + ".h";
    std::string output_source_file = outputdir + "/" + class_name + ".cpp";
    std::string class_source_file = class_name + ".cpp";

    if (debug) {
std::clog << "jsp_file_path      :" << jsp_file_path << std::endl;
std::clog << "jsp_file_name      :" << jsp_file_name << std::endl;
std::clog << "jsp_name_space     :" << jsp_name_space << std::endl;
std::clog << "class_name         :" << class_name << std::endl;
std::clog << "output_object_file :" << output_object_file << std::endl;
std::clog << "output_header_file :" << output_header_file << std::endl;
std::clog << "output_source_file :" << output_source_file << std::endl;
std::clog << "class_source_file  :" << class_source_file << std::endl;
    }

    /**
     * JSP եϡݻ륯饹 jsp_parser ơ¹Ԥ.
     */
    ifstream ifs(jsp_file_path.c_str());
    if (!ifs)
        throw std::runtime_error("jsp_compiler::run failed. can't open path:" + jsp_file_path);

    string jsp_file_content((istreambuf_iterator<char>(ifs)),
                            istreambuf_iterator<char>());
    ifs.close();

    jsp_parser p;
    if (!p.start(jsp_file_content))
        return false;

    /*
     * JSPβϷ̤ơη̤ JSP եǻѤƤ
     * taglib ǥ쥯ƥ֤ prefix/uri ޥåפȤƼ.
     */
    jsp_element_ptr jsp_root_element = p.result();
    map<string, string> uri_map = p.taguri_map(jsp_root_element);

    /**
     * JSP βϷ̤ˡºݤ TLD եƤϤ
     * ̤ȤФ Tag 饹ξöƼФ.
     */
    tld_parser t;
    t.start(home, context_root, taguri, uri_map, debug);

    list<tld_tag> tld_tags = t.tld_tags();
    list<tld_function> tld_functions  = t.tld_functions();

    /** ʹߤνǤȤѤ Iterator.  */
    list<tld_tag>::iterator            ite1;
    list<tld_function>::iterator    ite2;

    if (debug) {
    ite1 = tld_tags.begin();
    for (; ite1 != tld_tags.end(); ite1++) ite1->debug_dump();
    ite2 = tld_functions.begin();
    for (; ite2 != tld_functions.end(); ite2++) ite2->debug_dump();
    }
std::clog << __LINE__ << std::endl;

    /*
     * Ƥ Tag 饹٤ƤٲϤ
     * Servlet ɬפ Tag 饹 tld_tag ɲäƤ.
     */
    for (ite1 = tld_tags.begin(); ite1 != tld_tags.end(); ite1++) {
        string file_path(ite1->root() + "/" + ite1->tag_class() + ".h");

        ifstream ifs(file_path.c_str());
        if (!ifs) {
            cerr << "Can't open file:" << file_path << endl;
            throw std::runtime_error(string("Can't open file:") + file_path);
        }

        string file((istreambuf_iterator<char>(ifs)),
                    istreambuf_iterator<char>());

        tag_class_parser tp(file, ite1->tag_class(), debug);
        ite1->base_classes(tp.base_classes());
        ite1->functions(tp.functions());
    }
std::clog << __LINE__ << std::endl;

    /*
     * Ƥ Functions ѤΥ饹٤ƤٲϤ
     * Servlet ɬפʥ饹 tld_function ɲäƤ.
     */
    for (ite2 = tld_functions.begin(); ite2 != tld_functions.end(); ite2++) {
        string file_path(ite2->root() + "/" + ite2->function_class() + ".h");

        ifstream ifs(file_path.c_str());
        if (!ifs) {
            cerr << "Can't open file:" << file_path << endl;
            throw std::runtime_error(string("Can't open file:") + file_path);
        }

        string file((istreambuf_iterator<char>(ifs)),
                    istreambuf_iterator<char>());

        tag_class_parser tp(file, ite2->function_class(), debug);
        ite2->parameters(tp.parameters(ite2->func_name()));
    }

    /*
     * ѤߤΥ¸ߤƤġJSP ե
     * ϥӥѥԤʤʤ.
     */
    struct stat jsp_file_stat;
    struct stat src_stat;
    struct stat dso_stat;
    time_t src_time = 0;
    time_t dso_time = 0;
    time_t jsp_time = 0;

    if (::stat(output_object_file.c_str(), &dso_stat) == 0)
        dso_time = dso_stat.st_mtime;
    if (::stat(output_source_file.c_str(), &src_stat) == 0)
        src_time = src_stat.st_mtime;
    if (::stat(jsp_file_path.c_str(), &jsp_file_stat) == 0)
        jsp_time = jsp_file_stat.st_mtime;

    if (jsp_time < src_time && src_time <= dso_time) {
         /*
         * ֥ȥեϥɤɬפΤ
         *  dep_libs ˤϰ¸륯饹̾ͤꤷƤ.
         */
        for (ite1 = tld_tags.begin(); ite1 != tld_tags.end(); ite1++)
            dep_libs.insert(ite1->root() + "/" + ite1->tag_class());

        for (ite2 = tld_functions.begin(); ite2 != tld_functions.end(); ite2++)
            dep_libs.insert(ite2->root() + "/" + ite2->function_class());

        return true;
    }
std::clog << __LINE__ << std::endl;
    /*
     *  jsp  ѥޤ JSP ե̾Ǥ뤿
     *  JSP ե̾ڤФ
     */
    generator g;
    g.start(class_name,
            jsp_name_space,
            context_root,
            outputdir,
            jsp_root_element,
            tld_tags,
            tld_functions,
            debug);

    /*
     * ƥ˥󥹥ȡ뤵줿ѥ鵯ư.
     */
    compiler cc(outputdir, class_source_file, output_object_file);

    options_t::const_iterator j = opt.begin();
    for (; j != opt.end(); j++)
        cc.add_option(*j);

    for (j = inc.begin(); j != inc.end(); j++) {
        if ((*j)[0] != '-')
            cc.add_include("-I" + (*j));
        else
            cc.add_include(*j);
    }

    for (j = lib.begin(); j != lib.end(); j++) {
        if ((*j)[0] != '-')
            cc.add_library("-L" + (*j));
        else
            cc.add_library(*j);
    }

    for (j = shared.begin(); j != shared.end(); j++)
        cc.add_shared(*j);

    for (ite1 = tld_tags.begin(); ite1 != tld_tags.end(); ite1++) {
        if (ite1->root()[0] != '-')
            cc.add_include("-I" + ite1->root());
        else
            cc.add_include(ite1->root());
        dep_libs.insert(ite1->root() + "/" + ite1->tag_class());
    }

    for (ite2 = tld_functions.begin(); ite2 != tld_functions.end(); ite2++) {
        if (ite2->root()[0] != '-')
            cc.add_include("-I" + ite2->root());
        else
            cc.add_include(ite2->root());
        dep_libs.insert(ite2->root() + "/" + ite2->function_class());
    }

    set<string>::iterator i = dep_libs.begin();
    for (; i != dep_libs.end(); i++) {
        if (debug)
            std::cerr << "dep_lib:" << *i << std::endl;

        std::string dep_lib_name(*i + dso_suffix);
        boost::replace_all(dep_lib_name, "//", "/");
        if (boost::filesystem::exists(dep_lib_name))
            cc.add_library(*i + dso_suffix);
    }

    cc.system_compiler(CC);
    cc.compile();

    return true;
}
