/*-
 * Copyright (c) 2005 osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <cerrno>
#include <iostream>
#include <iomanip>
#include <iterator>
#include <fstream>
#include <stdexcept>
#include <boost/algorithm/string/classification.hpp>
#include <boost/algorithm/string/replace.hpp>
#include <boost/algorithm/string/erase.hpp>
#include <boost/algorithm/string/split.hpp>
#include <boost/algorithm/string/trim.hpp>
#include <boost/bind.hpp>
#include <boost/filesystem/path.hpp>
#include <boost/filesystem/operations.hpp>
#include <boost/filesystem/convenience.hpp>
#include <boost/lexical_cast.hpp>
using namespace std;
using namespace boost;

#include "util.h"
#include "generator.h"
using namespace sj;



generator::generator()
{ }


bool generator::start(const std::string& jsp_file,
                      const std::string& jsp_name_space,
                      const std::string& root,
                      const std::string& output_dir,
                      const jsp_element_ptr& jsp_element_root,
                      const tag_list_t& tags,
                      const func_list_t& tld_functions,
                      bool debug)
{
    _debug = debug;
    if (_debug) {
        std::clog << "*** sj::generator::start" << std::endl;;
        std::clog << "jsp_file  :" << jsp_file << std::endl;;
        std::clog << "jsp_name_space  :" << jsp_name_space << std::endl;;
        std::clog << "root      :" << root << std::endl;;
        std::clog << "output_dir:" << output_dir << std::endl;;
    }

    /*
     * jsp_file  root ХѥդʸʤΤ
     * ե̾ȥѥڤФƥ饹̾͡ॹڡȤƻѤ.
     */
    string::size_type pos = jsp_file.rfind("/");

    _name_space = jsp_name_space;
    _class_name = jsp_file;

    /* ͡ॹڡ */
    std::vector<std::string> sp;
    boost::split(sp, _name_space, boost::is_any_of("::"), boost::token_compress_on);
    std::vector<std::string>::iterator i = sp.begin();
    for (; i != sp.end(); i++) {
        if (i->empty())
            continue;
        _name_space_begin += "namespace " + *i + " { ";
        _name_space_end   += "} ";
    }


    /*
     * Tag 饹Υꥹȵڤ Functions 
     * ¾Υдؿ⻲Ƚ褦˥ѿسǼƤ.
     */
    _tags = tags;
    _tld_functions = tld_functions;

    /* ѤΥǥ쥯ȥ */
    filesystem::path p(output_dir);
    filesystem::create_directories(p);

    header_begin();

    default_include();
    depend_classes();
    init();
    destroy();
    service();
    service(jsp_element_root->jsp_elements());

    header_end();

    output(output_dir);

    if (_debug)
        debug_dump();

    return true;
}


void generator::header_begin()
{
    file_type f;
    f << "#ifndef __JSP_SERVLET_H__";
    f << "#define __JSP_SERVLET_H__";
    f << "";
    f << "#include <sl/class_loader/register.hpp>";
    f << "#include <servlet/servlet_headers.h>";
    f << "#include <servlet/jsp_headers.h>";
    f << "";
    f << _name_space_begin;
    f << "";
    f << "class " + _class_name + " : virtual public servlet::jsp::HttpJspPage {";
    f << "public :";
    f << "    " + _class_name + "() throw() { }";
    f << "    virtual ~" + _class_name + "() throw() { }";
    f << "";
    f << "    void jspInit() throw();"; 
    f << "    void jspDestroy() throw();";
    f << "    void service(servlet::ServletRequest&, servlet::ServletResponse&)";
    f << "        throw(servlet::ServletException);";
    f << "";
    f << "    void _jspService(servlet::http::HttpServletRequest&, servlet::http::HttpServletResponse&)";
    f << "        throw(servlet::ServletException);";
    f << "";

    _header_file.push_back(f);
}

void generator::header_end()
{
    file_type f;

    f << "";
    f << "}; // end of class";
    f << "";
    f << _name_space_end;
    f << "";
    f << "SL_REGIST_LOADABLE_CLASS(" + _name_space + "::" + _class_name + ")";
    f << "";
    f << "#endif // __JSP_SERVLET_H__";

    _header_file.push_back(f);
}

void generator::default_include()
{
    file_type f;

    f << "#include <sl/java.hpp>";
    f << "#include <sl/object.hpp>";
    f << "#include \"" + _class_name + ".h\"";
    f << "";

    _source_file.push_back(f);
}

void generator::depend_classes()
{
    file_type f;
    f << "";
    f << "using namespace servlet;";
    f << "using namespace servlet::http;";
    f << "using namespace servlet::jsp;";
    f << "using namespace servlet::jsp::el;";
    f << "using namespace servlet::jsp::tagext;";
    f << "";
    f << "namespace {";
    f << "";
    f << "    class __jsp_null_tag : virtual public Tag {";
    f << "    public :";
    f << "        bool operator!() const { return true; }";
    f << "        operator const void*() const { return 0; }";
    f << "        int doEndTag() throw (servlet::jsp::JspException) { return 0; }";
    f << "        int doStartTag() throw (servlet::jsp::JspException) { return 0; }";
    f << "        Tag& getParent() { return *this; }";
    f << "        void release()   { }";
    f << "        void setPageContext(PageContext& pc) { }";
    f << "        void setParent(Tag& t) { }";
    f << "    };";
    f << "";

    ++f.indent;
    function_mapper(f);
    --f.indent;

    f << "";
    f << "    template <class T>";
    f << "    class __jsp_fragment : virtual public JspFragment {";
    f << "    public :";
    f << "        typedef void (T::*__invoke_function)(JspContext&);";
    f << "";
    f << "        __jsp_fragment(T* p, __invoke_function f)";
    f << "            : _page(p), _function(f)";
    f << "        { }";
    f << "";
    f << "        void setJspContext(JspContext& c) { _context = &c; }";
    f << "        JspContext& getJspContext() { return *_context; }";
    f << "";
    f << "        void invoke()";
    f << "        {";
    f << "            ((_page)->*(_function))(*_context);";
    f << "        }";
    f << "";
    f << "        void invoke(sl::java::io::Writer& /* f */)";
    f << "        {";
    f << "            invoke();";
    f << "        }";
    f << "";
    f << "    private :";
    f << "        T*                 _page;";
    f << "        __invoke_function  _function;";
    f << "        JspContext*        _context;";
    f << "    };";
    f << "}";
    f << "";

    _source_file.push_back(f);
}

void generator::init()
{
    file_type f;
    f << "";

        if (!_name_space.empty())
        f << "using namespace ::" + _name_space + ";";

    f << "";
    f << "void ";
    f << _class_name + "::jspInit() throw()";
    f << "{";
    f << "";
    f << "}";
    f << "";

    _source_file.push_back(f);
}

void generator::destroy()
{
    file_type f;
    f << "void ";
    f << _class_name + "::jspDestroy() throw()";
    f << "{";
    f << "}";
    f << "";

    _source_file.push_back(f);
}

void generator::service()
{
    file_type f;
    f << "void ";
    f << _class_name + "::service(";
    f << "    ServletRequest& request, ServletResponse& response)";
    f << "    throw(ServletException)";
    f << "{";
    f << "    servlet::http::HttpServletRequest&  req =";
    f << "        dynamic_cast<servlet::http::HttpServletRequest&>(request);";
    f << "    servlet::http::HttpServletResponse& res =";
    f << "        dynamic_cast<servlet::http::HttpServletResponse&>(response);";
    f << "    _jspService(req, res);";
    f << "}";
    f << "";

    _source_file.push_back(f);
}

void generator::service(const vector<jsp_element_ptr>& e)
{
    file_type f;
    f << "void ";
    f << _class_name + "::_jspService(";
    f << "    HttpServletRequest& request, HttpServletResponse& response)";
    f << "    throw(ServletException)";
    f << "{";
    f << "    __jsp_null_tag __jsp_tag_instance;";
    f << "    __jsp_function_mapper __function_mapper;";
    f << "";
    f << "    try{";
    f << "";
    f << "    JspFactory& __jsp_factory = JspFactory::getDefaultFactory();";
    f << "    PageContext& pageContext = __jsp_factory.getPageContext(";
    f << "                                          *this,";
    f << "                                          request,";
    f << "                                          response,";
    f << "                                          std::string(),";
    f << "                                          true,";
    f << "                                          8192,";
    f << "                                          true);";
    f << "    ServletContext& application = pageContext.getServletContext();";
    f << "    ServletConfig& config = pageContext.getServletConfig();";
    f << "    HttpSession& session = pageContext.getSession();";
    f << "    JspWriter& out = pageContext.getOut();";
    f << "";
    f << "    servlet::jsp::el::VariableResolver& __variable_resolver =";
    f << "                        pageContext.getVariableResolver();";
    f << "    servlet::jsp::el::ExpressionEvaluator& __expression_evaluator = ";
    f << "                        pageContext.getExpressionEvaluator();";
    f << "";

    ++f.indent;

    _current_tag.push("__jsp_tag_instance");
    select_process(e, f);

    --f.indent;

    f << "        ";
    f << "    } catch (std::exception& e) {";
    f << "        std::cerr << \"Jsp Runtime Error :\" << e.what() << std::endl;";
    f << "        throw ServletException(e.what());";
    f << "    } catch (...) {";
    f << "        std::cerr << \"JSP Runtime Exception\" << std::endl;";
    f << "        throw ServletException(\"JSP Runtime Exception\");";
    f << "    }";
    f << "}";

    _source_file.push_back(f);
}

void generator::select_process(const vector<jsp_element_ptr>& e, file_type& f)
{
    file_type temp;

    vector<jsp_element_ptr>::const_iterator i = e.begin();
    for (; i != e.end(); i++) {
        switch ((*i)->type()) {
        case normal :    /* ̾ι */
            normal_line(f, (*i)->line());
            break;

        /*
         * <%= ... %> ǳ줿ʬ
         * '"' ǳ餺̤ printؿϤ.
         */
        case expression :
            f << string("out.print(") + (*i)->line() + ");";
            break;
        /*
         * <%! ... %> ǳ줿ʬ
         * إåեˤΤޤ޽񤭹.
         */
        case declaration :
            temp.clear();
            temp.push_back((*i)->line());
            _header_file.push_back(temp);
            break;
        /*
         * <% ... %> ǳ줿ʬ
         * եˤΤޤ޽񤭹.
         */
        case scriptlet :
            f << (*i)->line();
            break;
        /*
         * <%@ ... %> ǳ줿ʬ
         */
        case directive :
            if ((*i)->action() == "page")
                page_directive(*i, f);
            break;
        /*
         * ॿ
         */
        case action :
            tag_action(*i, f);
            break;
        /*
         * jsp ץեåդ
         */
        case jsp_std_action :
            jsp_action(*i, f);
            break;

        default :
            break;
        }
    }
}

void generator::normal_line(file_type& f, const std::string& line)
{
    string::size_type pos = line.find("${");
    if (pos == string::npos || (pos != 0 && line[pos - 1] == '\\')) {
        f << "out.print(\"" + quote_to_scalar(insert_escape(line))+"\");";
        return;
    }

    string s = line;
    do {
        if (pos != 0) {
            string b = s.substr(0, pos);
            f << "out.print(\"" + quote_to_scalar(insert_escape(b))+"\");";
        }

        string::size_type end = s.find('}', pos);
        if (end == string::npos)
            throw std::runtime_error("error");

        ++end;
        string el = s.substr(pos, end - pos);
        string code = el_expression(el);

        f << "out.print(" + code +");";

        string::size_type pos = line.find("${", end);
        if (pos == string::npos)
            s.erase(0, end);
        else
            s.erase(0, pos);

        if (s.length())
            f << "out.print(\"" + quote_to_scalar(insert_escape(s))+"\");";

    } while ((pos = s.find("${")) != string::npos);
}

void generator::tag_action(const jsp_element_ptr& e, file_type& f)
{
    /*
     * Υץեåȥ̾դȤʤʸ.
     * ͤϥΥ󥹥̾ѿ̾ؿδؿ̾˻Ѥ.
     */
    string p = e->prefix();
    string a = e->action();
    string unique_prefix = key_prefix + p + "_" + a;
    int count;

    map<string, int>::iterator i = _unique_prefix_count.find(unique_prefix);
    if (i == _unique_prefix_count.end()) {
        count = 0;
        _unique_prefix_count.insert(make_pair(unique_prefix, 1));
    } else {
        count = i->second;
        _unique_prefix_count[unique_prefix] = i->second+1;
    }

    string current_tag = unique_prefix + boost::lexical_cast<string>(count);
    string parent_tag  = _current_tag.top();

    if (_debug) {
        std::clog << "sj::generator::tag_action" << std::endl;
        std::clog << "prefix  :" << p << std::endl;
        std::clog << "action  :" << a << std::endl;
        std::clog << "unique_prefix:" << unique_prefix << std::endl;
        std::clog << "current :" << current_tag << std::endl;
        std::clog << "parent  :" << parent_tag << std::endl;
    }

    /*
     * Υץեåȥ̾饯饹̾ꤹ.
     */
    string tag_class;
    string tag_class_name;
    string tei_class;
    string tag_namespace;
    tld_tag tag;

    tag_list_t::iterator j = _tags.begin();
    for (; j != _tags.end(); j++) {
        if (j->prefix() == p && j->name() == a) {
            tag_class = j->tag_class();
            tei_class = j->tei_class();
            tag = *j;
        }
    }

    /*
     * 饹̾ '/'  '/' ޤ᤿ѥʬ
     * tag_class_name  Ƭ '/' ޤǤ tag_namespace .
     * ޤ tag_namespace  '/'  "::" ѴƤ.
     */
    string::size_type pos = tag_class.rfind('/');
    if (pos != string::npos) {
        tag_class_name = tag_class.substr(pos + 1);
        tag_namespace  = tag_class.substr(0, pos + 1);

        while ((pos = tag_namespace.find('/')) != string::npos)
            tag_namespace.replace(pos, 1, "::");
    } else
        tag_class_name = tag_class;


    /*
     * 饹 #include Ȥ¸Ƥ
     */
    string s = "#include \"" + tag_class + ".h\"";
    if (find(_depend.begin(), _depend.end(), s) == _depend.end())
        _depend.push_back(s);

    f << "";
    f << tag_namespace + tag_class_name + " " + current_tag + ";";
    f << "";

    /*
     * Ѥδؿ.
     */
    std::set<std::string> classes = tag.base_classes();
    if (classes.find("SimpleTag")        != classes.end() ||
        classes.find("SimpleTagSupport") != classes.end()) 
    {

        /* SimpleTag Υܥǥ̴ؿ */
        string body_func_name = current_tag + "_body";
        simple_tag_body(_class_name, body_func_name, e);

        string unique_fragment = key_prefix + "fragment";
        map<string, int>::iterator i =
                                    _unique_prefix_count.find(unique_fragment);
        if (i == _unique_prefix_count.end()) {
            count = 0;
            _unique_prefix_count.insert(make_pair(unique_fragment, 1));
        } else {
            count = i->second;
            _unique_prefix_count[unique_fragment] = i->second+1;
        }
        string fragment_name = unique_fragment +
                                    boost::lexical_cast<string>(count);

        f << "__jsp_fragment<" + _class_name + ">";
        f << "    " + fragment_name + "(this, &" + _class_name + "::" + body_func_name + ");";
        f << fragment_name + ".setJspContext(pageContext);";
        f << current_tag + ".setJspBody(" + fragment_name + ");";
        f << current_tag + ".setJspContext(pageContext);";

        /* Tag °ͤѥ */
        attribute_setter(current_tag, e, tag, f);

        f << "";
        f << current_tag + ".doTag();";
        f << "";

    } else {
        /*
         * SimpleTag ʳξΥɤ
         */

        /* dynamic_cast ǥ󥹥󥹤Υ饹ʤ */
        f << current_tag + ".setPageContext(pageContext);";
        f << current_tag + ".setParent(" + parent_tag + ");";

        /* Tag °ͤѥ */
        attribute_setter(current_tag, e, tag, f);

        f << "";
        f << "int " + current_tag + "_result = " +
                                            current_tag + ".doStartTag();";

        if (e->jsp_elements().size()) {
            f << "if (" + current_tag + "_result != Tag::SKIP_BODY) {";
            f << "    for (;;) {";
            f << "";

            _current_tag.push(current_tag);
            ++f.indent;

            select_process(e->jsp_elements(), f);

            --f.indent;
            _current_tag.pop();

            f << "";
            f << "        int __after_body = " +current_tag + ".doAfterBody();";
            f << "        if (__after_body != BodyTag::EVAL_BODY_AGAIN)";
            f << "            break;";
            f << "    }";
            f << "}";
        }

        f << "if (" + current_tag + ".doEndTag() == Tag::SKIP_PAGE) {";
        f << "    return;";
        f << "}";
    }

    f << "";
}

void generator::simple_tag_body(const std::string& class_name,
                                         const std::string& func_name,
                                         const jsp_element_ptr& e)
{
    file_type f;
    f << "";
    f << "void";
    f << class_name + "::" + func_name + "(JspContext& __jsp_context)";
    f << "{";
    f << "    JspWriter& out = __jsp_context.getOut();";
    f << "    servlet::jsp::el::VariableResolver& __variable_resolver =";
    f << "                     __jsp_context.getVariableResolver();";

    f << "    servlet::jsp::el::ExpressionEvaluator& __expression_evaluator = ";
    f << "                    __jsp_context.getExpressionEvaluator();";
    f << "";
    f << "    __jsp_function_mapper __function_mapper;";
    f << "";

    ++f.indent;
    select_process(e->jsp_elements(), f);
    --f.indent;

    f << "}";
    f << "";

    _source_file.push_back(f);

    /* إåɲä.  */
    f.clear();
    f << "    void";
    f << "    " + func_name + "(JspContext& __jsp_context);";

    _header_file.push_back(f);
}


void generator::jsp_action(const jsp_element_ptr& e, file_type& f)
{
    /*
     * <jsp:include>
     * <jsp:forward>
     * <jsp:useBean>
     * <jsp:setProperty>
     * <jsp:getProperty>
     * <jsp:plugin>
     * <jsp:body>
     * <jsp:doBody>
     * <jsp:invoke>
     * <jsp:attribute>
     * <jsp:element>
     * <jsp:output>
     * <jsp:root>
     * <jsp:text>
     */

    string action(e->action());

    typedef std::map<std::string, std::string> string_map_t;

    if (action == "include") {
        string page, flush;

        string_map_t::iterator i = e->attribute.find("page");
        if (i != e->attribute.end())
            page = i->second;

        i = e->attribute.find("flush");
        if (i != e->attribute.end())
            flush = i->second;

        f << "request.getRequestDispatcher(" + page +
                ").include(request, response);";

    } else if (action == "forward") {
        string page;

        string_map_t::iterator i = e->attribute.find("page");
        if (i != e->attribute.end())
            page = i->second;

        f << "request.getRequestDispatcher(" + page +
                ").forward(request, response);";

    } else if (action == "useBean") {
        string id, scope, clazz, beanName, type;

        string_map_t::iterator i = e->attribute.find("id");
        if (i != e->attribute.end())
            id = i->second;

        i = e->attribute.find("scope");
        if (i != e->attribute.end())
            scope = i->second;

        i = e->attribute.find("class");
        if (i != e->attribute.end())
            clazz = i->second;

        i = e->attribute.find("beanName");
        if (i != e->attribute.end())
            beanName = i->second;

        i = e->attribute.find("type");
        if (i != e->attribute.end())
            type = i->second;

        replace_all(clazz, ".", "::");
        if (type.empty())
            type = clazz;
        else
            replace_all(type, ".", "::");

        if (scope.empty())
            scope = "page";

        trim_if(id, is_any_of("\""));
        trim_if(scope, is_any_of("\""));
        trim_if(clazz, is_any_of("\""));
        trim_if(beanName, is_any_of("\""));
        trim_if(type, is_any_of("\""));

        f << type + " " + id + ";";
        f << "sl::java::lang::Object";
        f << "    __obj_" + id + "(" + id + ");";

        if (scope == "page")
            f << "pageContext.setAttribute(\"" + id + "\", __obj_" + id + ");";
        else if (scope == "request")
            f << "request.setAttribute(\"" + id + "\", __obj_" + id + ");";
        else if (scope == "application")
            f << "application.setAttribute(\"" + id + "\", __obj_" + id + ");";
        else if (scope == "session")
            f << "session.setAttribute(\"" + id + "\", __obj_" + id + ");";

        /* bean Υإåե #include Ȥ¸Ƥ */
        replace_all(clazz, "::", "/");
        string s = "#include \"" + clazz + ".h\"";
        if (find(_depend.begin(), _depend.end(), s) == _depend.end())
            _depend.push_back(s);

    } else if (action == "setProperty") {
        string name, property, param, value;

        string_map_t::iterator i = e->attribute.find("name");
        if (i != e->attribute.end())
            name = i->second;

        i = e->attribute.find("property");
        if (i != e->attribute.end())
            property = i->second;

        i = e->attribute.find("param");
        if (i != e->attribute.end())
            param = i->second;

        i = e->attribute.find("value");
        if (i != e->attribute.end())
            value = i->second;

    } else if (action == "getProperty") {
        string name, property;

        string_map_t::iterator i = e->attribute.find("name");
        if (i != e->attribute.end())
            name = i->second;

        i = e->attribute.find("property");
        if (i != e->attribute.end())
            property = i->second;

        trim_if(name, is_any_of("\""));
        trim_if(property, is_any_of("\""));

        property[0] = toupper(property[0]);
        property.insert(0, "get");

        f << "out.print(" + name + "." + property + "());";
    }
}

void generator::page_directive(const jsp_element_ptr& e, file_type& f)
{
    typedef map<string, string> string_map_t;
    string_map_t::const_iterator i = e->attribute.begin();

    for (; i != e->attribute.end(); i++) {
        if (i->first == "contentType") {
            f << "    response.setContentType(\"" + i->second + "\");";
            f << "";
        } else if (i->first == "import") {
            _import.push_back("#include <" + i->second + ">");

        } else if (i->first == "namespace") {
            _import.push_back("using namespace " + i->second + ";");
        }
    }
}

void print_line(const file_type& file, int& line_num)
{
    deque<string>::const_iterator i = file.begin();
    for (; i != file.end(); i++)
        cout << setw(4) << line_num++ << " " << *i << endl;
}

void generator::debug_dump()
{
    cerr << "----- header -----" << endl;

    int line = 1;
    vector<file_type>::iterator i = _header_file.begin();
    for (; i != _header_file.end(); i++)
        print_line(*i, line);

    cerr << "----- source -----" << endl;
    line = 1;

    /* ¸Ƥ륿Υ饹 #incldue ʬϤ */
    print_line(_depend, line);

    /* ¸Ƥإå #incldue ʬϤ */
    print_line(_import, line);

    for (i = _source_file.begin(); i != _source_file.end(); i++)
        print_line(*i, line);
}


void generator::output(const string& output_dir)
{
    string header_name = output_dir + "/" +  _class_name + ".h";
    string source_name = output_dir + "/" +  _class_name + ".cpp";

    ofstream ofs;
    ofs.open(header_name.c_str());
    if (!ofs) {
        cerr << "ե:" << header_name << " 򥪡ץޤǤ."
             << endl;
        return;
    }

    /* إå */
    vector<file_type>::iterator i = _header_file.begin();
    for (; i != _header_file.end(); i++)
        copy(i->begin(), i->end(), ostream_iterator<string, char>(ofs, "\n"));

    ofs.close();

    /*  */
    ofs.open(source_name.c_str());
    if (!ofs) {
        cerr << "ե:" << source_name << " 򥪡ץޤǤ."
             << endl;
        return;
    }

    /* ¸Ƥ륿Υ饹 #incldue ʬϤ */
    copy(_depend.begin(), _depend.end(),
                                ostream_iterator<string, char>(ofs, "\n"));

    /* ¸Ƥإå #incldue ʬϤ */
    copy(_import.begin(), _import.end(),
                                ostream_iterator<string, char>(ofs, "\n"));

    /* եμºݤΥʬϤ */
    i = _source_file.begin();
    for (; i != _source_file.end(); i++)
        copy(i->begin(), i->end(), ostream_iterator<string, char>(ofs, "\n"));

    ofs.close();
}

void generator::function_mapper(file_type& f)
{

    f << "class __jsp_function_mapper";
    f << "    : virtual public servlet::jsp::el::FunctionMapper";
    f << "{";
    f << "public :";
    f << "    __jsp_function_mapper()";
    f << "    { }";
    f << "";
    f << "    sl::java::lang::Object resolveFunction(const std::string& prefix,";
    f << "                                      const std::string& name,";
    f << "                                      std::vector<sl::java::lang::Object>& objs) const";
    f << "    {";

    std::string name, prefix, clazz, signature;
    std::string f_ret, f_name, f_param;

    func_list_t::iterator i = _tld_functions.begin();
    for (; i != _tld_functions.end(); i++) {

        if (_debug) {
            std::clog << "TLD Functions" << std::endl;
            std::clog << "prefix   :" << i->prefix() << std::endl;
            std::clog << "name     :" << i->name() << std::endl;
        }

        f << "        if (prefix == \"" + i->prefix() + "\" && "
             "name == \"" + i->name() + "\") {";

        f.indent++;
        vector<string> params = i->parameters();
            f << "        if (objs.size() != " +
                    boost::lexical_cast<string>(params.size()) + ")";
            f << "            throw std::invalid_argument(\""
                    "The number of arguments is illegal.\");";

        for (size_t n = 0; n < params.size(); n++) {
            string type = params[n];
            type = erase_qualifier(decode_xml(type));

            f << "        " + type + " arg" +
                boost::lexical_cast<string>(n) +
                " = sl::object_cast<" + type + ">(objs[" +
                boost::lexical_cast<string>(n) + "]);";
        }

        f << "        " + i->func_ret() + " ret = ";
        f << "            " + i->real_namespace() + "::" +
                              i->real_class() + "::" +
                              i->func_name() + "(";

        for (size_t n = 0; n < params.size(); n++) {
            string l("               arg");
            l += boost::lexical_cast<string>(n);

            if ((n + 1) < params.size())
                f << l + ",";
            else
                f << l;
        }
        f <<  "            );";

#if 1 /* XXXXX EL Function ͤϤɤΤ褦˰... */
        f << "        std::string s = boost::lexical_cast<std::string>(ret);";
        f << "        return sl::java::lang::Object(s);";
#endif
        f.indent--;
        f << "        }";

        /*
         * Function饹 #include Ȥ¸Ƥ
         */
        string s = "#include <boost/lexical_cast.hpp>";
        if (find(_depend.begin(), _depend.end(), s) == _depend.end())
            _depend.push_back(s);

        s = "#include \"" + i->function_class() + ".h\"";
        if (find(_depend.begin(), _depend.end(), s) == _depend.end())
            _depend.push_back(s);
    }

    f << "    ";
    f << "        throw std::runtime_error(\"There is no such function.\");";
    f << "    }";
    f << "";
    f << "    std::string resolveFunction(const std::string& prefix,";
    f << "                                const std::string& name) const";
    f << "    {";
    f << "        return std::string();";
    f << "    }";
    f << "    private :";
    f << "        //std::map<std::string, __jsp_func_p> __f_map";
    f << "};";
}

string generator::el_expression(const string& el)
{
    string tmp(el.begin() + 2, el.end() - 1);
    if (_debug) {
        clog << "generator::el_expression EL:" << tmp << endl;
    }
    string ret;

#if 0
    /* EL ͤ string ⤷ string ѴǽͤǤ뤿 */

    string::size_type pos = tmp.find(":");
    if (pos != string::npos) {

        string prefix = tmp.substr(0, pos);
        tmp.erase(0, pos + 1);

        string name   = tmp.substr(0, tmp.find("("));

        func_list_t::iterator i = _tld_functions.begin();
        for (; i != _tld_functions.end(); i++) {

            if (name == i->name() && prefix == i->prefix()) {
                ret = i->func_ret();
                break;
            }
        }
    }
#endif

    if (ret.empty())
        ret = "std::string";

    string code = "sl::object_cast<" + ret + ">("
                    "__expression_evaluator.evaluate(\"" +
                    quote_to_scalar(el) + "\", sl::java::lang::Class(), " +
                    "__variable_resolver, __function_mapper))";

    return code;
}

string generator::attribute_arg(const string& line,
                                const string& setter,
                                tld_tag& tag)
{
    string tmp(line);
    bool is_el_expr = false;

    if (!tmp.compare(1, 2, "${")) {
        trim_if(tmp, is_any_of("\""));
        trim_if(tmp, is_any_of("${}"));

        if (_debug) {
            std::clog << "generator::attribute_arg EL:" << tmp << std::endl;
        }
        is_el_expr = true;
    }

    string cast_type;

    /**
     * Tag 饹ƤдؿФ
     * setter Ʊдؿ򸡺.
     *
     * ǤϤȤꤢʣΥåؿ¸ߤϥ顼ˤƤ.
     */
    functions_map_t functions = tag.functions();
    functions_map_t::iterator i = functions.find(setter);
    if (i != functions.end()) {
        if (i->first == setter) {
            if (i->second.size() == 1)
                cast_type = i->second[0];
            else
                cerr << "The number of arguments is illegal." << endl;
        }
    }

    if (cast_type.empty())
        cast_type = "std::string";
    else
        cast_type = erase_qualifier(decode_xml(cast_type));

    string code;
    if (is_el_expr) {
        code = "sl::object_cast<" + cast_type + ">("
                "__expression_evaluator.evaluate(\"" +
                quote_to_scalar(tmp) +
                "\", sl::java::lang::Class(), " +
                "__variable_resolver, __function_mapper))";
    } else {
        code = "boost::lexical_cast<" + cast_type + ">(" + tmp + ")";

        /* boost/lexical_cast.hpp  #include Ȥ¸Ƥ */
        string s = "#include <boost/lexical_cast.hpp>";
        if (find(_depend.begin(), _depend.end(), s) == _depend.end())
            _depend.push_back(s);
    }

    return string(setter + "(" + code + ");");
}

void generator::attribute_setter(const std::string& tag_name,
                                const jsp_element_ptr& e,
                                const tld_tag& tag,
                                file_type& f)
{
    bool is_dynamic = false;

    std::set<std::string> classes = tag.base_classes();
    if (classes.find("DynamicAttributes") != classes.end())
        is_dynamic = true;

    typedef std::map<std::string, std::string> string_map_t;
    for (string_map_t::const_iterator i1 = e->attribute.begin();
         i1 != e->attribute.end(); i1++)
    {
        string setter;

        if (is_dynamic) {
            setter = "setDynamicAttribute";
        } else {
            setter = i1->first;
            setter[0] = toupper(setter[0]);
            setter.insert(0, "set");
        }

        string tmp(i1->second);
        bool is_el_expr = false;

        if (!tmp.compare(1, 2, "${")) {
            trim_if(tmp, is_any_of("\""));
            trim_if(tmp, is_any_of("${}"));
            if (_debug)
                std::clog << "generator::attribute_setter EL:" << tmp << std::endl;
            is_el_expr = true;
        }

        string code;

        if (is_dynamic) {
            if (is_el_expr) {
                code = "__expression_evaluator.evaluate(\"" +
                        quote_to_scalar(tmp) +
                        "\", sl::java::lang::Class(), " +
                        "__variable_resolver, __function_mapper)";
            } else {
                code = "sl::java::lang::Object("+ tmp + ")";
            }
            code = string(setter +
                        "(std::string(), " +
                        "\"" +
                        trim_copy_if(i1->first, is_any_of("\r\n\t ")) +
                        "\", " +
                        code + ");");
        } else { 
            string cast_type;

            /**
             * Tag 饹ƤдؿФ
             * setter Ʊдؿ򸡺.
             *
             * ǤϤȤꤢʣΥåؿ¸ߤ
             * 顼ˤƤ.
             */
            functions_map_t functions = tag.functions();
            functions_map_t::iterator i2 = functions.find(setter);
            if (i2 != functions.end()) {
                if (i2->first == setter) {
                    if (i2->second.size() == 1)
                        cast_type = i2->second[0];
                    else
                        cerr << "The number of arguments is illegal." << endl;
                }
            }

            if (cast_type.empty())
                cast_type = "std::string";
            else
                cast_type = erase_qualifier(decode_xml(cast_type));

            if (is_el_expr) {
                code = "sl::object_cast<" + cast_type + ">("
                        "__expression_evaluator.evaluate(\"" +
                        quote_to_scalar(tmp) +
                        "\", sl::java::lang::Class(), " +
                        "__variable_resolver, __function_mapper))";
            } else {
                code = "boost::lexical_cast<" + cast_type + ">(" + tmp + ")";
    
                /* boost/lexical_cast.hpp  #include Ȥ¸Ƥ */
                string s = "#include <boost/lexical_cast.hpp>";
                if (find(_depend.begin(), _depend.end(), s) == _depend.end())
                    _depend.push_back(s);
            }
            code = string(setter + "(" + code + ");");
        }

        f << tag_name + "." + code;
    }
}
