/*-
 * Copyright (c) 2005 osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>

#include <cerrno>
#include <cstdio>
#include <fstream>
#include <iostream>
#include <stdexcept>
#include <string>
#include <vector>
using namespace std;

#include "compiler.h"
using namespace sj;

#if defined(__CYGWIN__)
extern "C" FILE * popen(const char *, const char *);
extern "C" int pclose(FILE *);
#endif

//
// Constructor/Destructor
//

compiler::compiler(const string& path, const string& file, const string& output)
    : _output(output)
{
#ifdef DEBUG
    cerr << __FUNCTION__ << " path:" << path << endl;
    cerr << __FUNCTION__ << " file:" << file << endl;
    cerr << __FUNCTION__ << " out :" << output << endl;
#endif
    set_target_file(path + "/" + file);

    // ǥեȤΥѥꤷƤ.
#if 0
    _include.push_back("/usr/include");
    _library.push_back("/lib");
    _library.push_back("/usr/lib");
#endif
}


//
// Member functions.
//
    
void compiler::system_compiler(const string& cc)
{
    _CC = cc;
}
    
void compiler::add_file(const string& input)
{
    _files.push_back(input);
}
    
void compiler::add_include(const string& input)
{
    _include.push_back(input);
}
    
void compiler::add_library(const string& input)
{
    _library.push_back(input);
}
    
void compiler::add_shared(const string& input)
{
    _shared.push_back(input);
}
    
void compiler::add_object(const string& input)
{
    _objects.push_back(input);
}
    
void compiler::add_option(const string& input)
{
    _options.push_back(input);
}
    
void compiler::add_ld_option(const string& input)
{
    _ld_options.push_back(input);
}
    
void compiler::set_target_file(const string& target)
{
    _target = target;
}

void compiler::create_parametor(string& output)
{
    _options.erase(std::unique(_options.begin(), _options.end()), _options.end());
    _include.erase(std::unique(_include.begin(), _include.end()), _include.end());
    _library.erase(std::unique(_library.begin(), _library.end()), _library.end());
    _shared.erase (std::unique(_shared.begin(),  _shared.end()), _shared.end());

    vector<string>::iterator i;
    for (i = _options.begin(); i != _options.end(); i++)
        output.append((*i) + " ");
    for (i = _include.begin(); i != _include.end(); i++)
        output.append((*i) + " "); 
    for (i = _library.begin(); i != _library.end(); i++)
        output.append((*i) + " ");
    for (i = _shared.begin(); i != _shared.end(); i++)
        output.append((*i) + " ");
    for (i = _ld_options.begin(); i != _ld_options.end(); i++)
        output.append((*i) + " ");
}

void do_compile(const std::string cmd)
{
#ifdef DEBUG
    cerr << "The compilation is executed." << endl;
    cerr << cmd << endl;
#endif

    int fildes[2];
    if (::pipe(fildes) != 0) {
        perror("pipe");
        throw runtime_error("It failed in the compilation.");
    }

    pid_t pid = ::fork();
    if (pid < 0) {
        throw runtime_error("It failed in the compilation.");
    } else if (pid == 0) {
        ::dup2(fildes[1], STDOUT_FILENO);
        ::close(fildes[1]);
        ::close(fildes[0]);
        ::execl(std::string(cmd).c_str(), std::string(cmd).c_str(), NULL);
        ::perror("execl");
        exit(1);
    }
    ::close(fildes[1]);

    int status;
    if (::waitpid(pid, &status, 0) == -1) {
        throw runtime_error("It failed in the compilation.");
    }

    if (!WIFEXITED(status)) {
        throw runtime_error("It failed in the compilation.");
    }

    std::string result;
    char buf[1024];

    for (;;) {
        int n;
        if ((n = ::read(fildes[0], buf, sizeof(buf))) == 0)
            break;
        if (n < 0) {
            std::string msg;

            switch (errno) {
            case EINTR :
                msg = "interupted signal is received.";
                break;
            default :
                msg = "unknown signal was received in waitpid().";
                break;
            }
            throw runtime_error("It failed in the compilation.\n" + msg);
        }
        result += string(buf, buf + n);
    }
    ::close(fildes[0]);
}

void compiler::compile(const string& output)
{
    string params;
    create_parametor(params);

    if (_CC.empty())
        _CC = "CC";

    if (!output.empty())
        _output = output;

    if (!_files.empty()) {
        vector<string>::iterator i = _files.begin();
        for (; i != _files.end(); i++)
            _target.append(" " + (*i));
    }

    string options;
    vector<string>::iterator i = _options.begin();
    for (; i != _options.end(); i++)
        options.append((*i) + " ");

    string cmd(_CC + " " + options + _target +" -o "+ _output +" "+ params);
    cmd.append(" 2>&1");
#if 0
    do_compile(cmd);
    return;
#else
#ifdef DEBUG
    cerr << "The compilation is executed." << endl;
    cerr << cmd << endl;
#endif
    char buf[4098];
    FILE* fp = popen(cmd.c_str(), "r");
    if (!fp)
        throw runtime_error("It failed in the compilation.");

    std::string result;
    while (!feof(fp)) {
        fgets(buf, sizeof(buf), fp);
        result.append(buf);
    }

    int ret = pclose(fp);
    if (ret != 0) {
        throw compile_error("It failed in the compilation. \n" + result);
    }
#endif
}
