/*-
 * Copyright (c) 2005 Masashi Osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: xml_parser.cpp,v 1.4 2007/08/06 13:21:59 cvsuser Exp $
 */
#include <fstream>
#include <iostream>

#include <sl/xml/xml_parser.h>
using namespace sl::xml;

//
// Global variables.
//

xml_tag *__current;

//
// Constructor/Destructor
//

xml_parser::xml_parser()
{
	init();
}


xml_parser::xml_parser(const std::string &file_path) throw(xml_parser_error)
{
	init();
	read(file_path);
}

xml_parser::xml_parser(const std::string::iterator& first,
					   const std::string::iterator& last)
	throw(xml_parser_error)
{
	std::string file(first, last);
	init();
	parse(file);
}

xml_parser::~xml_parser()
{
	clear(_root_tag);
	destroy();
}


//
// Member functions.
//

void xml_parser::read(const std::string &file_path) throw(xml_parser_error)
{
	std::ifstream ifs(file_path.c_str());
	if (!ifs) {
		std::string msg = "xml_parser_error, ";
		throw xml_parser_error(msg + file_path + " is not found");
	}

	std::string file((std::istreambuf_iterator<char>(ifs)),
					  std::istreambuf_iterator<char>());
	ifs.close();

	parse(file);
}


void xml_parser::init() throw(xml_parser_error)
{
	_root_tag._name = "Root";

	_parser = XML_ParserCreate(NULL);
	if (!_parser)
		throw xml_parser_error(std::string("xml_parser_error exception, "
										   "Base library error: ") +
							   "XML_ParserCreate error");

	__current = &_root_tag;

	XML_SetUserData(_parser, &_root_tag);
	XML_SetElementHandler(_parser, start_tag, end_tag);
	XML_SetDefaultHandler(_parser, data_tag);
	XML_SetCommentHandler(_parser, com_tag);
}


void xml_parser::destroy()
{
	if (_parser) {
		XML_ParserFree(_parser);
		_parser = 0;
	}
}


void xml_parser::clear(xml_tag &e)
{
	tag_list_t tlist = e.get();
	tag_list_t::iterator i = tlist.begin();
	for (; i != tlist.end(); i++)
		clear(*i);
}


void xml_parser::parse(const std::string &file) throw(xml_parser_error)
{
	int n = 0;
	int status = XML_Parse(_parser, file.c_str(), file.length(), n);
	if (status == XML_STATUS_ERROR) {
		std::string msg(std::string("xml_parser_error exception, ") + 
						XML_ErrorString(XML_GetErrorCode(_parser)));
		throw xml_parser_error(msg);
	}
}


xml_tag xml_parser::get()
{
	return _root_tag.get().at(0);
}


// static
void xml_parser::start_tag(void * /*data*/, const char *name, const char **attr)
{
//	xml_tag *that = static_cast<xml_tag *>(data);

	xml_tag *child = new xml_tag();
	child->_name = name;

	for (int i = 0; attr[i]; i+= 2)
		child->attributes(string_map_t::value_type(attr[i], attr[i + 1]));

	__current->child_tag(child);
	child->_parent = __current;
	__current = child;
}


// static
void xml_parser::end_tag(void * /*data*/, const char * /*name*/)
{
//	xml_tag *that = static_cast<xml_tag *>(data);

	__current = __current->_parent;
}


// static
void xml_parser::data_tag(void * /*data*/, const XML_Char *s, int len)
{
//	xml_tag *that = static_cast<xml_tag *>(data);

	std::string input(s, len);

#if 0
	// ֤ԤϺޤ.
	std::string::size_type pos;
	while ((pos = input.find('\t')) != std::string::npos ||
		   (pos = input.find('\n')) != std::string::npos)
		input.erase(pos, 1);
#endif
	if (input.empty())
		return;

	__current->_value.append(input);
}


// static
void xml_parser::com_tag(void * /*data*/, const XML_Char * /*s*/)
{
	// xml_tag *that = static_cast<xml_tag *>(data);
}
