/*-
 * Copyright (c) 2005 Masashi Osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef SL_XML_CLASS_PARSER_CLASS_DEF_GRAMMAR_H
#define SL_XML_CLASS_PARSER_CLASS_DEF_GRAMMAR_H

#include <iostream>
#include <string>
#include <boost/algorithm/string/find.hpp>
#include <boost/bind.hpp>

#ifdef __DEBUG__
#define BOOST_SPIRIT_DEBUG
#endif
#define BOOST_SPIRIT_THREADSAFE
#include <boost/spirit.hpp>
#include <boost/spirit/phoenix.hpp>
#include <boost/spirit/utility/regex.hpp>
using namespace std;
using namespace boost;
using namespace boost::spirit;

#include <sl/xml/class_parser/callback_functors.h>

namespace sl {
namespace xml {

/**
 * @class class_def_grammar.
 *
 * ʸϤΥ롼饹.
 * namespace class дؿȰ¤βϥ롼Ǥ.
 *
 */
class class_def_grammar : public grammar<class_def_grammar> {
public :

    struct string_closure : closure<string_closure, std::string>
    {
        member1 val;
    };

    typedef string_closure                            __closure_type;

    struct append_a {
        typedef void result_type;

        void operator()(std::string& lhs,
                        const char* first, const char* last) const
        {
#ifdef __DEBUG__
std::cerr << "append1:[" << lhs << "][" << std::string(first, last) << "]"
      << std::endl;
#endif
            std::string rhs(first, last);

            if (!lhs.empty()) {
                iterator_range<string::iterator> t = find_tail(lhs, 2);
                if (std::string(t.begin(), t.end()) != "::" && rhs != "::")
                {
                    lhs.append(" ");
                }
            }
            lhs.append(rhs);
        }

        void operator()(std::string& lhs, const std::string& rhs) const
        {
#ifdef __DEBUG__
std::cerr << "append2:[" << lhs << "][" << rhs << "]" << std::endl;
#endif
            if (!lhs.empty()) {
                iterator_range<string::iterator> t = find_tail(lhs, 2);
                if (std::string(t.begin(), t.end()) != "::" && rhs != "::")
                {
                    lhs.append(" ");
                }
            }
            lhs.append(rhs);
        }

        void operator()(std::string& lhs, const char& ch) const
        {
#ifdef __DEBUG__
std::cerr << "append2:[" << lhs << "][" << ch << "]" << std::endl;
#endif
            if (!lhs.empty())
                lhs.append(" ");
            lhs.append(1, ch);
        }
    };

    class_def_grammar(callback_functors& f)
        : _space_f(f._space_f),
          _class_f(f._class_f),
          _based_f(f._based_f),
          _funct_f(f._funct_f),
          _arg_type_f(f._arg_type_f),
          _arg_name_f(f._arg_name_f),
          _append_f(append_a())
    { }

    phoenix::functor<callback_functors::space_functor>        _space_f;
    phoenix::functor<callback_functors::class_functor>        _class_f;
    phoenix::functor<callback_functors::based_functor>        _based_f;
    phoenix::functor<callback_functors::funct_functor>        _funct_f;
    phoenix::functor<callback_functors::arg_type_functor>    _arg_type_f;
    phoenix::functor<callback_functors::arg_name_functor>    _arg_name_f;

    phoenix::functor<append_a>                                _append_f;

    template <typename ScannerT>
    struct definition
    {

#define Namespace    str_p("namespace")
#define Class        (str_p("class")|str_p("struct")|str_p("union"))
#define Virtual        str_p("virtual")
#define Inline        str_p("inline")
#define Explicit    str_p("explicit")
#define Template    str_p("template")
#define Enum        str_p("enum")
#define Public        str_p("public")
#define Protected    str_p("protected")
#define Private        str_p("private")
#define Typename    str_p("typename")
#define Using        str_p("using")
#define Operator    str_p("operator")
#define Typedef        str_p("typedef")
#define Throw        str_p("throw")

        rule<ScannerT> top;
        rule<ScannerT, typename __closure_type::context_t>
            _access,
            _arg,
            _args,
            _block,
            _class,
            _comment,
            _const,
            _constructor,
            _derived,
            _destructor,
            _enum,
            _expr,
            _func,
            _func_arg,
            _func_body,
            _func_spec,
            _general,
            _namespace,
            _operator,
            _preprocess,
            _pure,
            _qualifier,
            _s_type,
            _static,
            _string,
            _symbol,
            _t_type,
            _template,
            _throw,
            _type,
            _typedef,
            _using,
            _var_def,
            _var_init,
            _volatile,
            program;

        definition(const class_def_grammar& self)
        {
            using phoenix::arg1;
            using phoenix::arg2;
            using boost::bind;

            const char* str_r("[a-zA-Z0-9_ <>/.\"]+");
            const char* expr("[\\[\\]\\(\\)a-zA-Z0-9_\\-\\*+=><'\\.,;:/&!\\| ]+");

            top  = program
                ;
            program
                = *(_namespace | _class | _general);
            _general
                = _comment
                | _preprocess
                | _using
                | _enum
                | _typedef >> ';'
                | _var_def >> ';'
                | _var_init >> ';'
                | _func
                ;
            _namespace
                = Namespace >> _symbol[self._space_f(arg1)] >>
                    '{' >>
                        *(_namespace
                        | _class
                        | _general) >>
                    ch_p('}') [self._space_f()]
                ;
            _class
                = *(_template) >>
                    Class >> _symbol[self._class_f(arg1)] >>
                    !(ch_p(':') >> _derived) >>
                    '{' >>
                        *( _access >> ':'
                        | _constructor
                        | _destructor
                        | _class
                        | _general) >>
                    ch_p('}') [self._class_f()] >> ';'
                ;
            _derived
                = !(Virtual) >> !(_access) >>
                    _type[self._based_f(arg1)] >> *(',' >> _derived)
                ;
            _template
                = Template >>
                    '<' >> +(str_p("class")|str_p("typename")) >>
                        _symbol >> *(_template) >>
                    '>'
                ;
            _access
                = Public
                | Protected
                | Private
                ;
            _constructor
                = !(_template | Explicit) >>
                    _type >>
                    _func_arg >>
                    (';' | !(':' >> _var_init >> *(',' >> _var_init)) >>
                    _func_body)

                ;
            _destructor
                = !(_template | Virtual) >>
                    '~' >> _type >>
                    _func_arg >>
                    (!_pure >> ';' | _func_body)
                ;
            _func
                = !(_template | _func_spec) >>
                    !_const[_func.val += arg1] >>
                    _type [_func.val += arg1] >>
                    _type [self._funct_f(arg1)] >>
                    _func_arg >>
                    (!_pure >> ';' | _func_body)[self._funct_f()]
                ;
            _func_spec
                = Virtual
                | Inline
                | Explicit
                ;
            _func_arg
                = '(' >> !_args >> ')' >> !_const >> !_throw
                ;
            _func_body
                = '{' >> _expr >> '}'
                ;
            _expr
                = *(regex_p(expr) | _string | _block)
                ;
            _block
                = '{' >> _expr >> '}'
                ;
            _preprocess
                = comment_p("#if", "#endif")
                | ch_p('#') >> regex_p(str_r)
                ;
            _enum
                = Enum >> '{' >> *(_symbol >> ',') >> '}' >> ';'
                ;
            _using
                = Using >> Namespace >> *_type >> ';'
                | Using >> *_type >> ';'
                ;
            _comment
                = comment_p("/*", "*/")
                | comment_p("//")
                ;
            _operator
                = _type >> Operator >> "=="
                | _type >> Operator >> "="
                | _type >> Operator >> "()"
                | Operator >> _type >> "()"
                ;
            _qualifier
                = _const   [_qualifier.val = arg1]
                | _volatile[_qualifier.val = arg1]
                ;
            _type
                = !_static   [self._append_f(_type.val, arg1)] >>
                  *_qualifier[self._append_f(_type.val, arg1)] >>
                  _s_type    [self._append_f(_type.val, arg1)] >>
                  !(_t_type  [self._append_f(_type.val, arg1)] >>
                    !(str_p("::") [self._append_f(_type.val, arg1, arg2)] >>
                      _s_type     [self._append_f(_type.val, arg1)])) >>
                  *(ch_p('*')[self._append_f(_type.val, arg1)] |
                    ch_p('&')[self._append_f(_type.val, arg1)]) >>
                  !_const
                ; 
            _t_type
                = ch_p('<') [self._append_f(_t_type.val, arg1)] >>
                    _type   [self._append_f(_t_type.val, arg1)] >>
                    *(ch_p(',')[self._append_f(_t_type.val, arg1)] >>
                      _type [self._append_f(_t_type.val, arg1)]) >>
                  ch_p('>') [self._append_f(_t_type.val, arg1)]
                ;
            _s_type
                = !str_p("::") [self._append_f(_s_type.val, arg1, arg2)] >>
                  _symbol      [self._append_f(_s_type.val, arg1)] >>
                  *(str_p("::")[self._append_f(_s_type.val, arg1, arg2)] >>
                    _symbol    [self._append_f(_s_type.val, arg1)])
                ;
            _symbol
                //= nondigit >> *(nondigit | digit)
                = regex_p("[a-zA-Z_]([a-zA-Z0-9_]*)")
                    [self._append_f(_symbol.val, arg1, arg2)]
                ;
            _string
                = regex_p("\"[[:print:]]*\"")
                ;
            _var_def
                = _type >> _symbol >> *(',' >> _symbol)
                | _type >> '(' >> *(ch_p('*')) >> _symbol >> ')' >>
                    _func_arg
                ;
            _var_init
                = _s_type >> '(' >>
                    (_s_type >> !_symbol | regex_p("[0-9]+") | _string) >>
                  ')'
                | _type >> _symbol >> '=' >> !ch_p('&') >> (!_type >> _symbol | _string)
                ;
            _args
                = _arg >> *(',' >> _arg)
                ;
            _arg
                = _type   [self._arg_type_f(arg1)] >>
                  !_symbol[self._arg_name_f(arg1)]
                ;
            _typedef
                = Typedef >> _type >> _symbol;
                ;
            _throw
                = Throw >> '(' >> !_type >> *(',' >> _type) >> ')'
                ;
            _pure
                = ch_p('=') >> ch_p('0')
                ;
            _const
                = str_p("const")[self._append_f(_const.val, arg1, arg2)]
                ;
            _volatile
                = str_p("volatile")[self._append_f(_volatile.val, arg1, arg2)]
                ;
            _static
                = str_p("static")[self._append_f(_static.val, arg1, arg2)]
                ;

            BOOST_SPIRIT_DEBUG_NODE(_access);
            BOOST_SPIRIT_DEBUG_NODE(_arg);
            BOOST_SPIRIT_DEBUG_NODE(_args);
            BOOST_SPIRIT_DEBUG_NODE(_block);
            BOOST_SPIRIT_DEBUG_NODE(_class);
            BOOST_SPIRIT_DEBUG_NODE(_comment);
            BOOST_SPIRIT_DEBUG_NODE(_const);
            BOOST_SPIRIT_DEBUG_NODE(_constructor);
            BOOST_SPIRIT_DEBUG_NODE(_derived);
            BOOST_SPIRIT_DEBUG_NODE(_destructor);
            BOOST_SPIRIT_DEBUG_NODE(_enum);
            BOOST_SPIRIT_DEBUG_NODE(_expr);
            BOOST_SPIRIT_DEBUG_NODE(_func);
            BOOST_SPIRIT_DEBUG_NODE(_func_arg);
            BOOST_SPIRIT_DEBUG_NODE(_func_body);
            BOOST_SPIRIT_DEBUG_NODE(_func_spec);
            BOOST_SPIRIT_DEBUG_NODE(_namespace);
            BOOST_SPIRIT_DEBUG_NODE(_operator);
            BOOST_SPIRIT_DEBUG_NODE(_preprocess);
            BOOST_SPIRIT_DEBUG_NODE(_qualifier);
            BOOST_SPIRIT_DEBUG_NODE(_s_type);
            BOOST_SPIRIT_DEBUG_NODE(_static);
            BOOST_SPIRIT_DEBUG_NODE(_string);
            BOOST_SPIRIT_DEBUG_NODE(_symbol);
            BOOST_SPIRIT_DEBUG_NODE(_t_type);
            BOOST_SPIRIT_DEBUG_NODE(_template);
            BOOST_SPIRIT_DEBUG_NODE(_throw);
            BOOST_SPIRIT_DEBUG_NODE(_type);
            BOOST_SPIRIT_DEBUG_NODE(_typedef);
            BOOST_SPIRIT_DEBUG_NODE(_using);
            BOOST_SPIRIT_DEBUG_NODE(_var_def);
            BOOST_SPIRIT_DEBUG_NODE(_var_init);
            BOOST_SPIRIT_DEBUG_NODE(_volatile);
            BOOST_SPIRIT_DEBUG_NODE(_general);
            BOOST_SPIRIT_DEBUG_NODE(program);
        }

        const rule<ScannerT>& start() const { return top; }
    };
};

} // namespace xml
} // namespace sl

#endif // SL_XML_CLASS_PARSER_CLASS_DEF_GRAMMAR_H
