/*-
 * Copyright (c) 2005 Masashi Osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef SL_XML_CLASS_PARSER_CALLBACK_FUNCTORS_H
#define SL_XML_CLASS_PARSER_CALLBACK_FUNCTORS_H

#include <iostream>
#include <string>
#include <boost/algorithm/string/replace.hpp>

namespace sl {
namespace xml {

/**
 * 괺 '&' '<' '>' 򥨥פ.
 *
 * ʤ餳ؿϤɤäΥ͡ॹڡˤޤȤ٤
 * Ȥꤢ֤.
 */
std::string& escape(std::string& s)
{
    boost::replace_all(s, "&", "&amp;");
    boost::replace_all(s, "<", "&lt;");
    boost::replace_all(s, ">", "&gt;");
    return s;
}


/**
 * @class callback_functors.
 *
 * class_def_grammar ιʸ˥ФȤƻĳ
 * ե󥯥饹¹Ԥޤ.
 * 줾Υե󥯥줾Ƥˤä XML_CLASS_PARSER ϤƤޤ.
 * ºݡXML_CLASS_PARSERϤ⤦̽ФäޤΤȤ֤.
 */
class callback_functors {
public :
    /** Constructor.
     * ФΥ饹󥹥󥹤˼ȤϿƤ.
     */
    callback_functors()
        : _depth(0),
          _space_f(*this),
          _class_f(*this),
          _based_f(*this),
          _funct_f(*this),
          _arg_type_f(*this),
          _arg_name_f(*this)
    {
        add_document("<?xml version=\"1.0\"?>");
        add_document("<definition>");
        node_depth_up();
    }

    /**
     * ̤XML_CLASS_PARSER.
     *
     * @return std::string.
     */
    std::string xml()
    {
        return _result + "</definition>";
    }

    void node_depth_up()
    {
        ++_depth;
    }

    void node_depth_down()
    {
        --_depth;
    }

    void add_document(const std::string& s)
    {
        for (int i = 0; i < _depth; i++)
            _result.append("    ");
        _result.append(s);
        _result.append("\n");
    }

private :
    std::string        _result;
    int                _depth;

public :

    // -------------------------------------------------------------------
    //
    //
    class space_functor {
    public :
        space_functor(callback_functors& parent) : _parent(parent) { }

        typedef void result_type;

        void operator()(const std::string& s=std::string())
        {
            if (!s.empty()) {
                std::string tmp(s);
                std::string::size_type pos = tmp.find(' ');
                if (pos != std::string::npos)
                    tmp.erase(pos);

                _parent.add_document("<namespace name=\"" +
                                     escape(tmp) + "\">");
                _parent.node_depth_up();
#ifdef __DEBUG__
                std::cerr << "namespace Enter [" << tmp << "]" << std::endl;
#endif
            } else {
                _parent.node_depth_down();
                _parent.add_document("</namespace>");
#ifdef __DEBUG__
                std::cerr << "namespace Exit" << std::endl;
#endif
            }
        }

    private :
        callback_functors& _parent;
    };


    // -------------------------------------------------------------------
    //
    //
    class class_functor {
    public :
        class_functor(callback_functors& parent) : _parent(parent) { }

        typedef void result_type;

        void operator()(const std::string& s=std::string())
        {
            if (!s.empty()) {
                std::string tmp(s);
                std::string::size_type pos = tmp.find(' ');
                if (pos != std::string::npos)
                    tmp.erase(pos);

                _parent.add_document("<class name=\"" +
                                     escape(tmp) + "\">");
                _parent.node_depth_up();
#ifdef __DEBUG__
                std::cerr << "class Enter [" << tmp << "]" << std::endl;
#endif
            } else {
                _parent.node_depth_down();
                _parent.add_document("</class>");
#ifdef __DEBUG__
                std::cerr << "class Exit" << std::endl;
#endif
            }
        }
    private :
        callback_functors& _parent;
    };


    // -------------------------------------------------------------------
    //
    //
    class based_functor {
    public :
        based_functor(callback_functors& parent) : _parent(parent) { }

        typedef void result_type;

        void operator()(const std::string& s)
        {
            std::string tmp(s);
            std::string::size_type pos = tmp.find(' ');
            if (pos != std::string::npos)
                tmp.erase(pos);

            _parent.add_document("<base-class name=\"" +
                                 escape(tmp) + "\"/>");
#ifdef __DEBUG__
            std::cerr << "based Enter [" << tmp << "]" << std::endl;
#endif
        }
    private :
        callback_functors& _parent;
    };


    // -------------------------------------------------------------------
    //
    //
    class funct_functor {
    public :
        funct_functor(callback_functors& parent) : _parent(parent) { }

        typedef void result_type;

        void operator()(const std::string& s=std::string())
        {
            if (!s.empty()) {
                std::string tmp(s);
                std::string::size_type pos = tmp.find(')');
                if (pos != std::string::npos)
                    tmp.erase(pos + 1);

                _parent.add_document("<function name=\"" +
                                     escape(tmp) + "\">");
                _parent.node_depth_up();
#ifdef __DEBUG__
                std::cerr << "funct Enter [" << tmp << "]" << std::endl;
#endif
            } else {
                _parent.node_depth_down();
                _parent.add_document("</function>");
#ifdef __DEBUG__
                std::cerr << "funct Exit" << std::endl;
#endif
            }
        }
    private :
        callback_functors& _parent;
    };


    // -------------------------------------------------------------------
    //
    //
    class arg_type_functor {
    public :
        arg_type_functor(callback_functors& parent)
            : _parent(parent) { }

        typedef void result_type;

        void operator()(const std::string& s)
        {
            std::string tmp(s);
            _parent.add_document("<arg>" + escape(tmp) + "</arg>");
#ifdef __DEBUG__
            std::cerr << "typed Enter [" << tmp << "]" << std::endl;
#endif
        }
    private :
        callback_functors& _parent;
    };


    // -------------------------------------------------------------------
    //
    //
    class arg_name_functor {
    public :
        arg_name_functor(callback_functors& parent)
            : _parent(parent) { }

        typedef void result_type;

        void operator()(const std::string& s)
        {
#ifdef __DEBUG__
            std::cerr << "typed Enter [" << s << "]" << std::endl;
#endif
        }
    private :
        callback_functors& _parent;
    };

    space_functor _space_f;
    class_functor _class_f;
    based_functor _based_f;
    funct_functor _funct_f;
    arg_type_functor _arg_type_f;
    arg_name_functor _arg_name_f;
};


} // namespace xml
} // namespace sl


#endif // SL_XML_CLASS_PARSER_CALLBACK_FUNCTORS_H
