/*-
 * Copyright (c) 2005 Masashi Osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: stacktrace.h,v 1.1 2008/03/01 03:28:31 cvsuser Exp $
 */

#ifndef SL_SYS_STACKTRACE_H
#define SL_SYS_STACKTRACE_H

#ifdef SL_STACKTRACE
#include <execinfo.h> /* "libexecinfo" */
#endif

#include <iostream>
#include <sstream>
#include <string>

namespace sl { namespace sys {

#ifdef SL_STACKTRACE
namespace detail {

	inline std::string demangle(const std::string& function)
	{
		if (function.empty() || function == "???")
			return function;

		if (function[0] != '_' || function[1] != 'Z')
			return function;

		char buffer[1024];
		std::string com = "/usr/bin/c++filt " + function;

		FILE *fp = popen(com.c_str(), "r");
		fgets(buffer, sizeof(buffer), fp);
		pclose(fp);

		std::string temp(buffer, strlen(buffer) - 1);

		std::string::size_type sep1 = temp.rfind('(');
		temp.erase(sep1);
		temp.append(" ()");

		return temp;
	}


	/**
	 * FreeBSD
	 * "0x01234567 <function+offset> at filename.so"
	 */
	inline std::string conv_bsd(const std::string& s)
	{
		std::string::size_type sep1 = s.find(' ');
		std::string::size_type sep2 = s.find('<');
		std::string::size_type sep3 = s.find('+', sep2);
		std::string::size_type sep4 = s.find('>', sep3);

		std::string address  = s.substr(0, sep1);
		std::string function = s.substr(sep2 + 1, sep3 - sep2 - 1);
		std::string offset   = s.substr(sep3 + 1, sep4 - sep3 - 1);
		std::string file	 = s.substr(sep4 + 1);

		if (address.empty())
			address = "0x????????";
		if (function.empty())
			function = "???";
		if (offset.empty())
			offset = " ---";

		return	address + "\t" +
				((function.length() >= 2 &&
				function[0] == '_' && function[1] == 'Z') ?
				demangle(function) : function) + " +" + offset + file;
	}

	inline std::string conv_linux(const std::string &s)
	{
		using namespace std;

		if (s.length() <= 10)
			return s;

		string address;
		string function;
		string file;

		string::size_type first;
		string::size_type second;

		/* /path/to/filename.so(function+offset) [0x1234567x] */

		first  = s.find("(");
		second = s.find("+", first);
		string::size_type addr_p = s.find("[");

		address = s.substr(addr_p);

		if (first != string::npos)
			file = s.substr(0, first);
		else
			file = s.substr(0, addr_p);

		if (first != string::npos && second != string::npos)
			function = s.substr(first + 1, second - (first + 1));
		else
			function = "???";

		while (address.length() < 11)
			address += " ";

		return address + " `" + demangle(function) + "` at " + file;
	}

} // namespace sl::sys::detail
#endif // SL_STACKTRACE

	inline std::string stack_trace()
	{
		std::ostringstream ss;

#ifdef SL_STACKTRACE
		void *	array[100];
		size_t	size;
		char **	symbols;
		size_t	i;
	
		size = backtrace(array, 100);
		symbols = backtrace_symbols(array, size);
	
		for (i = 0; i < size; i++) {
			ss << "#" << i << "  ";
#if defined(__linux__) || defined(__linux) || defined(linux)
			ss << detail::conv_linux(symbols[i]) << "\n";
#elif defined(__FreeBSD__)
			ss << detail::conv_bsd(symbols[i]) << "\n";
#else
			ss << symbols[i] << std::endl;
#endif
		}
#else
		ss << "cannot access stack information";
#endif // SL_STACKTRACE

		return ss.str();
	}

	inline void print_trace()
	{
		std::string result = stack_trace();
		std::cerr << result << std::endl;
	}


} } // namespace sl::sys


#endif // SL_SYS_STACKTRACE_H
