/*-
 * Copyright (c) 2005 Masashi Osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: class_loader.h,v 1.3 2006/10/29 10:03:13 cvsuser Exp $
 */

#ifndef SL_SYS_CLASS_LOADER_H
#define SL_SYS_CLASS_LOADER_H

#include <dlfcn.h>
#include <stdexcept>
#include <string>

#include <sl/sys/regist_loader.h>

namespace sl { 
  namespace sys { 
	
namespace {

	const std::string _create_key = "__create_";
	const std::string _delete_key = "__delete_";

	const std::string __AlradyLoaded= "alrady loaded other shared object.";
	const std::string __Unloaded	= "unloaded shared object yet.";
	const std::string __PathIsEmpty	= "path is empty.";
	const std::string __Open		= "dlopen error:";
	const std::string __Symbol		= "dlsym error:";
	const std::string __Close		= "dlclose error.";
	
} // End of namespace.


/**
 * ñΥɥ֥ȤΥɤȡ
 * Υ֥ȤϿƤ륯饹Υ󥹥󥹤.
 * Ʊ̾ΤΥɥ֥Ȥʾɤϡ
 *   ܤ˥ɤɥ֥Ȥĥ󥹥󥹤
 *   ˴ƤܤΥɤԤ.
 *   ܤΥ󥹥󥹤˴ʤܤξȿǤʤ.
 */
class class_loader {
public :
	/** Constructor. */
	class_loader() throw()
		: _mode(RTLD_LAZY), _loaded(false), _pointer(0) {}

	/**
	 * Constructor.
	 * ɥ֥ȤΥɤԤޤ.
	 * @see	load_shared_object()
	 */
	class_loader(const std::string &path)
		throw(std::logic_error, std::invalid_argument, std::runtime_error)
		: _mode(RTLD_NOW), _loaded(false), _pointer(0)
	{
		load_shared_object(path);
	}

	/**
	 * Destructor.
	 * @see	unload_shared_object().
	 */
	~class_loader()
		throw(std::logic_error, std::runtime_error)
	{
		unload_shared_object();
	}

	/**
	 * ɥ֥ȤΥɤԤޤ.
	 * @param		s	ɥ֥ȤΥѥ.
	 * @exception	logic_error			˥ɥ֥ȤɤƤ.
	 *				invalid_argument	ѥ.
	 *				runtime_error		dlopen()顼֤ޤ.
	 */
	void load_shared_object(const std::string &s)
		throw(std::logic_error, std::invalid_argument, std::runtime_error)
	{
		if (_loaded)
			throw std::logic_error(__AlradyLoaded);

		if (s.empty())
			throw std::invalid_argument(__PathIsEmpty);

		void *p = dlopen(s.c_str(), _mode);
		if (!p) {
			const char *msg;
			if ((msg = dlerror()) != NULL)
				throw std::runtime_error(__Open + msg);
			throw std::runtime_error(__Open + s);
		}
		_path    = s;
		_pointer = p;
		_loaded  = true;
	}

	/**
	 * ɥ֥ȤΥɤԤޤ.
	 * @exception	logic_error		DSO򥢥ɲǽʾ֤Ǥʤ.
	 *				runtime_error	dlclose()顼֤ޤ.
	 */
	void unload_shared_object()
		throw(std::logic_error, std::runtime_error)
	{
		if (!_loaded && !_pointer)
			throw std::logic_error(__Unloaded);

		if (dlclose(_pointer) < 0) 
			throw std::runtime_error(__Close);

		_pointer = 0;
		_loaded = false;
	}

	/**
	 * 饹󥹥󥹤ޤ.
	 * @param		symbol			饹̾.
	 * @exception	logic_error		󥹥󥹤ǽʾ֤ǤϤʤ.
	 *				runtime_error	dlsym()顼֤ޤ.
	 * @return		TΥݥ.
	 * class_loader󥹥󥹤Ƥ֤줿ݥ󥿤Ϻޤ.
	 * deleteˤäƺԤɬפޤ.
	 */
	template <class T>
	T *instance(const std::string &symbol)
		throw(std::logic_error, std::runtime_error)
	{
		if (!_loaded || !_pointer)
			throw std::logic_error(__Unloaded);

		typedef T *(func_t)();

		func_t *creater =
					(func_t*)dlsym(_pointer,(_create_key + symbol).c_str());
		if (!creater) {
			const char *msg;
			if ((msg = dlerror()) != NULL)
				throw std::runtime_error(__Symbol + msg);
			throw std::runtime_error(__Symbol + symbol);
		}

		return creater();
	}

private :
	class_loader(const class_loader &);
	class_loader &operator=(const class_loader &);

	int			_mode;
	bool			_loaded;
	std::string		_path;
	void *			_pointer;
};


  } // namespace sys
} //  namespace sl

#endif // SL_SYS_CLASS_LOADER_H
