/*-
 * Copyright (c) 2008 Masashi Osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef SL_OBJECT_STRING_RELATE_CAST_HPP
#define SL_OBJECT_STRING_RELATE_CAST_HPP

#include <string>
#include <stdexcept>
#include <boost/type_traits.hpp>
#include <boost/utility/enable_if.hpp>

#include <sl/object/cast.hpp>

namespace sl {

/**
 * const char* => std::string.
 */
template <>
inline std::string& object_cast<std::string>(object& o)
    throw(std::bad_cast)
{
    if (!o)
        throw std::bad_cast();

    std::string* p1 = object_pointer_cast<std::string>(o);
    if (p1)
        return *p1;

    const char** p2 = object_pointer_cast<const char*>(o);
    if (p2) {
        o = object(std::string(*p2));
        return object_cast<std::string>(o);
    }
    throw std::bad_cast();
}


template <>
inline std::string object_cast<std::string>(const object& o)
    throw(std::bad_cast)
{
    if (!o)
        throw std::bad_cast();

    std::string* p1 = object_pointer_cast<std::string>(o);
    if (p1)
        return *p1;

    const char** p2 = object_pointer_cast<const char*>(o);
    if (p2)
        return std::string(*p2);

    throw std::bad_cast();
}

/**
 * std::string => const char*
 */
template <>
inline const char*& object_cast<const char*>(object& o)
    throw(std::bad_cast)
{
    if (!o)
        throw std::bad_cast();

    const char** p1 = object_pointer_cast<const char*>(o);
    if (p1)
        return *p1;

    std::string* p2 = object_pointer_cast<std::string>(o);
    if (p2) {
        o = object(p2->c_str());
        return object_cast<const char*>(o);
    }
    throw std::bad_cast();
}


template <>
inline const char* object_cast<const char*>(const object& o)
    throw(std::bad_cast)
{
    if (!o)
        throw std::bad_cast();

    const char** p1 = object_pointer_cast<const char*>(o);
    if (p1)
        return *p1;

    std::string* p2 = object_pointer_cast<std::string>(o);
    if (p2)
        return p2->c_str();

    throw std::bad_cast();
}

} // namespace sl

#endif  // SL_OBJECT_STRING_RELATES_CAST_HPP
