/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: reader.cpp,v 1.2 2008/01/14 14:48:43 cvsuser Exp $
 */

#include <sys/select.h>
#include <sys/types.h>
#include <unistd.h>

#include <sl/net/http/http_function.h>
#include <sl/net/http/http_error.h>
#include <sl/net/http/message.h>
#include <sl/net/http/reader.h>
using namespace sl::net::http;

#define NEXT        0
#define CONTENT     1
#define FINISH      2
#define ERROR       3
#define BUF_LEN     4092


reader::reader()
{
	start();
}

int reader::read(int fd, message& msg,
					  time_t header_timeout, time_t body_timeout)
{
	if (_start == false)
		return 0;

	std::string remains;
	bool loop = true;

	while (_start && loop) {
		if (!fd_select(fd, header_timeout))
			return -1;

		char buf[BUF_LEN];
		int ret = _read(fd, buf, sizeof(buf));
		if (ret <= 0)
			return -2;

		switch (partial_reading(remains, std::string(buf, buf+ret), msg)) {
		case NEXT :
			continue;
		case FINISH :
			loop = false;
			break;
		case ERROR :
			return -3;
		}
	}

	int body_length = content_length(msg);

	// Message-Body ʤнλ.
	if (body_length <= 0)
		return 1;

	int last_length;

	// Хåե Message-Body 礭Ϥʾɤ߹ޤʤ.
	if (body_length > BUF_LEN) {
		last_length = BUF_LEN - remains.length();
		msg.attribute("warning", "content body was too large");

	//  Message-Body ɤ߹ߤλƤнλ.
	} else if (static_cast<size_t>(body_length) == remains.length()) {
		msg.msg_body(remains);
		return 1;

	// Ĥ Message-Body Ф.
	} else {
		last_length = body_length - remains.length();
	}

	if (last_length < 0) {
		remains.erase(remains.length() + last_length);
		msg.msg_body(remains);
		return 3;
	}

	if (!fd_select(fd, body_timeout))
		return -4;

	//
	// Message-Body  last_length Хɤ߹.
	//
	do {
		char buf[BUF_LEN];
		int ret = _read(fd, buf, last_length);
		if (ret < 0)
			return -5;

		remains.append(buf, buf+ret);
		if (ret == 0 || remains.length() == BUF_LEN)
			break;
		else
			last_length -= ret;
	} while (true);

	msg.msg_body(remains);
	return 2;
}

int reader::partial_reading(std::string& remains,
								 const std::string& newbuf, message& msg)
{
	remains.append(newbuf);
	std::string::size_type pos;
	int end;
	for (;;) {
		if ((pos = remains.find("\r\n")) == std::string::npos)
			if ((pos = remains.find("\n")) == std::string::npos)
				break;
			else
				end = 1;
		else
			end = 2;

		if (pos == 0) {
			remains.erase(0, pos + end);
			if (!msg.start_line().empty())
				return FINISH;

			continue;
		}
		msg.add_line(remains.substr(0, pos));
		remains.erase(0, pos + end);
	}
	return NEXT;
}

int reader::partial_body_reading(std::string& chunk,
									  message& msg, size_t length)
{
	msg.add_body(chunk);
	return (length <= msg.msg_body().length()) ? FINISH : NEXT;
}

bool reader::fd_select(int fd, time_t timeout)
{
	while (_start) {
		fd_set  fds;
		FD_ZERO(&fds);
		FD_SET(fd, &fds);

		struct timeval t;
		t.tv_sec = 1;
		t.tv_usec= 0;
		if (::select(FD_SETSIZE, &fds, NULL, NULL, &t) == -1)
			return false;

		if (FD_ISSET(fd, &fds))
			break;

		if (--timeout == 0) 
			return false;
	}

	return true;
}

int reader::_read(int fd, char* buffer, size_t size)
{
	return ::read(fd, buffer, size);
}

void reader::start()
{
	_start = true;
}

void reader::stop()
{
	_start = false;
}
