/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: message.cpp,v 1.12 2008/01/14 14:48:43 cvsuser Exp $
 */

#include <iostream>
#include <stdexcept>

#include <sl/net/http/http_function.h>
#include <sl/net/http/message.h>
using namespace sl::net::http;


message::message()
{
	cleanup();
}

void message::cleanup()
{
	_start_line.clear();
	_headers_map.clear();
	_attr_map.clear();
	_body.clear();
}

void message::parse(const std::string& /* text */)
{
}

void message::add_line(const std::string& line)
{
	if (start_line().length() == 0)
		start_line(line);
	else
		parse_line(line);
}


void message::start_line(const std::string& line)
{
	_start_line = line;
}


std::string message::start_line() const
{
	return _start_line;
}

void message::parse_line(const std::string& line)
{
	std::string::size_type pos = line.find(':');
	if (pos != std::string::npos) {
		std::string name = line.substr(0, pos);
		std::string value= line.substr(pos+ 1);

		msg_header(name, tri_left_copy(value));
	}
}


void message::msg_header(int /* code */, const std::string& /* value */)
{
	throw std::runtime_error("msg_header(int, const string) is not implement.");
}

void message::msg_header(const std::string& n, const std::string& v)
{
	_headers_map.insert(std::make_pair(n, v));
}

std::string message::msg_header(const std::string& name) const
{
	multimap_t::const_iterator i = _headers_map.begin();
	for (; i != _headers_map.end(); i++) {
		if (core::equals(i->first, name, core::is_iequal()))
			return i->second;
	}
	return "";
}

list_t message::msg_headers(const std::string& name) const
{
	list_t result;

	multimap_t::const_iterator i = _headers_map.begin();
	for (; i != _headers_map.end(); i++) {
		if (core::equals(i->first, name, core::is_iequal()))
			result.push_back(i->second);
	}
	return result;
}

multimap_t message::msg_headers() const
{
	return _headers_map;
}

void message::erase_header(const std::string& name)
{
	multimap_t::iterator i = _headers_map.begin();
	for (; i != _headers_map.end(); i++) {
		if (core::equals(i->first, name, core::is_iequal()))
			_headers_map.erase(i);
	}
}


void message::msg_body(const std::string& body)
{
	_body = body;
}

void message::add_body(const std::string& partial)
{
	_body.append(partial);
}

std::string message::msg_body() const
{
	return _body;
}


void message::attribute(const std::string& n, const std::string& v)
{
	_attr_map.insert(std::make_pair(n, v));
}

std::string message::attribute(const std::string& name) const
{
	multimap_t::const_iterator i = _attr_map.find(name);
	return i != _attr_map.end() ? i->second : "";
}

multimap_t message::attribute() const
{
	return _attr_map;
}

void message::erase_attribute(const std::string& name)
{
	_attr_map.erase(name);
}

std::string message::text() const
{
	std::string text;
	text.append(start_line() + "\r\n");

	multimap_t::const_iterator i = _headers_map.begin();
	for (; i != _headers_map.end(); i++)
		text.append(i->first + ": " + i->second + "\r\n");
	text.append("\r\n");

	if (!_body.empty())
		text.append(_body);

	return text;
}

std::string message::text_header() const
{
	std::string text;
	text.append(start_line() + "\r\n");

	multimap_t::const_iterator i = _headers_map.begin();
	for (; i != _headers_map.end(); i++)
		text.append(i->first + ": " + i->second + "\r\n");
	text.append("\r\n");

	return text;
}

std::string message::text_body() const
{
	std::string text;
	text.append(start_line() + "\r\n");

	multimap_t::const_iterator i = _headers_map.begin();
	for (; i != _headers_map.end(); i++)
		text.append(i->first + ": " + i->second + "\r\n");
	text.append("\r\n");

	return _body;
}

void message::stream(std::iostream& ifs)
{
	_stream = &ifs;
}

std::iostream& message::stream()
{
	return *_stream;
}
