/*-
 * Copyright (c) 2005 masashi osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: http_function.cpp,v 1.17 2008/01/21 17:00:55 cvsuser Exp $
 */

#include <sys/ioctl.h>
#include <sys/select.h>
#include <unistd.h>
#include <sstream>

#include <sl/net/http/http_error.h>
#include <sl/net/http/http_function.h>
#include <sl/net/http/message.h>
#include <sl/net/http/request.h>
#include <sl/net/http/response.h>


namespace sl {
  namespace net {
    namespace http {

	bool fd_select(int fd, time_t timeout)
	{
		while (timeout--) {
			fd_set  fds;
			FD_ZERO(&fds);
			FD_SET(fd, &fds);

			struct timeval t;
			t.tv_sec = 1;
			t.tv_usec= 0;
			if (::select(FD_SETSIZE, &fds, NULL, NULL, &t) < 0)
				throw http_error("select error occurred while executing it.");

			if (FD_ISSET(fd, &fds))
				break;
		}
		return true;
	}

	bool has_parameter(const message& msg)
	{
		if (msg.type() == msg_type::HTTP_REQUEST) {
			const request& res = static_cast<const request&>(msg);
			if (res.method() == "POST" &&
				res.msg_header("Content-Type") == "application/x-www-form-urlencoded")
			{
				return true;
			}
		}
		return false;
	}

	bool is_chunked(const message& msg)
	{
		std::string te = msg.msg_header("Transfer-Encoding");

		if (msg.version() == "HTTP/1.1" && !te.empty() &&
			te.find("identity") == std::string::npos)
			return true;
		return false;
	}

	int content_length(const message& msg)
	{
		std::string cl = msg.msg_header("Content-Length");

		// HTTP/1.0 ڤ HTTP/0.9 (HTTP/1.1 ʳ) Content-Length Ĵ٤
		if (msg.version() != "HTTP/1.1")
			return ::atoi(cl.c_str());

		if (!is_chunked(msg) && !cl.empty())
			return ::atoi(cl.c_str());
		return -1;
	}

	void write_message(int fd, message& msg)
	{
		std::string text = msg.text();
		::write(fd, text.data(), text.length());
	}

	std::string urldecode(const std::string& s)
	{
		std::string result;

		std::string::const_iterator i = s.begin();
		for (; i != s.end(); i++) {
			if (*i == '%' && i+1 != s.end() && i+2 != s.end())
				result.append(1, static_cast<char>((atoh(*++i) << 4) + atoh(*++i)));
			else if (*i == '+')
				result.append(1, ' ');
			else
				result.append(1, *i);
		}

		return result;
	}

	std::string urlencode(const std::string& s)
	{
		std::string result;

		std::string::const_iterator i = s.begin();
		for (; i != s.end(); i++) {
			if (*i == ' ')
				result.append(1, '+');
			else if (!isprint(*i)) {
				result.append(1, '%');
				result.append(1, htoa((*i & 0xf0) >> 4));
				result.append(1, htoa(*i & 0x0f));
			} else
				result.append(1, *i);
		}

		return result;
	}

	void read_unnecessary(int d, int legth)
	{
"XXXXX";
		long remains;
		if (legth > 0)
			remains = legth;
		else
			//if (::ioctl(d, FIONREAD, &remains) == -1)
				return;

		if (remains > 0) {
			char s[1024];
			for (int total = 0; total < remains; )
				total += ::read(d, s, sizeof(s));
		}
	}

	std::string& erase_crlf(std::string& s)
	{
		size_t end_offset = (!s.empty() && s[s.length() - 1] == '\0') ? 2 : 1;

		for (;;) {
			if (!s.empty() && s[s.length() - end_offset] == '\n' ||
							  s[s.length() - end_offset] == '\r')
			{
				s.erase(s.length() - end_offset, 1);
			} else
				break;
		}
		return s;
	}

	void read_line(std::iostream& is, std::string& s)
	{
		std::istreambuf_iterator<char> begin(is);
		std::istreambuf_iterator<char> end;

		while (!begin.equal(end)) {
			if (*begin == '\n') {
				begin++;
				break;
			}
			s.append(1, *begin++);
		}
		erase_crlf(s);
	}

	int read_message(std::iostream& is, message& msg)
	{
		while (is.good()) {
			std::string s;
			read_line(is, s);

			// null äȶʸǤ size 1 Ȥʤ뤿
			if (s.length() <= 1)
				break;

			msg.add_line(s);
		}

		if (!is.good())
			return -1;

		// POST + application/x-www-form-urlencoded Ǥ
		// Message-Body ʬɤ߹.
		if (has_parameter(msg)) {
			int body_length = content_length(msg);

			// Message-Body ʤнλ.
			if (body_length <= 0)
				return 1;

			std::string s;
			for (int i = 0; i < body_length; i++)
				 s.append(1, static_cast<char>(is.get()));
			msg.msg_body(s);
		} else {
			msg.stream(is);
		}

		return 1;
	}


	void write_headers(std::iostream& os, message& msg)
	{
		std::string headers = msg.text_header();
		os.write(headers.data(), headers.length());
		os.flush();
	}

	void write_body(std::iostream& os, message& msg)
	{
		std::string body = msg.text_body();
		os.write(body.data(), body.length());
		os.flush();
	}

    }  // namespace http
  } // namespace net
} // namespace sl
