/*-
 * Copyright (c) 2005 Masashi Osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: send_headers_packet.cpp,v 1.10 2006/12/29 19:34:59 cvsuser Exp $
 */

#include <iostream>
#include <map>

#include <sl/net/ajp13/function.h>
#include <sl/net/ajp13/send_headers_packet.h>
using namespace sl::net::ajp13;


send_headers_packet::send_headers_packet()
	: _http_status_code(0), _nu_headers(0)
{ }


Byte send_headers_packet::prefix_code() const
{
	return prefix_code::SEND_HEADERS;
}


packet &send_headers_packet::operator=(const ByteArray &array)
	throw(ajp13_error)
{
	const Byte * p = array;

	if (*p != this->prefix_code())
		throw ajp13_error("invalid prefix code");

	++p;	// skip prefix_code
	_http_status_code = get_integer(p); p += sizeof(Integer);
	_http_status_msg = get_string(p);	p += _http_status_msg.size();
	_nu_headers = get_integer(p);		p += sizeof(Integer);

	for (int i = 0; i < _nu_headers; i++)
		parse_headers(p);

	return *this;
}


send_headers_packet::operator ByteArray() const
{
	ByteArray	array;

	array += prefix_code();
	array += _http_status_code;
	array += _http_status_msg;
	array += _nu_headers;

	if (_nu_headers > 0) {
		std::map<Integer, String>::const_iterator i = _headers.begin();
		for (; i != _headers.end(); i++) {
			array += i->first;
			array += i->second;
		}
		std::map<String, String>::const_iterator j = _other_headers.begin();
		for (; j != _other_headers.end(); j++) {
			array += j->first;
			array += j->second;
		}
	} else
		array += static_cast<Byte>(0x00);

	return array;
}


Integer send_headers_packet::size() const
{
	Integer s = 0;
	s += sizeof(Byte);						// Byte		_prefix_code
	s += sizeof(Integer);					// Integer	_http_status_code
	s += _http_status_msg.size();			// String	_http_status_msg
	s += sizeof(Integer);					// Integer	_nu_headers

	std::map<Integer, String>::const_iterator i = _headers.begin();
	for (; i != _headers.end(); i++) {
		s += sizeof(Integer);				// Integer	header code.
		s += i->second.size();				// String
	}

	std::map<String, String>::const_iterator j = _other_headers.begin();
	for (; j != _other_headers.end(); j++) {
		s += j->first.size();				// String	header name.
		s += j->second.size();				// String	header value.
	}

	return s;
}


void send_headers_packet::status_code(Integer code)
{
	_http_status_code = code;
}

void send_headers_packet::status_msg(const std::string& msg)
{
	_http_status_msg  = msg;
}

Integer send_headers_packet::status_code() const
{
	return _http_status_code;
}

std::string send_headers_packet::status_msg() const
{
	return _http_status_msg;
}


void send_headers_packet::header(Integer code, const std::string &value)
	throw(ajp13_error)
{
	if (!res_header_code::find(code))
		throw ajp13_error("invalid header code");

	if (value.empty())
		throw ajp13_error("invalid header value");

	_headers[code] = value;
	++_nu_headers;
}


void send_headers_packet::header(const std::string &n, const std::string &v)
	throw(ajp13_error)
{
    Integer code;

    if ((code = res_header_code::to_code(n)) !=
        res_header_code::UNKNOWN) {
        _headers[code] = v;
	} else
        _other_headers.insert(std::map<String, String>::value_type(n, v));

    ++_nu_headers;
}


bool send_headers_packet::is_exist(const std::string& n) const
{
    Integer code;

    if ((code = res_header_code::to_code(n)) != res_header_code::UNKNOWN)
        return _headers.find(code) != _headers.end();
	else
        return _other_headers.find(n) != _other_headers.end();
}


void send_headers_packet::dump(std::ostream &os)
{
	os << "send_headers_packet" << std::endl;
	os << "prefix:[" << prefix_code() << "]" << std::endl;
	os << "status_code:[" << _http_status_code << "]" << std::endl;
	os << "status_msg: [" << _http_status_msg << "]" << std::endl;
	os << "nu_headers:[" << _nu_headers << "]" << std::endl;

	std::map<Integer, String>::const_iterator i = _headers.begin();
	for (; i != _headers.end(); i++) {
		os << "header_name: [" << i->first << "]" << std::endl;
		os << "header_value:[" << i->second << "]" << std::endl;
	}

	std::map<String, String>::const_iterator j = _other_headers.begin();
	for (; j != _other_headers.end(); j++) {
		os << "other_header_name: [" << j->first << "]" << std::endl;
		os << "other_header_value:[" << j->second << "]" << std::endl;
	}
}


void send_headers_packet::parse_headers(const Byte *&p)
{
	Integer i = get_integer(p);

	if (res_header_code::find(i)) {
		p += sizeof(Integer);
		_headers[i] = get_string(p); p += _headers[i].size();

	} else {
		String  name  = get_string(p); p += name.size();
		String  value = get_string(p); p += value.size();
		_other_headers.insert(
						std::map<String, String>::value_type(name, value));
	}
}


void send_headers_packet::clear()
{
	_http_status_code = 0;
	_http_status_msg.clear();
	_nu_headers = 0;

	_headers.clear();
	_other_headers.clear();
}


void send_headers_packet::erase_header(const std::string& n)
{
    Integer code;

    if ((code = res_header_code::to_code(n)) != res_header_code::UNKNOWN) {
        std::map<Integer, String>::iterator i = _headers.find(code);
        if (i != _headers.end()) {
            _headers.erase(i);
            --_nu_headers;
        }
    } else {
        std::map<String, String>::iterator i = _other_headers.find(n);
        if (i != _other_headers.end()) {
            _other_headers.erase(i);
            --_nu_headers;
        }
    }
}
