/*-
 * Copyright (c) 2008 Masashi Osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef SL_MANGLE_ITANIUM_GRAMMAR_HPP
#define SL_MANGLE_ITANIUM_GRAMMAR_HPP

#include <string>

#ifdef SL_MANGLE_DEBUG
#define BOOST_SPIRIT_DEBUG
#endif
#define BOOST_SPIRIT_THREADSAFE
#include <boost/spirit.hpp>
#include <boost/spirit/phoenix.hpp>
#include <boost/spirit/utility/regex.hpp>

#include <sl/mangle/itanium/encoder.hpp>

namespace sl {

/**
 * @class mangle_grammar.
 *
 * ιʸϥ롼饹.
 * 󥳡ɼ¹ԻˤƤ򤤤ϤΤ
 * Ǥޤʥ롼ɤȤ.
 */
class mangle_grammar : public boost::spirit::grammar<mangle_grammar> {
public :
    mangle_grammar(state& s)
        : _identifier   (sl::encoder::identifier(s)),
          _substitution (sl::encoder::substitution(s)),
          _function     (sl::encoder::function_type(s)),
          _template     (sl::encoder::template_type(s)),
          _builtin      (sl::encoder::builtin(s)),
          _cv_qualifiers(sl::encoder::cv_qualifiers(s)),
          _pointer      (sl::encoder::pointer(s)),
          _end_of_line  (sl::encoder::end_of_line(s)),
          _state(s)
    { }

    struct string_closure : boost::spirit::closure<string_closure, std::string>
    {
        member1 val;
    };

    typedef string_closure                            __closure_type;


    phoenix::functor<sl::encoder::identifier>    _identifier;
    phoenix::functor<sl::encoder::substitution>  _substitution;
    phoenix::functor<sl::encoder::function_type> _function;
    phoenix::functor<sl::encoder::template_type> _template;
    phoenix::functor<sl::encoder::builtin>       _builtin;
    phoenix::functor<sl::encoder::cv_qualifiers> _cv_qualifiers;
    phoenix::functor<sl::encoder::pointer>       _pointer;
    phoenix::functor<sl::encoder::end_of_line>   _end_of_line;
    sl::state _state;

    template <typename ScannerT>
    struct definition
    {
        boost::spirit::rule<ScannerT> top;
        boost::spirit::rule<ScannerT, typename __closure_type::context_t>
            _mangled_name,
            _encoding,
            _nested_name,
            _prefix,
            _template_prefix,
            _source_name,
            _identifier,
            _CV_qualifiers,
            _pointer,
            _builtin_type,
            _function_type,
            _function_param,
            _template_type,
            _template_param,
            _substitution;

        definition(const mangle_grammar& self)
        {
            using namespace boost::spirit;
            using phoenix::arg1;
            using phoenix::arg2;

            const char* identifier_regex("[a-zA-Z0-9_:]+");
top
    = _mangled_name
    ;
_mangled_name
    = _encoding[self._end_of_line()]
    ;
_encoding
    = _nested_name
    ;
_nested_name
    = *(_source_name)
    ;
_source_name
    = *(_CV_qualifiers) >> _identifier >> *(_pointer)
    ;
_identifier
    = _substitution >> *(_template_type)
    | _builtin_type
    | regex_p(identifier_regex)[self._identifier(arg1, arg2)] >> *(_template_type) >> *(_function_type)
    ;
_substitution
    // std::basic_string<char,std::char_traits<char>,std::allocator<char> >
    = *str_p("::") >>
      (str_p("std") >> "::" >> "basic_string" >>
        "<" >> "char" >> "," >> "std" >> "::" >> "char_traits" >>
           "<" >> "char" >> ">" >> "," >> "std" >> "::" >> "allocator" >>
              "<" >> "char" >> ">" >> ">"
      )[self._substitution(arg1, arg2)]

    // ::std::basic_iostream<char, std::char_traits<char> >
    | *str_p("::") >>
      (str_p("std") >> "::" >> "basic_iostream" >>
        "<" >> "char" >> "," >> "std" >> "::" >> "char_traits" >>
           "<" >> "char" >> ">" >> ">"
      )[self._substitution(arg1, arg2)]

    // ::std::basic_istream<char, std::char_traits<char> >
    | *str_p("::") >>
      (str_p("std") >> "::" >> "basic_istream" >>
        "<" >> "char" >> "," >> "std" >> "::" >> "char_traits" >>
           "<" >> "char" >> ">" >> ">"
      )[self._substitution(arg1, arg2)]

    // ::std::basic_ostream<char, std::char_traits<char> >
    | *str_p("::") >>
      (str_p("std") >> "::" >> "basic_ostream" >>
        "<" >> "char" >> "," >> "std" >> "::" >> "char_traits" >>
           "<" >> "char" >> ">" >> ">"
      )[self._substitution(arg1, arg2)]

    // std::allocator
    | *str_p("::") >>(str_p("std") >> "::" >> "allocator")[self._substitution(arg1, arg2)]
    | *str_p("::") >> (str_p("std") >> "::" >> "basic_string")[self._substitution(arg1, arg2)]
    | *str_p("::") >> (str_p("std") >> "::" >> "string")[self._substitution(arg1, arg2)]
    | *str_p("::") >> (str_p("std") >> "::")[self._substitution(arg1, arg2)]
    ;

_CV_qualifiers
    = str_p("const")[self._cv_qualifiers(arg1, arg2)]
    | str_p("volatile")[self._cv_qualifiers(arg1, arg2)]
    ;
_builtin_type
    = (str_p("void")
    | "wchar_t"
    | "bool"
    | str_p("long") >> "long"
    | *(str_p("signed") | "unsigned" ) >> "char"
    | *(str_p("signed") | "unsigned" ) >> "int"
    | *(str_p("signed") | "unsigned" ) >> "long"
    | *(str_p("signed") | "unsigned" ) >> "short"
    | str_p("float")
    | str_p("double")
    | str_p("__int64")
    | str_p("__int128")
    | str_p("__float80")
    | str_p("__float128"))[self._builtin(arg1, arg2)]
    ;
_pointer
    = *(_CV_qualifiers) >> str_p("*")[self._pointer(arg1, arg2)] >> *(_CV_qualifiers)
    | *(_CV_qualifiers) >> str_p("&")[self._pointer(arg1, arg2)] >> *(_CV_qualifiers)
    ;
_function_type
    = str_p("(")[self._function(arg1, arg2)] >>
        _function_param >>
      str_p(")")[self._function(arg1, arg2)]
    ;
_function_param
    = _nested_name >> *( str_p(",") >> _nested_name )
    ;
_template_type
    = str_p("<")[self._template(arg1, arg2)] >>
        _template_param >>
      str_p(">")[self._template(arg1, arg2)];
    ;
_template_param
    = _nested_name >> *( str_p(",") >> _nested_name )
    ;

BOOST_SPIRIT_DEBUG_NODE(_mangled_name);
BOOST_SPIRIT_DEBUG_NODE(_encoding);
BOOST_SPIRIT_DEBUG_NODE(_nested_name);
BOOST_SPIRIT_DEBUG_NODE(_prefix);
BOOST_SPIRIT_DEBUG_NODE(_template_prefix);
BOOST_SPIRIT_DEBUG_NODE(_source_name);
BOOST_SPIRIT_DEBUG_NODE(_identifier);
BOOST_SPIRIT_DEBUG_NODE(_CV_qualifiers);
BOOST_SPIRIT_DEBUG_NODE(_pointer);
BOOST_SPIRIT_DEBUG_NODE(_builtin_type);
BOOST_SPIRIT_DEBUG_NODE(_function_type);
BOOST_SPIRIT_DEBUG_NODE(_function_param);
BOOST_SPIRIT_DEBUG_NODE(_template_type);
BOOST_SPIRIT_DEBUG_NODE(_template_param);
BOOST_SPIRIT_DEBUG_NODE(_substitution);

        }

        const boost::spirit::rule<ScannerT>& start() const { return top; }
    };
};

} // namespace sl

#endif // SL_MANGLE_ITANIUM_GRAMMAR_HPP
