/*-
 * Copyright (c) 2008 Masashi Osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef SL_MANGLE_ITANIUM_ENCODER_HPP
#define SL_MANGLE_ITANIUM_ENCODER_HPP

#ifdef SL_MANGLE_DEBUG
#include <iostream>
#endif
#include <string>
#include <vector>
#include <boost/algorithm/string/classification.hpp>
#include <boost/algorithm/string/split.hpp>
#include <boost/lexical_cast.hpp>

#include <sl/mangle/itanium/state.hpp>

namespace sl {
namespace encoder {

/**
 * C++Ȥ߹߷Ƚ
 *
 * @param s ̾
 * @return Ȥ߹߷Ǥ뤫Ƚ
 *
 * @note ̩ΤǤ __intxx Ȥ⤢褦Ǥ.
 */
inline bool is_builtin(const std::string& s)
{
    if (s == "void"               || s == "v" ||
        s == "wchar_t"            || s == "w" ||
        s == "bool"               || s == "b" ||
        s == "char"               || s == "c" ||
        s == "signed char"        || s == "a" ||
        s == "unsigned char"      || s == "h" ||
        s == "short"              || s == "s" ||
        s == "unsigned short"     || s == "t" ||
        s == "int"                || s == "i" ||
        s == "unsigned int"       || s == "j" ||
        s == "long"               || s == "l" ||
        s == "unsigned long"      || s == "m" ||
        s == "long long"          || s == "x" ||
        s == "unsigned long long" || s == "y" ||
        s == "float"              || s == "f" ||
        s == "double"             || s == "d")
        return true;
    return false;
}

/**
 * פĥ饹Ǥ뤫Ƚꤹ
 *
 * ˤ namespace + class  ¤ΤΥͥޤ "::" Ĥ
 * ɤȽȤʤ. Ƭ "::" Ͻ.
 *
 * @param s ̾
 * @return ݻηǤ뤫Ƚ
 */
inline bool is_scoped_classes(const std::string& s)
{
    return !s.empty() ? (s[0] == 'N') : false;
}

/**
 * is_scoped_classes(const std::string&s) ̤֤.
 */
inline bool is_non_scoped_classes(const std::string& s)
{
    return !s.empty() ? ::isdigit(s[0]) : false;
}

/**
 * const, volatile Ƚ̤֤.
 *
 * @param s ̾
 * @return const, volatile Ǥ뤫Ƚ
 *
 * @note ̩ΤǤ __intxx Ȥ⤢褦Ǥ.
 */
inline bool is_qualifiers(const std::string& s)
{
    if (s == "const" || s == "K" || s == "volatile" || s == "V")
        return true;
    return false;
}

/**
 * ʸ s ʸ¸ߤηǤ뤫Ƚ̤֤.
 *
 * @param s ̾
 * @return زǽȽ
 *
 */
inline bool is_substitution(const std::string& s)
{
    if (s == "Ss" || s == "Sd" || s == "Si" || s == "So" ||
        s == "Sb" || s == "Ss" || s == "Sa" || s == "St")
        return true;
    return false;
}

/**
 * ݥ*, & Ǥ뤫Ƚ̤֤.
 *
 * @param s ̾
 * @return *(P)ޤ&(R) Ǥ뤫Ƚ
 *
 * @note ̩ΤǤ __intxx Ȥ⤢褦Ǥ.
 */
inline bool is_pointer(const std::string& s)
{
    if (s == "*" || s == "P" || s == "&" || s == "R")
        return true;
    return false;
}

inline bool is_definition(const std::string& s)
{
    return is_builtin(s) || is_qualifiers(s) || is_pointer(s);
}

inline bool is_column(const std::string& s)
{
    return !is_builtin(s) && !is_qualifiers(s) &&
           !is_substitution(s) && !is_pointer(s);
}

/**
 * ʸ s Ф̾﷿Υޥ󥰥¹Ԥ.
 *
 * @param ̾
 * @return ʸĹ + ̾ η֤.
 */
inline std::string make_word(const std::string& s)
{
    return boost::lexical_cast<std::string>(s.length()) + s;
}

/**
 * C++ Ȥ߹߷Υޥ󥰥¹Ԥ.
 *
 * @param ̾
 * @return ޥ󥰥̤ʸ.
 */
inline std::string make_builtin_word(const std::string& s)
{
    if (s == "void")               return std::string("v");
    if (s == "wchar_t")            return std::string("w");
    if (s == "bool")               return std::string("b");
    if (s == "char")               return std::string("c");
    if (s == "signed char")        return std::string("a");
    if (s == "unsigned char")      return std::string("h");
    if (s == "short")              return std::string("s");
    if (s == "unsigned short")     return std::string("t");
    if (s == "int")                return std::string("i");
    if (s == "unsigned int")       return std::string("j");
    if (s == "long")               return std::string("l");
    if (s == "unsigned long")      return std::string("m");
    if (s == "long long")          return std::string("x");
    if (s == "unsigned long long") return std::string("y");
    if (s == "float")              return std::string("f");
    if (s == "double")             return std::string("d");
    return s;
}

/**
 * زǽʻʸФޥ󥰥¹Ԥ.
 *
 * @param ̾
 * @return ޥ󥰥̤ʸ.
 */
inline std::string make_substitution_word(const std::string& s)
{
    if (!s.compare(0, 18, "std::basic_string<"))   return std::string("Ss");
    if (!s.compare(0, 20, "std::basic_iostream<")) return std::string("Sd");
    if (!s.compare(0, 19, "std::basic_istream<"))  return std::string("Si");
    if (!s.compare(0, 19, "std::basic_ostream<"))  return std::string("So");
    if (!s.compare(0, 17, "std::basic_string"))    return std::string("Sb");
    if (!s.compare(0, 11, "std::string"))          return std::string("Ss");
    if (!s.compare(0, 14, "std::allocator"))       return std::string("Sa");
    if (!s.compare(0, 5, "std::"))                 return std::string("St");
    return s;
}


template <typename Iterator, typename Value>
Iterator find_last(Iterator first, Iterator last, Value v)
{
    std::reverse_iterator<Iterator> rfirst(last);
    std::reverse_iterator<Iterator> rlast(first);

    while (rfirst != rlast && *first != v)
        ++first;
    return rfirst != rlast ? (++rfirst).base() : last;
}

template <typename Iterator, typename Pred>
Iterator find_last_if(Iterator first, Iterator last, Pred p)
{
    std::reverse_iterator<Iterator> rfirst(last);
    std::reverse_iterator<Iterator> rlast(first);

    while (rfirst != rlast && !p(*rfirst))
        ++first;
    return rfirst != rlast ? (++rfirst).base() : last;
}

/**
 * Ƭˤ륨󥳡ɤ줿̾κǸǤΰ֤򸡺.
 *
 * ̾δؿƥץ졼ȴؿξͤη̾Ϻư
 * ɬפ뤿᤽κݤ˻ѤƬη(==ͤη)
 * ּΤ˻Ѥ.
 * 
 * @param first ޥ󥰥뤷̾Υ塼Ƭ 
 * @param first ޥ󥰥뤷̾Υ塼
 * @return ǽη̾ΰ
 */
template <typename Iterator>
Iterator find_first_type_end(Iterator first, Iterator last)
{
#ifdef SL_MANGLE_DEBUG
std::clog << "find_first_type_end" << std::endl;
#endif
    if (first == last)
        return last;

    /* ʲ4ĤƬ֤礬뤿ᤢäФ */
    if (*first == "P") ++first;
    if (*first == "R") ++first;
    if (*first == "V") ++first;
    if (*first == "K") ++first;

    if (is_builtin(*first))
        return first;

    /*  "::" ޤޤʤ饹 */
    if (is_non_scoped_classes(*first))
        return first;

    /* "N" ǻϤޤ륯饹 */
    /*
     * "E" ʸܰ˷ν򸡺.
     * ƥץ졼 "I" äϼ "E" Ф
     */
    size_t skip = 0;
    while (first != last) {
        if (*first == "E" && skip-- == 0) break;
        if (*first == "I") ++skip;
        ++first;
    }
    return first;
}

/**
 * ˤ뷿̾ƬǤΰ֤򸡺.
 *
 * constvolatileݥ* 仲& Ϸ̾θ˽и礬뤬
 * ޥ󥰥ϷƬ "PRKV" Ȥ뤿˻Ѥ.
 *
 * @param first ޥ󥰥뤷̾Υ塼Ƭ 
 * @param first ޥ󥰥뤷̾Υ塼
 * @return Ǹη̾κǽΰ
 */
template <typename Iterator>
Iterator find_last_type_first(Iterator first, Iterator last)
{
#ifdef SL_MANGLE_DEBUG
std::clog << "find_last_type_first" << std::endl;
#endif
    std::reverse_iterator<Iterator> rfirst(last);
    std::reverse_iterator<Iterator> rlast(first);
    if (rfirst == rlast)
        return first;

    // Ǹΰ "N"(桼)  builtin ΰ֤õ
    if (is_definition(*rfirst) || is_scoped_classes(*rfirst)) {
        if (rfirst + 1 != rlast && *(rfirst + 1) == "St") ++rfirst;
    } else if (*rfirst == "E") {
        int skip = 1;
        while (rfirst != rlast) {
            if (*rfirst == "I") ++skip;
            if (*rfirst == "E") --skip;
            if (*rfirst == "N" && skip == 0) break;
            ++rfirst;
        }
        if (rfirst == rlast)
            return first;
    }

    // ʲ4ĤƬ֤빹˸ʤ
    if (rfirst + 1 != rlast && *(rfirst + 1) == "K") ++rfirst;
    if (rfirst + 1 != rlast && *(rfirst + 1) == "V") ++rfirst;
    if (rfirst + 1 != rlast && *(rfirst + 1) == "R") ++rfirst;
    if (rfirst + 1 != rlast && *(rfirst + 1) == "P") ++rfirst;
    return (++rfirst).base();
}

/**
 * νҤǤ "P","R","V","K" Ф֤.
 *
 * @param ηΥơ.
 * @return ֤ƥ졼.
 */
template <typename Iterator>
inline Iterator skip_prvk(Iterator i)
{
    if (*i == "P") ++i;
    if (*i == "R") ++i;
    if (*i == "V") ++i;
    if (*i == "K") ++i;
    return i;
}

inline size_t match_count(sl::state::stack_type& types,
                          sl::state::stack_type& tmp)
{
    size_t count = 0;
    sl::state::stack_type::const_iterator i = types.begin();
    sl::state::stack_type::const_iterator j = tmp.begin();
    for (; i != types.end() && j != tmp.end(); i++, j++) {
        if (*i != *j)
            break;
        ++count;
    }
    return count;
}


/**
 * ޥ󥰥뤷̾ "E" ɲä.
 *
 * @param ֥饹.
 */
inline void add_prev_type_E(sl::state& state)
{
    if (state._type_end == 0)
        return;

    // previous type end
    if (!state._stack.empty() && is_column(state._stack.back()) &&
        state._prev_word != "<" && state._prev_word != ">" &&
        state._prev_word != "(" && state._prev_word != ")")
    {
        state._stack.push_back("E");
        --state._type_end;
    }
}


/**
 * ѡ¹Ի˥ХåŪ˼¹Ԥե󥯥δ쥯饹.
 *
 * ե󥯥ϥޥåʸ encode ؿ¹ԤʤΤ
 * ե󥯥Ȥɬפʷ result_type ȥޥ󥰥¹Ի
 * ֤ݻѥ饹 sl::state ¸ô.
 */
class action {
public :
    typedef void result_type;
    action(sl::state& result) : _state(result) { }

    void operator()(const std::string::iterator& f,
                    const std::string::iterator& l)
    { preproc(std::string(f, l)); }
    void operator()(const std::string& rhs)
    { preproc(rhs); }
    void operator()()
    { preproc(); }

    void preproc(const std::string& s=std::string())
    {
#ifdef SL_MANGLE_DEBUG
        std::clog << _state << std::endl;
#endif
        encode(s);
#ifdef SL_MANGLE_DEBUG
        std::clog << _state << std::endl;
#endif
        _state._prev_word = s;
    }

    virtual void encode(const std::string& s)=0;

protected :
    sl::state& _state;
};


/**
 * builtin  substitution Ȥƥޥåʤʸ(̾η̾ʤ)
 * ɤ߹˼¹Ԥե󥯥.
 *
 * ̾ "xxx::yyy::zzz" Τ褦ʷ̾.
 */
class identifier : public action {
public :
    identifier(sl::state& result) : action(result) { }

    void encode(const std::string& s)
    {
#ifdef SL_MANGLE_DEBUG
        std::clog << "identifier:[" << s << "]" << std::endl;
#endif
        add_prev_type_E(_state);

        std::vector<std::string> sp;
        boost::split(sp, s, boost::is_any_of("::"),
                     boost::token_compress_on);

        std::string encode_type;
        bool builtin_type = false;
        size_t type_count = 0;

        std::vector<std::string>::iterator i = sp.begin();
        for (; i != sp.end(); i++) {
            if (i->empty())
                continue;

            std::string tmp = make_builtin_word(*i);
            if (tmp != *i) {
                encode_type = tmp;
                builtin_type = true;
                break;
            } else
                encode_type += make_word(*i);

            ++type_count;
        }

        /*
         * identifier  ">" ֤뷿
         *  ƥץ졼ȥ饹Υ(ؿ)Ǥ뤿
         * ̾Ƭ "N" ɲä(ɬפǤ)Ԥʤ.
         *
         * std::vector<int>::aaa          -> _ZNSt6vectorIiE3aaaE
         * int std::vector<int>::aaa(bbb) -> _ZNSt6vectorIiE3aaaE3bbb
         */
        if (_state._prev_word == ">") {
            sl::state::stack_type::iterator i =
              find_last_type_first(_state._stack.begin(), _state._stack.end());
            if (i == _state._stack.end())
                i = _state._stack.begin();
            if (*skip_prvk(i) != "N") {
                _state._stack.insert(skip_prvk(i), "N");
                ++_state._type_end;
            }
        }

        // std::xxxx<int ...> / StN4xxxxE -> St4xxxx
        //                   ^^^       ^    ^^^
        /*
         * ̾Ƭ "N" ղý.
         * builtin std::("St") ǳϤ뷿ξ
         * ͡ॹڡޤޤʤ "N" 
         * ޤľ ">" ξϥƥץ졼ȥ饹ü첽̾
         * Сؿ̾Ǥ귿ΰȤʤ뤿 "N" Ĥʤ.
         *
         * xxx<yyy>::zzz  _Z3xxxI3yyyE3zzz Ȥʤ
         * xxx<yyy>::zzz  _Z3xxxI3yyyE3zzz Ȥʤ
         */
        if (!builtin_type && (_state._prev_word != ">") &&
            (_state._stack.empty() || _state._stack.back() != "St"))
        {
            /*
             * ͡ॹڡ̵꤬ξ "N" פäݤ
             */
            if (type_count > 1) {
                _state._stack.push_back("N");
                ++_state._type_end;
            }
        }
        _state._stack.push_back(encode_type);

    }
};


/**
 * ɸƥץ졼ȷ̾ṳ֤̂̾ե󥯥.
 *
 * std::basic_string  std::allocator Ф "Ss"  "Sa" ʤɤ
 * ̾򸡺ƥåݻ.
 * 
 * @note ط̾ "E" Ĥɬפʤ.
 */
class substitution : public action {
public :
    substitution(sl::state& result) : action(result) { }

    void encode(const std::string& s)
    {
#ifdef SL_MANGLE_DEBUG
        std::clog << "substitution:[" << s << "]" << std::endl;
#endif
        add_prev_type_E(_state);

        std::string tmp = make_substitution_word(s);
        if (tmp == s)
            throw std::runtime_error("parameter is not substitution.");
        _state._stack.push_back(tmp);
    }
};


/**
 * C++ Ȥ߹߷ɤ߹߻˼¹Ԥե󥯥.
 *
 * void  char, int ʤɤȤ߹ߤηФ "v"  "c", "i" ʤɤ
 * ά줿ʸ򸡺ƥåݻ.
 *
 * @note ط̾ "E" ĤɬפϤʤ.
 */
class builtin : public action {
public :
    builtin(sl::state& result) : action(result) { }

    void encode(const std::string& s)
    {
#ifdef SL_MANGLE_DEBUG
        std::clog << "buitin:[" << s << "]" << std::endl;
#endif
        add_prev_type_E(_state);

        std::string tmp = make_builtin_word(s);
        if (tmp != s)
            _state._stack.push_back(tmp);
        else
            throw std::runtime_error("parameter is not builtin type.");

    }
};


/**
 * ؿѥ᡼γʸ "("λʸ ")" ˼¹Ԥե󥯥.
 *
 * @note ʸ "("  ">" Ǥ
 * ü첽줿ƥץ졼ȴؿμ¹ "xxx yyy::zzz<AAA>(BBB)" Ȥʤ
 * ξͤؿѥ᡼ηɬפ.
 * ʸ "(" ̾ξϥƥץ졼ȥ饹Υдؿ
 * ̾δؿȤʤ.
 *
 * xxx yyy::zzz<AAA>::func(BBB) ϥƥץ졼ȥдؿ
 * xxx yyy::zzz::func(BBB) ̾δؿ(饹ФǤʤ̾δؿ
 * ޤäƱͤ˥ޥ󥰥뤵.
 */
class function_type : public action {
public :
    function_type(sl::state& result) : action(result) { }

    void encode(const std::string& s)
    {
#ifdef SL_MANGLE_DEBUG
        std::clog << "function_type:[" << s << "]" << std::endl;
#endif
        if (s == "(") {
            if (_state._prev_word == ">")
                template_function_begin(s);
            else
                non_template_function_begin(s);

        } else if (s == ")")
            add_prev_type_E(_state);
        else
            throw std::runtime_error("paramter is not function type.");
    }

    /**
     * ü첽ƥץ졼ȴؿδؿʸ"("Υ󥳡ɽ.
     *
     * ƥץ졼ȴؿξ̾δؿȤϰۤʤͤη
     * ޥ󥰥ʸ˴ޤޤ.
     * ͤη֤̾ϴؿ̾ľȤʤ뤿
     * ǽ˽и̾ư뤳ȤˤäƼ¸.
     *
     * AAA xxx<yyy>(zzz)  _ZN3xxxIN3yyyEEEN3AAAEN3zzzE ȥޥ󥰥뤵.
     */
    void template_function_begin(const std::string& s)
    {
        sl::state::stack_type::iterator i = 
            find_first_type_end(_state._stack.begin(), _state._stack.end());
        if (i != _state._stack.end()) ++i; /* 1ʸ˿ʤ */

        sl::state::stack_type r;  // ͷΰ¸
        r.assign(_state._stack.begin(), i);
        _state._stack.erase(_state._stack.begin(), i);

        /* νʤΤɬפ "E" 򤹤٤ղä */
        add_prev_type_E(_state);
        for (size_t n = 0; n < _state._type_end; n++) {
            _state._stack.push_back("E");
            --_state._type_end;
        }
        std::copy(r.begin(), r.end(), std::back_inserter(_state._stack));
    }

    /**
     * ƥץ졼ȥдؿʳδؿʸ"("Υ󥳡ɽ.
     *
     * "("  ">" ʤǥƥץ졼ȥдؿʳ
     * ̾δؿǤꡢξͤηϥޥ󥰥η̤
     * 뤿ᤳǤƬη.
     */
    void non_template_function_begin(const std::string& s)
    {
        /*
         * ˥åݻƤͤη
         */
        sl::state::stack_type::iterator i = 
            find_first_type_end(_state._stack.begin(), _state._stack.end());
        if (i != _state._stack.end()) ++i; /* 1ʸ˿ʤ */
        _state._stack.erase(_state._stack.begin(), i);

        /* ؿ̾ "E" ɲä */
        add_prev_type_E(_state);
    }
};


/**
 * ƥץ졼Ȥγʸ "<"λʸ ">" ˼¹Ԥե󥯥.
 * 
 */
class template_type : public action {
public :
    template_type(sl::state& result) : action(result) { }

    void encode(const std::string& s)
    {
#ifdef SL_MANGLE_DEBUG
        std::clog << "template_type:[" << s << "]" << std::endl;
#endif
        /*
         * ƥץ졼γ.
         * ̾ˤϥƥץ졼Ȥޤޤ뤿ᤳλǤϳȤʤ
         * "I" ղäƽλ.
         *
         * ˤΥƥץ졼ȳʸ"<"η̾
         * 桼Ǥ꤫ĥ͡ॹڡޤޤʤ
         * "N" ΤĤƤʤ֤Ǥ뤬ƥץ졼"<"ޤ
         * ɬ "N" ɬפǤ뤿ᤳƤ.
         */
        if (s == "<") {
            sl::state::stack_type::iterator i =
                find_last_type_first(_state._stack.begin(), _state._stack.end());
            if (i == _state._stack.end())
                i = _state._stack.begin();

            if (*i == "P") ++i;
            if (*i == "R") ++i;
            if (*i == "V") ++i;
            if (*i == "K") ++i;
            if (!is_builtin(*i) && !is_substitution(*i) && *i != "N") {
                _state._stack.insert(i, "N");
                ++_state._type_end;
            }
            _state._stack.push_back("I");

        } else if (s == ">") {
            add_prev_type_E(_state);
            _state._stack.push_back("E");
        } else
            throw std::runtime_error("paramter is not template type.");
    }
};


class cv_qualifiers : public action {
public :
    cv_qualifiers(sl::state& result) : action(result) { }

    void encode(const std::string& s)
    {
#ifdef SL_MANGLE_DEBUG
        std::clog << "cv_qualifiers:[" << s << "]" << std::endl;
#endif
        sl::state::stack_type::iterator i =
            find_last_type_first(_state._stack.begin(),_state._stack.end());

        if (_state._prev_word == "(" || _state._prev_word == "<") {
            if (s == "volatile") _state._stack.push_back("V");
            if (s == "const")    _state._stack.push_back("K");

        } else if (s == "volatile") {
            if (i != _state._stack.end() && *i == "P") ++i;
            if (i != _state._stack.end() && *i == "R") ++i;
            if (i != _state._stack.end() && *i == "N") ++i;
            if (i == _state._stack.end() || *i != "V")
                _state._stack.insert(i, "V");

        } else if (s == "const") {
            if (i != _state._stack.end() && *i == "P") ++i;
            if (i != _state._stack.end() && *i == "R") ++i;
            if (i != _state._stack.end() && *i == "V") ++i;
            if (i != _state._stack.end() && *i == "N") ++i;
            if (i == _state._stack.end() || *i != "K")
                _state._stack.insert(i, "K");
        } else
            throw std::runtime_error("parameter is not cv-qualifiers.");
    }
};


class pointer : public action {
public :
    pointer(sl::state& result) : action(result) { }

    void encode(const std::string& s)
    {
#ifdef SL_MANGLE_DEBUG
        std::clog << "pointer:[" << s << "]" << std::endl;
#endif
        add_prev_type_E(_state);

        std::string word;
        if (s == "*")      word = "P";
        else if (s == "&") word = "R";
        else
            throw std::runtime_error("parameter is not pointer or referece.");
        sl::state::stack_type::iterator i =
            find_last_type_first(_state._stack.begin(), _state._stack.end());
        if (i != _state._stack.end() && *i == word) ++i;
        _state._stack.insert(i, word);
    }
};


class end_of_line : public action {
public :
    end_of_line(sl::state& result) : action(result) { }

    void encode(const std::string& s)
    {
#ifdef SL_MANGLE_DEBUG
        std::clog << "end_of_line:[" << s << "]" << std::endl;
#endif
        /*
         * ̾Τߤξ identifier ¹Ը¨ EOL ȤʤΤ
         * Ƿ "E" ղä
         */
        add_prev_type_E(_state);

        for (size_t n = 0; n < _state._type_end; n++)
            _state._stack.push_back("E");

        /*
         * XXXXX
         * ƬηȤ߹߷ʳξƬ "_Z" դƤ
         */
        state::stack_type::iterator i =  _state._stack.begin();
        if (i != _state._stack.end() && *i == "P") ++i;
        if (i != _state._stack.end() && *i == "R") ++i;
        if (i != _state._stack.end() && *i == "V") ++i;
        if (i != _state._stack.end() && *i == "K") ++i;
        if (i != _state._stack.end() && *i == "N") ++i;
        if (i != _state._stack.end() &&
            !is_builtin(*i) && !is_substitution(*i))
        	_state._stack.insert(_state._stack.begin(), "_Z");

        for (i = _state._stack.begin(); i != _state._stack.end(); i++)
            _state._encode_name += *i;
    }
};

} // namespace encoder
} // namespace sl

#endif // SL_MANGLE_ITANIUM_ENCODER_HPP
