/*-
 * Copyright (c) 2005 Masashi Osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef JAVA_UTIL_DATE_H
#define JAVA_UTIL_DATE_H

#include <time.h>    // time(), localtime_r()

#include <string>
#include <sl/java/lang/Object.h>

namespace sl {
namespace java {
namespace util {

class Date : public lang::objectable {
public :
    Date(long date=0) : _time(date) { }

    virtual ~Date() { }

    bool operator==(const Date& another) const
    {
        return this->_time == another._time;
    }

    bool operator<(const Date& r) const
    {
        return this->_time < r._time;
    }

    bool operator>(const Date& r) const
    {
        return this->_time > r._time;
    }

    bool after(const Date& when) const
    {
        return *this < when;
    }

    bool before(const Date& when) const
    {
        return *this > when;
    }

    int compareTo(const Date& anotherDate) const
    {
        if (this->getTime() > anotherDate.getTime())
            return 1;
        else if (this->getTime() < anotherDate.getTime())
            return -1;

        return 0;
    }

    bool equals(const java::lang::Object& obj) const
    {
        try {
            return *this == sl::object_cast<Date>(obj);
        } catch (std::bad_cast& e) { }
        return false;
    }

    long getTime() const
    {
        return _time;
    }

    /** dow mon dd hh:mm:ss zzz yyyy */
    std::string toString() const
    {
        char time_s[64];
        struct tm result;
        localtime_r(&_time, &result);

        int ret = strftime(time_s, 32, "%a %b %d %H:%M:%S %Y", &result);
        return ret != 0 ? time_s : std::string();
    }

private :
    time_t    _time;
};

} // namespace util
} // namespace java
} // namespace sl

#endif // JAVA_UTIL_DATE_H
