/*-
 * Copyright (c) 2005 Masashi Osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef JAVA_UTIL_CALENDER_H
#define JAVA_UTIL_CALENDER_H

#include <time.h>    // time(), localtime_r()
#include <stdexcept>
#include <string>
#include <vector>

#include <sl/java/lang/Object.h>
#include <sl/java/util/Date.h>
#include <sl/java/util/Locale.h>


namespace sl {
namespace java {
namespace util {

class Calendar : public lang::objectable {
public :
    static const int AM = 0;
    static const int PM = 1;
    static const int AM_PM = 9;

    static const int JANUARY   = 0;
    static const int FEBRUARY  = 1;
    static const int MARCH     = 2;
    static const int APRIL     = 3;
    static const int MAY       = 4;
    static const int JUNE      = 5;
    static const int JULY      = 6;
    static const int AUGUST    = 7;
    static const int SEPTEMBER = 8;
    static const int OCTOBER   = 9;
    static const int NOVEMBER  = 10;
    static const int DECEMBER  = 11;
    static const int UNDECIMBER= 12;

    static const int SUNDAY    = 1;
    static const int MONDAY    = 2;
    static const int TUESDAY   = 3;
    static const int WEDNESDAY = 4;
    static const int THURSDAY  = 5;
    static const int FRIDAY    = 6;
    static const int SATURDAY  = 7;

    static const int FIELD_COUNT   = 17;
    static const int ERA           = 0;
    static const int YEAR          = 1;
    static const int MONTH         = 2;
    static const int WEEK_OF_YEAR  = 3;
    static const int WEEK_OF_MONTH = 4;
    static const int DATE          = 5;
    static const int DAY_OF_MONTH  = 5;
    static const int DAY_OF_YEAR   = 6;
    static const int DAY_OF_WEEK   = 7;
    static const int DAY_OF_WEEK_IN_MONTH = 8;
    static const int HOUR          = 10;
    static const int HOUR_OF_DAY   = 11;
    static const int MINUTE        = 12;
    static const int SECOND        = 13;
    static const int MILLISECOND   = 14;
    static const int ZONE_OFFSET   = 15;
    static const int DST_OFFSET    = 16;

    virtual ~Calendar() { }

    bool operator==(const Calendar& another) const
    {
        return this->_clock == another._clock;
    }

    bool operator<(const Calendar& r) const
    {
        return this->_clock < r._clock;
    }

    bool operator>(const Calendar& r) const
    {
        return this->_clock > r._clock;
    }

    static Calendar getInstance()
    {
        return Calendar();
    }

    virtual void add(int field, int amount)
    {
        fields[field] += amount;
    }

    bool after(const lang::Object& when) const
    {
        try {
            return *this > sl::object_cast<Calendar>(when);
        } catch (std::bad_cast& e) { }
        return false;
    }

    bool before(const lang::Object& when) const
    {
        try {
            return *this < sl::object_cast<Calendar>(when);
        } catch (std::bad_cast& e) { }
        return false;
    }

    void clear()
    {
        fields.clear();
    }

    void clear(int field)
    {
        throw std::runtime_error("Not Implements.");
    }

    int compareTo(const Calendar& anotherCalendar) const
    {
        if (this->getTime() > anotherCalendar.getTime())
            return 1;
        else if (this->getTime() < anotherCalendar.getTime())
            return -1;
        return 0;
    }

    bool equals(const lang::Object& obj) const
    {
        try {
            return *this == sl::object_cast<Calendar>(obj);
        } catch (std::bad_cast& e) { }
        return false;
    }

    int get(int field) const
    {
        return fields.at(field);
    }

    int getActualMaximum(int field) const
    {
        throw std::runtime_error("Not Implements.");
    }

    int getActualMinimum(int field) const
    {
        throw std::runtime_error("Not Implements.");
    }

    static std::vector<Locale> getAvailableLocales()
    {
        throw std::runtime_error("Not Implements.");
    }

    int getFirstDayOfWeer() const
    {
        throw std::runtime_error("Not Implements.");
    }

    virtual int getGreatestMinimum(int field) const
    {
        throw std::runtime_error("Not Implements.");
    }

    virtual int getLeastMaximum(int field) const
    {
        throw std::runtime_error("Not Implements.");
    }

    int getMinimalDaysInFirstWeek() const
    {
        throw std::runtime_error("Not Implements.");
    }

    virtual int getMinimum(int field) const
    {
        throw std::runtime_error("Not Implements.");
    }

    Date getTime() const
    {
        return Date(_clock);
    }

    long getTimeInMillis() const
    {
        throw std::runtime_error("Not Implements.");
    }

    bool isLenient() const
    {
        throw std::runtime_error("Not Implements.");
    }

    bool isSet(int field) const
    {
        throw std::runtime_error("Not Implements.");
    }

    virtual void roll(int field, bool up)
    {
        throw std::runtime_error("Not Implements.");
    }

    void roll(int field, int amount)
    {
        throw std::runtime_error("Not Implements.");
    }

    void set(int field, int value)
    {
        fields[field] += value;
    }

    void set(int year, int month, int date, int hour_of_day=0, int minute=0)
    {
        throw std::runtime_error("Not Implements.");
    }

    void setFirstDayOfWeek(int value)
    {
        throw std::runtime_error("Not Implements.");
    }

    void setLenient(bool lenient)
    {
        throw std::runtime_error("Not Implements.");
    }

    void setMinimalDaysInFirstWeek(int value)
    {
        throw std::runtime_error("Not Implements.");
    }

    void setTime(const Date& date)
    {
        __member_fields_init(date.getTime());
    }

    void setTimeInMillis(long millis)
    {
        __member_fields_init(millis);
    }

    std::string toString() const
    {
        struct tm result;
        result.tm_year = fields[YEAR] - 1900;
        result.tm_mon = fields[MONTH];
        result.tm_mday = fields[DATE];
        result.tm_yday = fields[DAY_OF_YEAR];
        result.tm_wday = fields[DAY_OF_WEEK];
        result.tm_hour = fields[HOUR];
        result.tm_hour = fields[HOUR_OF_DAY];
        result.tm_min = fields[MINUTE];
        result.tm_sec = fields[SECOND];

        char time_s[64];
        int ret = strftime(time_s, 32, "%a %b %d %H:%M:%S %Y", &result);
        return ret != 0 ? time_s : std::string();
    }

protected :

    Calendar() : fields(FIELD_COUNT)
    {
        ::time(&_clock);
        __member_fields_init(_clock);
    }

    virtual void computeFields()
    {
    }

    virtual void computeTime()
    {
    }

    void complete()
    {
    }

    int internalGet(int field) const
    {
        throw std::runtime_error("Not Implements.");
    }

protected :
    std::vector<int> fields;

private :
    time_t        _clock;
    struct tm    _time;

    void __member_fields_init(time_t clock)
    {
        ::localtime_r(&_clock, &_time);

        fields[ERA]             = 0;
        fields[YEAR]            = _time.tm_year + 1900;
        fields[MONTH]           = _time.tm_mon;
        fields[WEEK_OF_YEAR]    = 0;
        fields[WEEK_OF_MONTH]   = 0;
        fields[DATE]            = _time.tm_mday;
        fields[DAY_OF_YEAR]     = _time.tm_yday;
        fields[DAY_OF_WEEK]     = _time.tm_wday;
        fields[DAY_OF_WEEK_IN_MONTH]    = 0;
        fields[HOUR]            = _time.tm_hour;
        fields[HOUR_OF_DAY]     = _time.tm_hour;
        fields[MINUTE]          = _time.tm_min;
        fields[SECOND]          = _time.tm_sec;
        fields[MILLISECOND]     = _time.tm_sec;
        fields[ZONE_OFFSET]     = 0;
        fields[DST_OFFSET]      = 0;
    }
};

} // namespace util
} // namespace java
} // namespace sl

#endif // JAVA_UTIL_CALENDER_H
