/*-
 * Copyright (c) 2005 Masashi Osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef JAVA_LANG_STRING_H
#define JAVA_LANG_STRING_H

#include <cctype>
#include <functional>
#include <iostream>
#include <sstream>
#include <stdexcept>
#include <string>

#include <sl/java/lang/Object.h>
#include <sl/java/lang/Exception.h>
#include <sl/java/lang/ClassCastException.h>

namespace sl {
namespace java {
namespace lang { 

class String : public objectable {
public :
    String() { }

    String(const char *s) : _string(s) { }

    String(const std::string &s) : _string(s) { }

    bool operator!=(const String& s)
    {
        return _string != s._string;
    }

    operator std::string&()
    {
        return _string;
    }

    operator const std::string&() const
    {
        return _string;
    }

    String operator+(const String& r) const
    {
        String s(_string);
        return s += r;
    }

    String operator+(int i) const
    {
        String s(_string);
        return s += i;
    }

    String& operator+=(const String& r)
    {
        _string += r._string;
        return *this;
    }

    String operator+=(int i)
    {
        std::stringstream ss;
        ss << i;
        return String(_string += ss.str());
    }

    // Compares this String to another Object. 
    int compareTo(const Object &o) const
        throw (ClassCastException)
    {
        try {
            String s = object_cast<String>(o);
            return this->compareTo(s);
        } catch (std::bad_cast) {
            throw ClassCastException("");
        }
    }

    // Compares two strings lexicographically. 
    int compareTo(const String &another) const 
    {
        return this->_string.compare(another._string);
    }

    // Compares two strings lexicographically. 
    int compareTo(const char* another) const 
    {
        return this->_string.compare(std::string(another));
    }

    int indexOf(const String& s) const
    {
        std::string::size_type pos = this->_string.find(s);
        return pos == std::string::npos ? -1 : static_cast<int>(pos);
    }

    String toUpperCase() const
    {
        std::string result;
        std::transform(_string.begin(), _string.end(),
                    std::back_inserter(result), std::ptr_fun(::toupper));
        return result;
    }

    String replaceFirst(const std::string& /* regex */, String /* replacement */) 
    {
        throw std::runtime_error("Not implementation");
        return "";
    }

    String replaceAll(const std::string& /* regex */, String /* replacement */) 
    {
        throw std::runtime_error("Not implementation");
        return "";
    }

    template <typename ValueType>
    static String valueOf(ValueType target) throw (Exception)
    {
        try {
            std::stringstream ss;
            ss << target;
            return ss.str();
        } catch (std::bad_cast) {
            throw Exception("");
        }
    }

    static String valueOf(Object obj)
    {
        if (obj)
            return std::string(obj.to_string());
        return std::string("null");
    }

private :
    std::string _string;
};

inline std::ostream &operator<<(std::ostream &os, const String &s)
{
    return os << static_cast<const std::string&>(s);
}

inline std::istream &operator>>(std::istream &is, String &s)
{
    return is >> static_cast<std::string&>(s);
}

inline String operator+(const std::string& l, const String& r)
{
    return String(l) + r;
}

inline bool operator==(const String& l, const std::string& r)
{
    return l.operator const std::string&() == r;
}

} // namespace lang
} // namespace java
} // namespace sl


#endif // JAVA_LANG_STRING_H
