/*-
 * Copyright (c) 2005 Masashi Osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef JAVA_LANG_CLASS_LOADER_H
#define JAVA_LANG_CLASS_LOADER_H

#include <map>
#include <string>

#include <sl/class_loader.hpp>
#include <sl/java/lang/Class.h>
#include <sl/java/lang/ClassNotFoundException.h>

namespace sl {
namespace java {
namespace lang {

class ClassLoader {
public :
    ClassLoader() { }

    virtual ~ClassLoader() { }

protected :
    virtual Class loadClass(const std::string &name)
        throw(java::lang::ClassNotFoundException)
    {
        return findSystemClass(name);
    }

    virtual void unloadClass(const std::string &name)
    {
        return unloadSystemClass(name);
    }

protected :

    Class findSystemClass(const std::string &name)
        throw(java::lang::ClassNotFoundException)
    {
        try {
            sl::class_loader* p = new sl::class_loader(name);
            _map.insert(std::make_pair(name, p));
            return Class(p);
        } catch (std::exception& e) {
            throw java::lang::ClassNotFoundException(e.what());
        }
    }

    virtual void unloadSystemClass(const std::string &name)
    {
        std::map<std::string, sl::class_loader*>::iterator i = _map.find(name);
        if (i != _map.end()) {
            i->second->unload();
            delete i->second;
            _map.erase(i);
        }
    }

private :
    std::map<std::string, sl::class_loader*> _map;
};

} // namespace lang
} // namespace java
} // namespace sl

#endif // JAVA_LANG_CLASS_LOADER_H
