/*-
 * Copyright (c) 2005 Masashi Osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: fdstream.h,v 1.9 2006/11/04 08:13:23 cvsuser Exp $
 */

#ifndef IO_FDSTREAM_H
#define IO_FDSTREAM_H

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <unistd.h>

#include <algorithm>
#include <iostream>
#include <iomanip>
#include <ios>
#include <stdexcept>

#if __DEBUG
#include <sl/sys/clock.h>
#define debug(...)
#endif

namespace io {

	class callback_function {
	public :
		virtual ~callback_function() { }
		virtual int operator()(int d, char* buffer, size_t size)=0;
	};

	class default_reader : public callback_function {
	public :
		int operator()(int d, char* s, size_t n)
		{
			return ::read(d, s, n);
		}
	};

	class default_writer : public callback_function {
		int operator()(int d, char* s, size_t n)
		{
			return ::write(d, s, n);
		}
	};

	typedef callback_function*	reader_t;
	typedef callback_function*	writer_t;


	class fdstreambuf : public std::streambuf {

		enum { _default_buffer = 4092 };

	public :
		typedef char						char_type;
		typedef std::char_traits<char>		traits_type;
		typedef traits_type::int_type		int_type;
		typedef traits_type::pos_type		pos_type;
		typedef traits_type::off_type		off_type;

		// Constractor/Destoractor.
		fdstreambuf(int d)
			: _descriptor(d),
			  _timeout(0),
			  _curtime(0),
			  _length(_default_buffer),
			  _rlength(-1),
			  _currlen(0)
		{
			_reader = &_default_reader;
			_writer = &_default_writer;

			_gbuffer = new char[_length];
			_pbuffer = new char[_length];
			setg(_gbuffer, _gbuffer + _length, _gbuffer + _length);
			setp(_pbuffer, _pbuffer + _length);
		}

		~fdstreambuf()
		{
			delete _gbuffer;
			delete _pbuffer;
		}

		/**
		 * եǥץꤷޤ.
		 * ޤߤΥХåե򥯥ꥢޤ.
		 *
		 * @param	ꤹե/åȥǥץ.
		 */
		void reset(int d)
		{
			_descriptor = d;

			setg(_gbuffer, _gbuffer + _length, _gbuffer + _length);
			setp(_pbuffer, _pbuffer + _length);
		}

		/**
		 * ɤ߹߻ΥǡԤ֤ꤷޤ.
		 *
		 * @param	Ԥ().
		 */
		void timeout(time_t t)
		{
			_curtime = t;
			_timeout = t;
		}

		/**
		 * ɤ߹ߥǡκꤷޤ.
		 *
		 * ºݤˤŬʥΥǡ٤ɤ߹िᡢ
		 * δؿǻꤹͤɤ߹ߤξ¤ǤϤʤ
		 * [ɬפʺ]ȤƻѤ졢ǡμФԤʤäˡ
		 * ɤ߹ǡĤǤäƤ n ХȤͤФ
		 * EOF ֤ưȤʤޤ.
		 *
		 * HTTP  Content-Length  ܥǥδطͤʡ
		 * ɤ߽ФǡΥХȿϤ狼äƤꡢͤ
		 * ɤߤǡʤԤ碌Ƥ
		 * EOF֤ưȤ뤿˺ޤ.
		 *
		 * Υ饹󥹥󥹤ڤӡåȤˤ
		 * ǡĤΤǡδؿ˰ -1 ꤷ
		 * ĤΥǡɤ߹ɬפޤ.
		 *
		 * ɤ߹Ǥδؿ¹Ԥưʤ
		 * ǽޤ.
		 *
		 * @paran	int  ɬפʺ. -1 ̵. -1ʳͭ.
		 *
		 * @note	ServletAPI  getContentLength ̤ -1 ֤
		 *			Ǥ⥤󥿡ե碌int ˤƤ.
		 */
		void rlength(int n)
		{
			_rlength = n;
			_currlen = egptr() - gptr();
#ifdef __DEBUG
	debug(stdout, "%s\n", __FUNCTION__);
	debug(stdout, "_rlength %d\n", _rlength);
	debug(stdout, "_currlen %d\n", _currlen);
#endif
		}

		void reading_method(reader_t p)
		{
			_reader = p;
		}

		void writing_method(writer_t p)
		{
			_writer = p;
		}

		std::streambuf* pubsetbuf(char_type* s, std::streamsize n)
		{
			if (n <= 0)
				throw std::runtime_error("std::streambuf::pubsetbuf");

			delete _gbuffer;
			delete _pbuffer;

			_length = n;
			_gbuffer = new char[_length];
			_pbuffer = new char[_length];
			setg(_gbuffer, _gbuffer + _length, _gbuffer + _length);
			setp(_pbuffer, _pbuffer + _length);
		}

	protected :

		std::streamsize xsputn(const char_type* s, std::streamsize n)
		{
#ifdef __DEBUG
	debug(stdout, "%s\n", __FUNCTION__);
#endif
			int wval = epptr() - pptr();
			if (n <= wval) {
				memcpy(pptr(), s, n * sizeof(char_type));
				pbump(n);
				return n;
			}
			memcpy(pptr(), s, wval * sizeof(char_type));
			pbump(wval);

			if (overflow() != traits_type::eof())
				return wval + xsputn(s + wval, n - wval);

			return wval;
		}

		int sync()
		{
#ifdef __DEBUG
	debug(stdout, "%s\n", __FUNCTION__);
#endif
			if (pptr() && pbase() < pptr() && pptr() <= epptr()) {

#ifdef __DEBUG
	debug(stdout, "write start. size of %d\n", pptr() - pbase());
	debug(stdout, "%f\n", sys::clock::elapsed());
#endif
				int size = (*_writer)(_descriptor, pbase(), pptr() - pbase());
#ifdef __DEBUG
	debug(stdout, "write at %d bytes.\n", size);
	debug(stdout, "%f\n", sys::clock::elapsed());
#endif
				if (size <= 0)
					return -1;

				std::copy(pbase() + size, pptr(), pbase());
				pbump(pptr() - pbase() - size);
				setp(pbase(), epptr());
				return traits_type::not_eof(size);
			}
			return traits_type::eof();
		}

		int_type overflow(int_type c = traits_type::eof())
		{
#ifdef __DEBUG
	debug(stdout, "%s\n", __FUNCTION__);
#endif
			if (!pbase())
				return traits_type::eof();

			if (c == traits_type::eof())
				return sync();

			if (pptr() == epptr())
				sync();

			*pptr() = traits_type::to_char_type(c);
			pbump(1);

			return c;
		}

		int_type underflow()
		{
#ifdef __DEBUG
	debug(stdout, "%s. %f\n", __FUNCTION__, sys::clock::elapsed());
#endif
			// ɤ߹߿λ꤬
			// ɤ߹߿ȸߤɤ߹߿Ʊξ
			// ʾɤ߹ޤʤ.
			if (_rlength != -1 && _rlength <= _currlen) {
#ifdef __DEBUG
	debug(stdout, "_rlength=%d _currlen=%d\n", _rlength, _currlen);
#endif
				return traits_type::eof();
			}

			if (!gptr())
				return traits_type::eof();

			if (gptr() < egptr())
				return traits_type::to_int_type(*gptr());

			if (!select(_timeout)) {
#ifdef __DEBUG
	debug(stdout, "select failed. return traits_type::eof().\n");
#endif
				return traits_type::eof();
			}

			size_t read_size = _length - static_cast<size_t>(gptr() - egptr());
#ifdef __DEBUG
	debug(stdout, "read start. size of %d\n", read_size);
	debug(stdout, "%f\n", sys::clock::elapsed());
#endif
			int size = (*_reader)(_descriptor, eback(), read_size);

#ifdef __DEBUG
	debug(stdout, "read at %d bytes.\n", size);
	debug(stdout, "%f\n", sys::clock::elapsed());
#endif
			// ɤ߹߿λ꤬
			if (_rlength != -1)
				_currlen += size;

			if (size <= 0)
				throw std::runtime_error("read failed in `streambuf::underflow'");

			if (size == 0) {
				setg(_gbuffer, _gbuffer + _length, _gbuffer + _length);
				return traits_type::eof();
			}
			setg(eback(), eback(), eback() + size);
			return traits_type::to_int_type(*gptr());
		}

		int_type uflow()
		{
#ifdef __DEBUG
	debug(stdout, "%d\n", __FUNCTION__);
#endif
			return std::streambuf::uflow();
		}

	private :

		bool select(time_t timeout)
		{
			_curtime = timeout;
#ifdef __DEBUG
	debug(stdout, "%s. %f\n", __FUNCTION__, sys::clock::elapsed());
#endif
			do {
				fd_set  fds;
				FD_ZERO(&fds);
				FD_SET(_descriptor, &fds);

				struct timeval t;
				t.tv_sec = 1;
				t.tv_usec= 0;
				int ret = ::select(_descriptor+1, &fds, 0, 0, timeout ? &t : 0);
#ifdef __DEBUG
	debug(stdout, "select return %d. %f\n", ret, sys::clock::elapsed());
#endif
				if (ret == -1)
					throw std::runtime_error("select failed in 'fdstreabuf'");
   				return true;
			} while (--_curtime > 0);
			return false;
		}

		default_reader	_default_reader;
		default_writer	_default_writer;
		reader_t	_reader;
		writer_t	_writer;
		char*		_gbuffer;
		char*		_pbuffer;
		int			_descriptor;
		time_t		_timeout;
		time_t		_curtime;
		size_t		_length;

		int		_rlength;
		int		_currlen;
	};



	class basic_ifdstream : public std::istream {
	public:

		basic_ifdstream(int d=-1) : std::istream(0), _fdbuf(d)
		{
			this->init(&_fdbuf);
		}

		~basic_ifdstream() { }

		void reset(int d)
		{
			_fdbuf.reset(d);
		}

		void timeout(time_t t)
		{
			_fdbuf.timeout(t);
		}

		void rlength(int n)
		{
			_fdbuf.rlength(n);
		}

		void reading_method(reader_t p)
		{
			_fdbuf.reading_method(p);
		}

	private:
		fdstreambuf	_fdbuf;
	};


	class basic_ofdstream : public std::ostream {
	public:

		basic_ofdstream(int d=-1) : std::ostream(0), _fdbuf(d)
		{
			this->init(&_fdbuf);
			unsetf(std::ios::skipws);
		}

		~basic_ofdstream() { }

		void reset(int d)
		{
			_fdbuf.reset(d);
		}

		void timeout(time_t t)
		{
			_fdbuf.timeout(t);
		}

		void writing_method(reader_t p)
		{
			_fdbuf.writing_method(p);
		}

	private:
		fdstreambuf	_fdbuf;
	};


	class basic_iofdstream : public std::iostream {
	public:

		basic_iofdstream(int d=-1) : std::iostream(0), _fdbuf(d)
		{
			this->init(&_fdbuf);
			unsetf(std::ios::skipws);
		}

		~basic_iofdstream() { }

		void reset(int d)
		{
			_fdbuf.reset(d);
		}

		void timeout(time_t t)
		{
			_fdbuf.timeout(t);
		}

		void rlength(int n)
		{
			_fdbuf.rlength(n);
		}

		void reading_method(reader_t p)
		{
			_fdbuf.reading_method(p);
		}

		void writing_method(reader_t p)
		{
			_fdbuf.writing_method(p);
		}

	private:
		fdstreambuf	_fdbuf;
	};

	typedef basic_ifdstream		ifdstream;
	typedef basic_ofdstream		ofdstream;
	typedef basic_iofdstream	iofdstream;

} // end of namespace io

#endif // IO_FDSTREAM_H
