/*-
 * Copyright (c) 2008 Masashi Osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef SL_INET_URI_PARSER_HPP
#define SL_INET_URI_PARSER_HPP

/*
3.2.2 http URL
    http_URL = "http:" "//" host [ ":" port ] [ abs_path [ "?" query ]]

5.1 ꥯȥ饤
    Method SP Request-URI SP HTTP-Version CRLF

5.1.1 ᥽å
    Method = "OPTIONS" | "GET" | "POST" | ... | ... | extension-method
    extention-method = token

5.1.2  Request-URI
    Request-URI    = "*" | absoluteURI | abs_path | authority

    - RFC2396 -
      URI-reference = [ absoluteURI | relativeURI ] [ "#" fragment ]
      absoluteURI   = scheme ":" ( hier_part | opaque_part )
      relativeURI   = ( net_path | abs_path | rel_path ) [ "?" query ]

      hier_part     = ( net_path | abs_path ) [ "?" query ]
      opaque_part   = uric_no_slash *uric

      uric_no_slash = unreserved | escaped | ";" | "?" | ":" | "@" |
                      "&" | "=" | "+" | "$" | ","

      net_path      = "//" authority [ abs_path ]
      abs_path      = "/"  path_segments
      rel_path      = rel_segment [ abs_path ]

      rel_segment   = 1*( unreserved | escaped |
                          ";" | "@" | "&" | "=" | "+" | "$" | "," )

      scheme        = alpha *( alpha | digit | "+" | "-" | "." )

      authority     = server | reg_name

      reg_name      = 1*( unreserved | escaped | "$" | "," |
                          ";" | ":" | "@" | "&" | "=" | "+" )

      server        = [ [ userinfo "@" ] hostport ]
      userinfo      = *( unreserved | escaped |
                         ";" | ":" | "&" | "=" | "+" | "$" | "," )

      hostport      = host [ ":" port ]
      host          = hostname | IPv4address
      hostname      = *( domainlabel "." ) toplabel [ "." ]
      domainlabel   = alphanum | alphanum *( alphanum | "-" ) alphanum
      toplabel      = alpha | alpha *( alphanum | "-" ) alphanum
      IPv4address   = 1*digit "." 1*digit "." 1*digit "." 1*digit
      port          = *digit

      path          = [ abs_path | opaque_part ]
      path_segments = segment *( "/" segment )
      segment       = *pchar *( ";" param )
      param         = *pchar
      pchar         = unreserved | escaped |
                      ":" | "@" | "&" | "=" | "+" | "$" | ","

      query         = *uric

      fragment      = *uric

      uric          = reserved | unreserved | escaped
      reserved      = ";" | "/" | "?" | ":" | "@" | "&" | "=" | "+" |
                      "$" | ","
      unreserved    = alphanum | mark
      mark          = "-" | "_" | "." | "!" | "~" | "*" | "'" |
                      "(" | ")"

      escaped       = "%" hex hex
      hex           = digit | "A" | "B" | "C" | "D" | "E" | "F" |
                              "a" | "b" | "c" | "d" | "e" | "f"

      alphanum      = alpha | digit
      alpha         = lowalpha | upalpha

      lowalpha = "a" | "b" | "c" | "d" | "e" | "f" | "g" | "h" | "i" |
                 "j" | "k" | "l" | "m" | "n" | "o" | "p" | "q" | "r" |
                 "s" | "t" | "u" | "v" | "w" | "x" | "y" | "z"
      upalpha  = "A" | "B" | "C" | "D" | "E" | "F" | "G" | "H" | "I" |
                 "J" | "K" | "L" | "M" | "N" | "O" | "P" | "Q" | "R" |
                 "S" | "T" | "U" | "V" | "W" | "X" | "Y" | "Z"
      digit    = "0" | "1" | "2" | "3" | "4" | "5" | "6" | "7" |
                 "8" | "9"

    [414(Request-URI Too Long)]

    HTTP-Version = "HTTP" "/" 1*DIGIT "." 1*DIGIT

*/

//#define BOOST_SPIRIT_DEBUG
#define BOOST_SPIRIT_THREADSAFE
#include <boost/spirit.hpp>
#include <boost/spirit/phoenix.hpp>

#include <sl/inet/parser_util.hpp>

namespace sl {

struct uri_parser : public boost::spirit::grammar<uri_parser> {

    phoenix::functor<sl::inet::append_action>    _append;
    phoenix::functor<sl::inet::assign_action>    _assign;

    std::string& _reg_name;
    std::string& _scheme;
    std::string& _userinfo;
    std::string& _host;
    std::string& _port;
    std::string& _abs_path;
    std::string& _query;

    uri_parser( std::string& reg_name,
                std::string& scheme,
                std::string& userinfo,
                std::string& host,
                std::string& port,
                std::string& abs_path,
                std::string& query)
        : _append(sl::inet::append_action()),
          _assign(sl::inet::assign_action()),
          _reg_name(reg_name),
          _scheme(scheme),
          _userinfo(userinfo),
          _host(host),
          _port(port),
          _abs_path(abs_path),
          _query(query)
    { }

    ~uri_parser()
    { }

    struct uri_closure : boost::spirit::closure<uri_closure, std::string>
    {
        member1 val;
    };

    template<typename ScannerT>
    struct definition {
        boost::spirit::rule<ScannerT>
            top;
        boost::spirit::rule<ScannerT, uri_closure::context_t>
            absoluteURI,
            relativeURI,
            hier_part,
            opaque_part,
            uric_no_slash,
            net_path,
            abs_path,
            rel_path,
            rel_segment,
            scheme,
            authority,
            reg_name,
            server,
            userinfo,
            hostport,
            host,
            hostname,
            domainlabel,
            toplabel,
            IPv4address,
            port,
            path,
            path_segments,
            segment,
            param,
            pchar,
            query,
            fragment,
            uric,
            reserved,
            unreserved,
            mark,
            escaped,
            hex,
            alphanum,
            alpha,
            lowalpha,
            upalpha,
            digit;

#define ASSIGN(X)    self._assign(X.val, arg1, arg2)
#define APPEND(X)    self._append(X.val, arg1, arg2)

        definition(const uri_parser& self)
        {
            using namespace boost::spirit;
            using phoenix::arg1;
            using phoenix::arg2;

    /* Request-URI = "*" | absoluteURI | [ abs_path[ "?" query ]] | authority */

    //top = absoluteURI | relativeURI >> !( str_p("#") >> fragment );
    top = absoluteURI
        | abs_path                      [assign_a(self._abs_path)]
          >> !("?" >> query             [assign_a(self._query)]
              )
        | authority;

    absoluteURI   = scheme              [assign_a(self._scheme)] >>
                    str_p(":") >>
                    ( hier_part
                    | opaque_part
                    );

    relativeURI   = ( net_path
                    | abs_path          [assign_a(self._abs_path)]
                    | rel_path ) >>
                      !( "?" >>
                         query          [assign_a(self._query)]
                       );

    hier_part     = ( net_path
                    | abs_path          [assign_a(self._abs_path)]
                    ) >>
                   !( "?" >>
                      query             [assign_a(self._query)]
                    );

    opaque_part   = uric_no_slash       [opaque_part.val  = arg1] >>
                    *uric               [opaque_part.val += arg1];

    uric_no_slash = unreserved
                  | escaped
                  | chset<>(";?:@&=+$,");

    net_path      = str_p("//")         [net_path.val  = arg1] >>
                    authority           [net_path.val += arg1] >>
                    !( abs_path )       [assign_a(self._abs_path)];

    abs_path      = str_p("/")          [APPEND(abs_path)]  >>
                    path_segments       [abs_path.val += arg1];

    rel_path      = rel_segment         [rel_path.val = arg1] >>
                    !( abs_path )       [assign_a(self._abs_path)];

    rel_segment   = +( unreserved       [rel_segment.val = arg1]
                     | escaped          [rel_segment.val = arg1]
                     | chset<>(";@&=+$,")[rel_segment.val = arg1]
                     );

    scheme        = alpha               [scheme.val = arg1] >>
                    *( alpha            [scheme.val += arg1]
                     | digit            [scheme.val += arg1]
                     | str_p("+")       [scheme.val += arg1]
                     | str_p("-")       [scheme.val += arg1]
                     | str_p(".")       [scheme.val += arg1]
                     );

    authority     = server
                  | reg_name            [assign_a(self._reg_name)];

    reg_name      = +( unreserved       [reg_name.val += arg1]
                     | escaped          [reg_name.val += arg1]
                     | chset<>("$,;:@&=+")[reg_name.val += arg1]
                     );

    server        = !( !( userinfo >>
                          str_p("@")
                        )               [assign_a(self._userinfo)] >>
                        hostport        [assign_a(self._host)]
                     );
    userinfo      = *( unreserved       [userinfo.val += arg1]
                     | escaped          [userinfo.val += arg1]
                     | chset<>(";:&=+$,")[userinfo.val += arg1]
                     );

    hostport      = host                [hostport.val = arg1] >>
                   !( str_p(":") >>
                      port              [assign_a(self._port)]
                    );

    host          = hostname            [host.val = arg1]
                  | IPv4address         [host.val = arg1];

    hostname      = *( domainlabel      [hostname.val += arg1] >>
                       str_p(".")       [hostname.val += arg1]
                     ) >>
                     toplabel           [hostname.val += arg1] >>
                     !( str_p(".")      [hostname.val += arg1]
                     );
    domainlabel   = alphanum            [domainlabel.val  = arg1]
                  | alphanum            [domainlabel.val  = arg1] >>
                    *( alphanum         [domainlabel.val += arg1]
                     | str_p("-")       [domainlabel.val += arg1]
                     ) >> alphanum      [domainlabel.val += arg1];

    toplabel      = alpha               [toplabel.val  = arg1]
                  | alpha               [toplabel.val  = arg1] >>
                    *( alphanum         [toplabel.val += arg1]
                     | str_p("-")       [toplabel.val += arg1]
                     ) >> alphanum      [toplabel.val += arg1];

    IPv4address   = +digit              [IPv4address.val += arg1] >>
                    str_p(".")          [IPv4address.val += arg1] >>
                    +digit              [IPv4address.val += arg1] >>
                    str_p(".")          [IPv4address.val += arg1] >>
                    +digit              [IPv4address.val += arg1] >>
                    str_p(".")          [IPv4address.val += arg1] >>
                    +digit              [IPv4address.val += arg1];

    port          = *digit              [port.val += arg1];

    path          = !( abs_path         [path.val = arg1]
                     | opaque_part      [path.val = arg1]
                     );
    path_segments = segment             [path_segments.val = arg1] >>
                  *( str_p("/") >>
                     segment
                  )                     [APPEND(path_segments)];

    segment       = *pchar              [segment.val += arg1] >>
                  *( str_p(";")         [segment.val = arg1] >>
                     param              [segment.val = arg1]
                  );
    param         = *pchar              [param.val = arg1];
    pchar         = unreserved          [pchar.val = arg1]
                  | escaped             [pchar.val = arg1]
                  | chset<>(":@&=+$,")  [pchar.val = arg1];

    query         = *uric               [query.val += arg1];
    fragment      = *uric               [fragment.val += arg1];

    uric          = reserved            [uric.val = arg1]
                  | unreserved          [uric.val = arg1]
                  | escaped             [uric.val = arg1];
    reserved      = chset<>(";/?:@&=+$,")[reserved.val = arg1];
    unreserved    = alphanum            [unreserved.val = arg1]
                  | mark                [unreserved.val = arg1];
    mark          = chset<>("-_.!~*'()")[mark.val = arg1];

    escaped       = str_p("%")          [APPEND(escaped)]     >>
                    hex                 [escaped.val += arg1] >>
                    hex                 [escaped.val += arg1];
    hex           = xdigit_p            [hex.val = arg1];
    alphanum      = alnum_p             [alphanum.val = arg1];
    alpha         = alpha_p             [alpha.val = arg1];
    digit         = digit_p             [digit.val = arg1];


    BOOST_SPIRIT_DEBUG_NODE(top);
    BOOST_SPIRIT_DEBUG_NODE(absoluteURI);
    BOOST_SPIRIT_DEBUG_NODE(relativeURI);
    BOOST_SPIRIT_DEBUG_NODE(hier_part);
    BOOST_SPIRIT_DEBUG_NODE(opaque_part);
    BOOST_SPIRIT_DEBUG_NODE(uric_no_slash);
    BOOST_SPIRIT_DEBUG_NODE(net_path);
    BOOST_SPIRIT_DEBUG_NODE(abs_path);
    BOOST_SPIRIT_DEBUG_NODE(rel_path);
    BOOST_SPIRIT_DEBUG_NODE(rel_segment);
    BOOST_SPIRIT_DEBUG_NODE(scheme);
    BOOST_SPIRIT_DEBUG_NODE(authority);
    BOOST_SPIRIT_DEBUG_NODE(reg_name);
    BOOST_SPIRIT_DEBUG_NODE(server);
    BOOST_SPIRIT_DEBUG_NODE(userinfo);
    BOOST_SPIRIT_DEBUG_NODE(hostport);
    BOOST_SPIRIT_DEBUG_NODE(host);
    BOOST_SPIRIT_DEBUG_NODE(hostname);
    BOOST_SPIRIT_DEBUG_NODE(domainlabel);
    BOOST_SPIRIT_DEBUG_NODE(toplabel);
    BOOST_SPIRIT_DEBUG_NODE(IPv4address);
    BOOST_SPIRIT_DEBUG_NODE(port);
    BOOST_SPIRIT_DEBUG_NODE(path);
    BOOST_SPIRIT_DEBUG_NODE(path_segments);
    BOOST_SPIRIT_DEBUG_NODE(segment);
    BOOST_SPIRIT_DEBUG_NODE(param);
    BOOST_SPIRIT_DEBUG_NODE(pchar);
    BOOST_SPIRIT_DEBUG_NODE(query);
    BOOST_SPIRIT_DEBUG_NODE(fragment);
    BOOST_SPIRIT_DEBUG_NODE(uric);
    BOOST_SPIRIT_DEBUG_NODE(reserved);
    BOOST_SPIRIT_DEBUG_NODE(unreserved);
    BOOST_SPIRIT_DEBUG_NODE(mark);
    BOOST_SPIRIT_DEBUG_NODE(escaped);
    BOOST_SPIRIT_DEBUG_NODE(hex);
    BOOST_SPIRIT_DEBUG_NODE(alphanum);
    BOOST_SPIRIT_DEBUG_NODE(alpha);
    BOOST_SPIRIT_DEBUG_NODE(lowalpha);
    BOOST_SPIRIT_DEBUG_NODE(upalpha);
    BOOST_SPIRIT_DEBUG_NODE(digit);
        }
#undef ASSIGN
#undef APPEND

        const boost::spirit::rule<ScannerT>& start() const { return top; }
    };
};

} // namespace sl


#endif // SL_INET_URI_PARSER_HPP
