/*-
 * Copyright (c) 2005 Masashi Osakabe
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef SL_INET_TYPES_H
#define SL_INET_TYPES_H

#include <list>
#include <map>
#include <string>
#include <vector>
#include <boost/algorithm/string/predicate.hpp>

#include <sl/inet/http_util.hpp>

namespace sl { namespace http {


class msg_type {
public :
    static const int HTTP_REQUEST;
    static const int HTTP_RESPONSE;
};

class method_code {
public :
    static const int OPTIONS;
    static const int GET;
    static const int HEAD;
    static const int POST;
    static const int PUT;
    static const int DELETE;
    static const int TRACE;
    static const int CONNECT;

    static const std::string to_string(int code)
    {
        std::map<int, std::string>::iterator i = instance()._map.find(code);
        return i != instance()._map.end() ? i->second : "";
    }

    static bool find(int code)
    {
        return instance()._map.find(code) != instance()._map.end();
    }

    static bool find(const std::string &value)
    {
        std::map<int, std::string>::iterator i = instance()._map.begin();
        for (; i != instance()._map.end(); i++) {
            if (i->second == value)
                return true;
        }
        return false;
    }

private :
    method_code();

    static method_code &instance()
    {
        static method_code instance;
        return instance;
    }

    std::map<int, std::string>        _map;
};


class status_code {

    typedef std::map<int, std::string>    map_type;

public :
    // 1. Informational 1xx
    static const int CONTINUE;
    static const int SWITCHING_PROTOCOLS;

    // 2. Successful 2xx
    static const int OK;
    static const int CREATED;
    static const int ACCEPTED;
    static const int NON_AUTHORITATIVE_INFORMATION;
    static const int NO_CONTENT;
    static const int RESET_CONTENT;
    static const int PARTIAL_CONTENT;

    // 3. Redirection 3xx
    static const int MULTIPLE_CHOICES;
    static const int MOVED_PERMANENTLY;
    static const int FOUND;
    static const int SEE_OTHER;
    static const int NOT_MODIFIED;
    static const int USE_PROXY;
//    static const int (Unused);
    static const int TEMPORARY_REDIRECT;

     // 4. Client Error 4xx
    static const int BAD_REQUEST;
    static const int UNAUTHORIZED;
    static const int PAYMENT_REQUIRED;
    static const int FORBIDDEN;
    static const int NOT_FOUND;
    static const int METHOD_NOT_ALLOWED;
    static const int NOT_ACCEPTABLE;
    static const int PROXY_AUTHENTICATION_REQUIRED;
    static const int REQUEST_TIMEOUT;
    static const int CONFLICT;
    static const int GONE;
    static const int LENGTH_REQUIRED;
    static const int PRECONDITION_FAILED;
    static const int REQUEST_ENTITY_TOOLARGE;
    static const int REQUEST_URI_TOOLONG;
    static const int UNSUPPORTED_MEDIATYPE;
    static const int REQUESTED_RANGE_NOTSATISFIABLE;
    static const int EXPECTATION_FAILED;

    // 5. Server Error 5xx
    static const int INTERNAL_SERVER_ERROR;
    static const int NOT_IMPLEMENTED;
    static const int BAD_GATEWAY;
    static const int SERVICE_UNAVAILABLE;
    static const int GATEWAY_TIMEOUT;
    static const int HTTPVERSION_NOT_SUPPORTED;

    static const std::string to_string(int code)
    {
        map_type::iterator i = instance()._map.find(code);
        return i != instance()._map.end() ? i->second : "";
    }

    static bool find(int code)
    {
        return instance()._map.find(code) != instance()._map.end();
    }

private :
    status_code();

    static status_code &instance()
    {
        static status_code instance;
        return instance;
    }

    map_type _map;
};

class general_header_code {
public :
    static const int Cache_Control;
    static const int Connection;
    static const int Date;
    static const int Pragma;
    static const int Trailer;
    static const int Transfer_Encoding;
    static const int Upgrade;
    static const int Via;
    static const int Warning;

    static const int Allow;
    static const int Content_Encoding;
    static const int Content_Language;
    static const int Content_Length;
    static const int Content_Location;
    static const int Content_MD5;
    static const int Content_Range;
    static const int Content_Type;
    static const int Expires;
    static const int Last_Modified;

    static const std::string to_string(int code)
    {
        std::map<int, std::string>::iterator i = instance()._map.begin();
        for (; i != instance()._map.end(); i++) {
            if (i->first, code)
                return i->second;
        }
        return "";
    }

    static bool find(int code)
    {
        return instance()._map.find(code) != instance()._map.end();
    }

    static bool find(const std::string &value)
    {
        std::map<int, std::string>::iterator i = instance()._map.begin();
        for (; i != instance()._map.end(); i++) {
            if (boost::equals(i->second, value, boost::is_iequal()))
                return true;
        }
        return false;
    }

private :
    general_header_code();

    static general_header_code &instance()
    {
        static general_header_code instance;
        return instance;
    }

    std::map<int, std::string>        _map;
};

class request_header_code {
public :
    static const int Accept;
    static const int Accept_Charset;
    static const int Accept_Encoding;
    static const int Accept_Language;
    static const int Autorization;
    static const int Expect;
    static const int Form;
    static const int Host;
    static const int If_Match;
    static const int If_Modified_Since;
    static const int If_None_Match;
    static const int If_Range;
    static const int If_Unmodified_Since;
    static const int Max_Forwards;
    static const int Proxy_Authorization;
    static const int Range;
    static const int Referer;
    static const int TE;
    static const int User_Agent;

    static const std::string to_string(int code)
    {
        std::map<int, std::string>::iterator i = instance()._map.begin();
        for (; i != instance()._map.end(); i++) {
            if (i->first, code)
                return i->second;
        }
        return "";
    }

    static bool find(int code)
    {
        return instance()._map.find(code) != instance()._map.end();
    }

    static bool find(const std::string &value)
    {
        std::map<int, std::string>::iterator i = instance()._map.begin();
        for (; i != instance()._map.end(); i++) {
            if (boost::equals(i->second, value, boost::is_iequal()))
                return true;
        }
        return false;
    }

private :
    request_header_code();

    static request_header_code &instance()
    {
        static request_header_code instance;
        return instance;
    }

    std::map<int, std::string>        _map;
};

class response_header_code {
public :
    static const int Accept_Ranges;
    static const int Age;
    static const int ETag;
    static const int Location;
    static const int Proxy_Authenticate;
    static const int Retry_After;
    static const int Server;
    static const int Vary;
    static const int WWW_Authenticate;

    static const std::string to_string(int code)
    {
        std::map<int, std::string>::iterator i = instance()._map.begin();
        for (; i != instance()._map.end(); i++) {
            if (i->first, code)
                return i->second;
        }
        return "";
    }

    static bool find(int code)
    {
        return instance()._map.find(code) != instance()._map.end();
    }

    static bool find(const std::string &value)
    {
        std::map<int, std::string>::iterator i = instance()._map.begin();
        for (; i != instance()._map.end(); i++) {
            if (boost::equals(i->second, value, boost::is_iequal()))
                return true;
        }
        return false;
    }

private :
    response_header_code();

    static response_header_code &instance()
    {
        static response_header_code instance;
        return instance;
    }

    std::map<int, std::string> _map;
};

} } // namespace sl::http

#include <sl/inet/types.cpp>

#endif // SL_INET_TYPES_H
